package filtering

import (
	"context"

	aviaAPI "a.yandex-team.ru/travel/app/backend/api/avia/v1"
	commonAPI "a.yandex-team.ru/travel/app/backend/api/common/v1"
	"a.yandex-team.ru/travel/app/backend/internal/avia/search/filtering/helpers"
	"a.yandex-team.ru/travel/app/backend/internal/avia/search/filtering/transferobservers"
	aviaSearchProto "a.yandex-team.ru/travel/app/backend/internal/avia/search/proto/v1"
	"a.yandex-team.ru/travel/app/backend/internal/avia/search/searchcommon"
)

type transferFilter struct {
	selectedAirportsForward  map[uint64]struct{}
	selectedAirportsBackward map[uint64]struct{}
}

func (tf *transferFilter) getFilterID() string {
	return "TransferFilter"
}

func newTransferFilter(filters *aviaAPI.SearchFiltersReq) *transferFilter {
	tf := &transferFilter{}

	tf.selectedAirportsForward = make(map[uint64]struct{})
	tf.selectedAirportsBackward = make(map[uint64]struct{})
	if filters != nil && filters.Transfer != nil && filters.Transfer.Airports != nil {
		for _, a := range filters.Transfer.Airports.Forward {
			if a.State {
				tf.selectedAirportsForward[a.StationId] = struct{}{}
			}
		}
		for _, a := range filters.Transfer.Airports.Backward {
			if a.State {
				tf.selectedAirportsBackward[a.StationId] = struct{}{}
			}
		}
	}

	return tf
}

func (tf *transferFilter) initFilterResponse(
	ctx context.Context,
	filters *aviaAPI.SearchFiltersReq,
	snippets map[string]*aviaSearchProto.Snippet,
	reference *aviaSearchProto.Reference,
	searchContext *aviaSearchProto.SearchContext,
	filterResponse *aviaAPI.SearchFiltersRsp,
) *aviaAPI.SearchFiltersRsp {
	quickTransfer := &aviaAPI.SearchFiltersRsp_QuickTransferFilter{}
	filterResponse.QuickTransfer = quickTransfer
	filterResponse.Transfer = &aviaAPI.SearchFiltersRsp_TransferFilter{
		NoTransfer: quickTransfer.State,
		OneTransferOrLess: &aviaAPI.SearchFiltersRsp_BoolFilterState{
			Enabled: true,
		},
		NoNightTransfer: &aviaAPI.SearchFiltersRsp_BoolFilterState{
			Enabled: true,
		},
		NoAirportChange: &aviaAPI.SearchFiltersRsp_BoolFilterState{
			Enabled: true,
		},
	}

	observers := tf.createObservers(reference)
	for _, snippet := range snippets {
		for _, o := range observers {
			o.Observe(snippet.Transfers)
		}
	}
	for _, o := range observers {
		o.FillInitialFilterResponse(filters, filterResponse)
	}

	return filterResponse
}

func (tf *transferFilter) needToSkip(filterResponse *aviaAPI.SearchFiltersRsp, transfers *aviaSearchProto.Transfers) bool {
	if filterResponse.QuickTransfer.State.Value && (len(transfers.ForwardTransfers) != 0 || len(transfers.BackwardTransfers) != 0) {
		return true
	}
	if filterResponse.Transfer.NoTransfer.Value && (len(transfers.ForwardTransfers) != 0 || len(transfers.BackwardTransfers) != 0) {
		return true
	}
	if filterResponse.Transfer.OneTransferOrLess.Value && (len(transfers.ForwardTransfers) > 1 || len(transfers.BackwardTransfers) > 1) {
		return true
	}
	for _, t := range transfers.ForwardTransfers {
		if t.NightTransfer && filterResponse.Transfer.NoNightTransfer.Value {
			return true
		}
		if t.AirportChange && filterResponse.Transfer.NoAirportChange.Value {
			return true
		}
		if filterResponse.Transfer.TransferDuration.Selected != nil {
			if t.DurationMinutes < filterResponse.Transfer.TransferDuration.Selected.MinimumMinutes {
				return true
			}
			if t.DurationMinutes > filterResponse.Transfer.TransferDuration.Selected.MaximumMinutes {
				return true
			}
		}
		if _, found := tf.selectedAirportsForward[t.ArrivalStationId]; found {
			return true
		}
		if _, found := tf.selectedAirportsForward[t.DepartureStationId]; found {
			return true
		}
	}
	for _, t := range transfers.BackwardTransfers {
		if t.NightTransfer && filterResponse.Transfer.NoNightTransfer.Value {
			return true
		}
		if t.AirportChange && filterResponse.Transfer.NoAirportChange.Value {
			return true
		}
		if filterResponse.Transfer.TransferDuration.Selected != nil {
			if t.DurationMinutes < filterResponse.Transfer.TransferDuration.Selected.MinimumMinutes {
				return true
			}
			if t.DurationMinutes > filterResponse.Transfer.TransferDuration.Selected.MaximumMinutes {
				return true
			}
		}
		if _, found := tf.selectedAirportsBackward[t.ArrivalStationId]; found {
			return true
		}
		if _, found := tf.selectedAirportsBackward[t.DepartureStationId]; found {
			return true
		}
	}
	return false
}

func (tf *transferFilter) filter(
	filters *aviaAPI.SearchFiltersReq,
	snippets map[string]*aviaSearchProto.Snippet,
	reference *aviaSearchProto.Reference,
	filterResponse *aviaAPI.SearchFiltersRsp,
) (map[string]struct{}, map[string]struct{}) {
	excludedSnippetKeys := make(map[string]struct{}, len(snippets))
	for sKey, s := range snippets {
		if tf.needToSkip(filterResponse, s.Transfers) {
			excludedSnippetKeys[sKey] = struct{}{}
		}
	}
	return excludedSnippetKeys, nil
}

func (tf *transferFilter) updateFilterResponse(
	ctx context.Context,
	snippets map[string]*aviaSearchProto.Snippet,
	excludedSnippetKeysByOthers map[string]struct{},
	excludedVariantKeysByOthers map[string]struct{},
	reference *aviaSearchProto.Reference,
	filterResponse *aviaAPI.SearchFiltersRsp,
) *aviaAPI.SearchFiltersRsp {
	observers := tf.createObservers(reference)
	for sKey, snippet := range snippets {
		if _, needToSkip := excludedSnippetKeysByOthers[sKey]; needToSkip {
			continue
		}
		for _, o := range observers {
			o.Observe(snippet.Transfers)
		}
	}
	for _, o := range observers {
		o.UpdateFilterResponse(filterResponse)
	}

	var withTransferMinPrice *commonAPI.Price = nil
	var noTransferMinPrice *commonAPI.Price = nil
	for sKey, s := range snippets {
		if _, needToSkip := excludedSnippetKeysByOthers[sKey]; needToSkip {
			continue
		}
		noTransfers := len(s.Transfers.ForwardTransfers) == 0 && len(s.Transfers.BackwardTransfers) == 0
		for vKey, v := range s.Variant {
			if _, needToSkipVariant := excludedVariantKeysByOthers[helpers.BuildFullVariantKey(sKey, vKey)]; needToSkipVariant {
				continue
			}
			price := &commonAPI.Price{
				Currency: v.Price.Currency,
				Value:    v.Price.Value,
			}
			if noTransfers {
				if noTransferMinPrice == nil || searchcommon.CompareOnlyPriceAsc(price, noTransferMinPrice) {
					noTransferMinPrice = price
				}
			} else {
				if withTransferMinPrice == nil || searchcommon.CompareOnlyPriceAsc(price, withTransferMinPrice) {
					withTransferMinPrice = price
				}
			}
		}
	}
	filterResponse.QuickTransfer.MinPriceNoTransfer = noTransferMinPrice
	filterResponse.QuickTransfer.MinPriceWithTransfer = withTransferMinPrice

	return filterResponse
}

func (tf *transferFilter) createObservers(reference *aviaSearchProto.Reference) []transferobservers.TransferObserver {
	return []transferobservers.TransferObserver{
		transferobservers.NewNoTransferObserver(),
		transferobservers.NewOneOrLessTransferObserver(),
		transferobservers.NewNightTransferObserver(),
		transferobservers.NewAirportChangeObserver(),
		transferobservers.NewTransferDurationObserver(),
		transferobservers.NewTransferAirportsObserver(tf.selectedAirportsForward, tf.selectedAirportsBackward, reference),
	}
}
