package transferobservers

import (
	"sort"

	aviaAPI "a.yandex-team.ru/travel/app/backend/api/avia/v1"
	"a.yandex-team.ru/travel/app/backend/internal/avia/search/filtering/helpers"
	aviaSearchProto "a.yandex-team.ru/travel/app/backend/internal/avia/search/proto/v1"
)

type TransferAirortsObserver struct {
	selectedAirportsForward  map[uint64]struct{}
	selectedAirportsBackward map[uint64]struct{}

	airportsForward  map[uint64]struct{}
	airportsBackward map[uint64]struct{}

	reference *aviaSearchProto.Reference
}

func NewTransferAirportsObserver(
	selectedAirportsForward map[uint64]struct{},
	selectedAirportsBackward map[uint64]struct{},
	reference *aviaSearchProto.Reference,
) *TransferAirortsObserver {
	return &TransferAirortsObserver{
		selectedAirportsForward:  selectedAirportsForward,
		selectedAirportsBackward: selectedAirportsBackward,
		airportsForward:          make(map[uint64]struct{}),
		airportsBackward:         make(map[uint64]struct{}),
		reference:                reference,
	}
}

func (o *TransferAirortsObserver) Observe(transfers *aviaSearchProto.Transfers) {
	for _, t := range transfers.ForwardTransfers {
		o.airportsForward[t.ArrivalStationId] = struct{}{}
		o.airportsForward[t.DepartureStationId] = struct{}{}
	}
	for _, t := range transfers.BackwardTransfers {
		o.airportsBackward[t.ArrivalStationId] = struct{}{}
		o.airportsBackward[t.DepartureStationId] = struct{}{}
	}
}

func (o *TransferAirortsObserver) FillInitialFilterResponse(filters *aviaAPI.SearchFiltersReq, filterResponse *aviaAPI.SearchFiltersRsp) {
	if len(o.airportsForward) == 0 && len(o.airportsBackward) == 0 {
		return
	}

	filterResponse.Transfer.Airports = &aviaAPI.SearchFiltersRsp_TransferAirportsFilter{
		Forward: o.buildInitialDirectionAirportsResponse(
			o.airportsForward,
			o.selectedAirportsForward,
		),
		Backward: o.buildInitialDirectionAirportsResponse(
			o.airportsBackward,
			o.selectedAirportsBackward,
		),
	}
}

func (o *TransferAirortsObserver) buildInitialDirectionAirportsResponse(
	directionAirports map[uint64]struct{},
	selectedDirectionAirports map[uint64]struct{},
) []*aviaAPI.SearchFiltersRsp_TransferAirport {
	if len(directionAirports) == 0 {
		return nil
	}

	settlementCounter := helpers.NewKeyCounter()
	for aID := range directionAirports {
		settlementID := o.reference.Stations[aID].SettlementId
		if settlementID != 0 {
			settlementCounter.Increment(settlementID)
		}
	}

	direction := make([]*aviaAPI.SearchFiltersRsp_TransferAirport, 0, len(directionAirports))
	for id := range directionAirports {
		_, found := selectedDirectionAirports[id]
		station := o.reference.Stations[id]
		var a = &aviaAPI.SearchFiltersRsp_TransferAirport{
			StationId: id,
			State: &aviaAPI.SearchFiltersRsp_BoolFilterState{
				Enabled: true,
				Value:   found,
			},
			StationTitle:    station.Title,
			SettlementTitle: "",
			AviaCode:        station.AviaCode,
		}
		if station.SettlementId != 0 {
			settlementTitle := o.reference.Settlements[station.SettlementId].Title
			hasOtherStationsInSettlement := settlementCounter.GetCount(station.SettlementId) > 1
			if hasOtherStationsInSettlement || station.Title != settlementTitle {
				a.SettlementTitle = settlementTitle
			}
		}
		direction = append(direction, a)
	}

	sort.SliceStable(direction, func(i, j int) bool {
		return direction[i].StationId < direction[j].StationId
	})

	return direction
}

func (o *TransferAirortsObserver) UpdateFilterResponse(filterResponse *aviaAPI.SearchFiltersRsp) {
	if filterResponse.Transfer.Airports != nil {
		for _, a := range filterResponse.Transfer.Airports.Forward {
			if _, found := o.airportsForward[a.StationId]; !found {
				a.State.Enabled = false
			}
		}
		for _, a := range filterResponse.Transfer.Airports.Backward {
			if _, found := o.airportsBackward[a.StationId]; !found {
				a.State.Enabled = false
			}
		}
	}
}
