package airport

import (
	"golang.org/x/exp/maps"

	aviaAPI "a.yandex-team.ru/travel/app/backend/api/avia/v1"
	"a.yandex-team.ru/travel/app/backend/internal/avia/search/filtering2/filterinterface"
	"a.yandex-team.ru/travel/app/backend/internal/avia/search/filtering2/helpers"
	aviaSearchProto "a.yandex-team.ru/travel/app/backend/internal/avia/search/proto/v1"
	"a.yandex-team.ru/travel/library/go/containers"
)

type AirportForwardDepartureFilter struct {
	needToApplyFilter bool
	idsInRequest      containers.Set[uint64]
}

func (af *AirportForwardDepartureFilter) GetFilterID() string {
	return "AirportForwardDepartureFilter"
}

func NewAirportForwardDepartureFilter(filters *aviaAPI.SearchFiltersReq) *AirportForwardDepartureFilter {
	idsInRequest := containers.SetOf[uint64]()
	if filters != nil && filters.Airport != nil && filters.Airport.Forward != nil && filters.Airport.Forward.Departure != nil {
		for _, a := range filters.Airport.Forward.Departure.Airports {
			if a.State {
				idsInRequest.Add(a.StationId)
			}
		}
	}

	return &AirportForwardDepartureFilter{
		needToApplyFilter: true,
		idsInRequest:      idsInRequest,
	}
}

func (af *AirportForwardDepartureFilter) InitFilterResponse(
	filters *aviaAPI.SearchFiltersReq,
	snippets map[string]*aviaSearchProto.Snippet,
	reference *aviaSearchProto.Reference,
	searchContext *aviaSearchProto.SearchContext,
	filterResponse *aviaAPI.SearchFiltersRsp,
) *aviaAPI.SearchFiltersRsp {
	if filterResponse.Airport == nil {
		filterResponse.Airport = &aviaAPI.SearchFiltersRsp_AirportFilter{}
	}
	if filterResponse.Airport.Forward == nil {
		filterResponse.Airport.Forward = &aviaAPI.SearchFiltersRsp_DirectionAirports{}
	}

	idsInSnippets := containers.SetOf[uint64]()
	for _, s := range snippets {
		idsInSnippets.Add(reference.Flights[s.Forward[0]].StationFromId)
	}
	af.needToApplyFilter = len(af.idsInRequest) > 0 && !maps.Equal(idsInSnippets, af.idsInRequest)

	filterResponse.Airport.Forward.Departure = &aviaAPI.SearchFiltersRsp_DepartureOrArrivalAirports{
		SettlementId: getSettlementID(searchContext.PointFrom),
		Airports:     getAirports(idsInSnippets, af.idsInRequest, reference),
		Title:        helpers.BuildDepartureTitle(searchContext.PointFrom, reference),
	}

	return filterResponse
}

func (af *AirportForwardDepartureFilter) Filter(
	filters *aviaAPI.SearchFiltersReq,
	snippets map[string]*aviaSearchProto.Snippet,
	reference *aviaSearchProto.Reference,
	filterResponse *aviaAPI.SearchFiltersRsp,
) *filterinterface.ExcludedKeys {
	excludedKeys := filterinterface.NewExcludedKeys()

	if !af.needToApplyFilter {
		return excludedKeys
	}

	for sKey, s := range snippets {
		if !af.idsInRequest.Contains(reference.Flights[s.Forward[0]].StationFromId) {
			excludedKeys.AddSnippetKey(sKey)
		}
	}
	return excludedKeys
}

func (af *AirportForwardDepartureFilter) UpdateFilterResponse(
	snippets map[string]*aviaSearchProto.Snippet,
	excludedKeysByOthers *filterinterface.ExcludedKeys,
	reference *aviaSearchProto.Reference,
	filterResponse *aviaAPI.SearchFiltersRsp,
) *aviaAPI.SearchFiltersRsp {
	idsInSnippets := containers.SetOf[uint64]()
	for sKey, s := range snippets {
		if excludedKeysByOthers.ContainsSnippetKey(sKey) {
			continue
		}
		idsInSnippets.Add(reference.Flights[s.Forward[0]].StationFromId)
	}

	for _, f := range filterResponse.Airport.Forward.Departure.Airports {
		if !idsInSnippets.Contains(f.StationId) {
			f.State.Enabled = false
		}
	}

	return filterResponse
}
