package filtering2

import (
	aviaAPI "a.yandex-team.ru/travel/app/backend/api/avia/v1"
	commonAPI "a.yandex-team.ru/travel/app/backend/api/common/v1"
	"a.yandex-team.ru/travel/app/backend/internal/avia/search/filtering2/filterinterface"
	aviaSearchProto "a.yandex-team.ru/travel/app/backend/internal/avia/search/proto/v1"
	"a.yandex-team.ru/travel/app/backend/internal/avia/search/searchcommon"
)

type baggageFilter struct{}

func (q *baggageFilter) GetFilterID() string {
	return "BaggageFilter"
}

func NewBaggageFilter() *baggageFilter {
	return &baggageFilter{}
}

func (q *baggageFilter) InitFilterResponse(
	filters *aviaAPI.SearchFiltersReq,
	snippets map[string]*aviaSearchProto.Snippet,
	reference *aviaSearchProto.Reference,
	searchContext *aviaSearchProto.SearchContext,
	filterResponse *aviaAPI.SearchFiltersRsp,
) *aviaAPI.SearchFiltersRsp {
	quickBaggage := &aviaAPI.SearchFiltersRsp_QuickBaggageFilter{
		State: &aviaAPI.SearchFiltersRsp_BoolFilterState{
			Enabled: true,
			Value:   false,
		},
	}

	if filters != nil {
		quickBaggage.State.Value = filters.QuickBaggage
	}

	filterResponse.QuickBaggage = quickBaggage
	return filterResponse
}

func (q *baggageFilter) needToSkip(filters *aviaAPI.SearchFiltersRsp, variant *aviaSearchProto.Variant) bool {
	if filters.QuickBaggage.State.Value && !variant.Baggage.Included {
		return true
	}
	return false
}

func (q *baggageFilter) Filter(
	filters *aviaAPI.SearchFiltersReq,
	snippets map[string]*aviaSearchProto.Snippet,
	reference *aviaSearchProto.Reference,
	filterResponse *aviaAPI.SearchFiltersRsp,
) *filterinterface.ExcludedKeys {
	excludedKeys := filterinterface.NewExcludedKeys()
	for sKey, s := range snippets {
		hasRefinedVariants := false
		excludedVariantKeysLocal := make(map[string]struct{}, len(s.Variant))
		for vKey, variant := range s.Variant {
			if q.needToSkip(filterResponse, variant) {
				excludedVariantKeysLocal[vKey] = struct{}{}
			} else {
				hasRefinedVariants = true
			}
		}
		if !hasRefinedVariants {
			excludedKeys.AddSnippetKey(sKey)
		} else {
			for vKey := range excludedVariantKeysLocal {
				excludedKeys.AddVariantKey(sKey, vKey)
			}
		}
	}
	return excludedKeys
}

func (q *baggageFilter) UpdateFilterResponse(
	snippets map[string]*aviaSearchProto.Snippet,
	excludedKeysByOthers *filterinterface.ExcludedKeys,
	reference *aviaSearchProto.Reference,
	filterResponse *aviaAPI.SearchFiltersRsp,
) *aviaAPI.SearchFiltersRsp {
	var withBaggageMinPrice *commonAPI.Price = nil
	var noBaggageMinPrice *commonAPI.Price = nil
	for sKey, s := range snippets {
		needToSkip := excludedKeysByOthers.ContainsSnippetKey(sKey)
		if needToSkip {
			continue
		}
		for vKey, v := range s.Variant {
			needToSkipVariant := excludedKeysByOthers.ContainsVariantKey(sKey, vKey)
			if needToSkipVariant {
				continue
			}
			price := &commonAPI.Price{
				Currency: v.Price.Currency,
				Value:    v.Price.Value,
			}
			if v.Baggage.Included {
				if withBaggageMinPrice == nil || searchcommon.CompareOnlyPriceAsc(price, withBaggageMinPrice) {
					withBaggageMinPrice = price
				}
			} else {
				if noBaggageMinPrice == nil || searchcommon.CompareOnlyPriceAsc(price, noBaggageMinPrice) {
					noBaggageMinPrice = price
				}
			}
		}
	}
	filterResponse.QuickBaggage.MinPriceNoBaggage = noBaggageMinPrice
	filterResponse.QuickBaggage.MinPriceWithBaggage = withBaggageMinPrice

	hasBaggage := withBaggageMinPrice != nil
	if !hasBaggage {
		filterResponse.QuickBaggage.State.Enabled = false
	}

	// Разрешаем выключить фильтр, если он вдруг включен, но с багажем нет рейсов
	if filterResponse.QuickBaggage.State.Value && !hasBaggage {
		filterResponse.QuickBaggage.State.Enabled = true
	}

	return filterResponse
}
