package filtering2

import (
	"context"
	"fmt"

	"a.yandex-team.ru/library/go/core/log"
	aviaAPI "a.yandex-team.ru/travel/app/backend/api/avia/v1"
	"a.yandex-team.ru/travel/app/backend/internal/avia/search/filtering2/airport"
	"a.yandex-team.ru/travel/app/backend/internal/avia/search/filtering2/filterinterface"
	"a.yandex-team.ru/travel/app/backend/internal/avia/search/filtering2/timefilter"
	"a.yandex-team.ru/travel/app/backend/internal/avia/search/filtering2/transfer"
	aviaSearchProto "a.yandex-team.ru/travel/app/backend/internal/avia/search/proto/v1"
)

func ApplyFilters(
	ctx context.Context,
	logger log.Logger,
	snippets map[string]*aviaSearchProto.Snippet,
	reference *aviaSearchProto.Reference,
	searchContext *aviaSearchProto.SearchContext,
	filters *aviaAPI.SearchFiltersReq,
) (
	map[string]*aviaSearchProto.Snippet,
	*aviaAPI.SearchFiltersRsp,
) {
	fs := []filterinterface.SearchFilterInterface{
		NewBaggageFilter(),
		NewAviacompanyFilter(ctx, logger, filters),

		transfer.NewNoTransferFilter(),
		transfer.NewOneTransferOrLess(),
		transfer.NewNoNightTransferFilter(),
		transfer.NewNoAirportChangeFilter(),
		transfer.NewTransferDurationFilter(),
		transfer.NewTransferAirportsForward(filters),

		airport.NewAirportForwardDepartureFilter(filters),
		airport.NewAirportForwardArrivalFilter(filters),

		timefilter.NewTimeForwardDepartureFilter(),
		timefilter.NewTimeForwardArrivalFilter(),
	}

	roundTrip := searchContext.DateBackward != nil
	if roundTrip {
		fs = append(
			fs,
			transfer.NewTransferAirportsBackward(filters),

			airport.NewAirportBackwardDepartureFilter(filters),
			airport.NewAirportBackwardArrivalFilter(filters),

			timefilter.NewTimeBackwardDepartureFilter(),
			timefilter.NewTimeBackwardArrivalFilter(),
		)
	}

	filterResponse := &aviaAPI.SearchFiltersRsp{}
	for _, f := range fs {
		filterResponse = f.InitFilterResponse(filters, snippets, reference, searchContext, filterResponse)
	}

	// Для понимания что дизейблить нужно знать, что осталось после других фильтров.
	// Поэтому сохраняем ключи того, что нужно отфильтровать, для каждого фильтра.
	excludedKeysByFilter := filterinterface.NewExcludedKeysByFilter(fs)
	for _, f := range fs {
		excludedKeysByFilter[f.GetFilterID()] = f.Filter(filters, snippets, reference, filterResponse)
	}

	// Для дизейбла, строим что отфильтровали другие фильтры.
	for _, f := range fs {
		excludedKeysByOthers := excludedKeysByFilter.UniteWithoutOne(f.GetFilterID())
		filterResponse = f.UpdateFilterResponse(
			snippets,
			excludedKeysByOthers,
			reference,
			filterResponse,
		)
	}

	// Фильтруем сниппеты и варианты.
	excludedKeys := excludedKeysByFilter.UniteAll()
	refinedSnippets := make(map[string]*aviaSearchProto.Snippet)
	for sKey, s := range snippets {
		needToSkip := excludedKeys.ContainsSnippetKey(sKey)
		if !needToSkip {
			variants := make(map[string]*aviaSearchProto.Variant, len(s.Variant))
			for vKey, v := range s.Variant {
				needToSkipVariant := excludedKeys.ContainsVariantKey(sKey, vKey)
				if !needToSkipVariant {
					variants[vKey] = v
				}
			}
			refinedSnippets[sKey] = &aviaSearchProto.Snippet{
				Key:                     s.Key,
				Forward:                 s.Forward,
				Backward:                s.Backward,
				Transfers:               s.Transfers,
				Variant:                 variants,
				Badges:                  s.Badges,
				ForwardDurationMinutes:  s.ForwardDurationMinutes,
				BackwardDurationMinutes: s.BackwardDurationMinutes,
			}
		}
	}
	return refinedSnippets, filterResponse
}

func CheckPreconditions(filters *aviaAPI.SearchFiltersReq) error {
	if filters == nil || filters.Transfer == nil {
		return nil
	}
	if filters.QuickTransfer != filters.Transfer.NoTransfer {
		return fmt.Errorf("filters: quick_transfer must be the same as transfer.no_transfer")
	}

	return nil
}
