package timefilter

import (
	aviaAPI "a.yandex-team.ru/travel/app/backend/api/avia/v1"
	"a.yandex-team.ru/travel/app/backend/internal/avia/search/filtering2/filterinterface"
	"a.yandex-team.ru/travel/app/backend/internal/avia/search/filtering2/helpers"
	aviaSearchProto "a.yandex-team.ru/travel/app/backend/internal/avia/search/proto/v1"
	aviaProtoV2 "a.yandex-team.ru/travel/avia/library/proto/common/v2"
)

type TimeBackwardArrivalFilter struct {
	needToApplyFilter bool
}

func (t *TimeBackwardArrivalFilter) GetFilterID() string {
	return "TimeBackwardArrivalFilter"
}

func NewTimeBackwardArrivalFilter() *TimeBackwardArrivalFilter {
	return &TimeBackwardArrivalFilter{}
}

func (t *TimeBackwardArrivalFilter) InitFilterResponse(
	filters *aviaAPI.SearchFiltersReq,
	snippets map[string]*aviaSearchProto.Snippet,
	reference *aviaSearchProto.Reference,
	searchContext *aviaSearchProto.SearchContext,
	filterResponse *aviaAPI.SearchFiltersRsp,
) *aviaAPI.SearchFiltersRsp {
	if filterResponse.DepartureAndArrival == nil {
		filterResponse.DepartureAndArrival = &aviaAPI.SearchFiltersRsp_DepartureAndArrivalFilter{}
	}

	// Создаем объекты
	filterResponse.DepartureAndArrival.BackwardArrival = &aviaAPI.SearchFiltersRsp_DepartureOrArrivalState{
		Title: helpers.BuildArrivalTitle(searchContext.PointFrom, reference),
	}

	// Заполняем пункты отправления и прибытия
	if searchContext.PointFrom.Type == aviaProtoV2.PointType_POINT_TYPE_SETTLEMENT {
		filterResponse.DepartureAndArrival.BackwardArrival.SettlementId = searchContext.PointFrom.Id
	} else if searchContext.PointFrom.Type == aviaProtoV2.PointType_POINT_TYPE_STATION {
		filterResponse.DepartureAndArrival.BackwardArrival.StationId = searchContext.PointFrom.Id
	}

	if len(snippets) == 0 {
		t.needToApplyFilter = false
		return filterResponse
	}

	// Проставляем максимальный интервал
	minBackwardArrival := ""
	maxBackwardArrival := ""
	for _, s := range snippets {
		backwardArrival := reference.Flights[s.Backward[len(s.Backward)-1]].ArrivalStr
		if minBackwardArrival == "" || minBackwardArrival > backwardArrival {
			minBackwardArrival = backwardArrival
		}
		if maxBackwardArrival == "" || maxBackwardArrival < backwardArrival {
			maxBackwardArrival = backwardArrival
		}
	}
	filterResponse.DepartureAndArrival.BackwardArrival.All = &aviaAPI.SearchFiltersRsp_DepartureOrArrivalInterval{
		MinDatetime: minBackwardArrival,
		MaxDatetime: maxBackwardArrival,
	}

	// Проставим выбранный интервал из того, что пришло в запросе
	if filters != nil && filters.DepartureAndArrival != nil {
		f := filters.DepartureAndArrival
		filterResponse.DepartureAndArrival.BackwardArrival.Selected = copyIntervalFromRequest(f.BackwardArrival)
	} else {
		f := filterResponse.DepartureAndArrival
		f.BackwardArrival.Selected = copyIntervalFromResponse(f.BackwardArrival.All)
	}

	// Выбранное должно быть не больше максимального
	fitSelectedIntoAll(filterResponse.DepartureAndArrival.BackwardArrival)

	t.needToApplyFilter = needToApplyFilter(filterResponse.DepartureAndArrival.BackwardArrival)

	return filterResponse
}

func (t *TimeBackwardArrivalFilter) Filter(
	filters *aviaAPI.SearchFiltersReq,
	snippets map[string]*aviaSearchProto.Snippet,
	reference *aviaSearchProto.Reference,
	filterResponse *aviaAPI.SearchFiltersRsp,
) *filterinterface.ExcludedKeys {
	excludedKeys := filterinterface.NewExcludedKeys()
	if !t.needToApplyFilter {
		return excludedKeys
	}

	for sKey, s := range snippets {
		if !inInterval(reference.Flights[s.Backward[len(s.Backward)-1]].ArrivalStr, filterResponse.DepartureAndArrival.BackwardArrival.Selected) {
			excludedKeys.AddSnippetKey(sKey)
		}
	}
	return excludedKeys
}

func (t *TimeBackwardArrivalFilter) UpdateFilterResponse(
	snippets map[string]*aviaSearchProto.Snippet,
	excludedKeysByOthers *filterinterface.ExcludedKeys,
	reference *aviaSearchProto.Reference,
	filterResponse *aviaAPI.SearchFiltersRsp,
) *aviaAPI.SearchFiltersRsp {
	return filterResponse
}
