package timefilter

import (
	aviaAPI "a.yandex-team.ru/travel/app/backend/api/avia/v1"
	"a.yandex-team.ru/travel/app/backend/internal/avia/search/filtering2/filterinterface"
	"a.yandex-team.ru/travel/app/backend/internal/avia/search/filtering2/helpers"
	aviaSearchProto "a.yandex-team.ru/travel/app/backend/internal/avia/search/proto/v1"
	aviaProtoV2 "a.yandex-team.ru/travel/avia/library/proto/common/v2"
)

type TimeBackwardDepartureFilter struct {
	needToApplyFilter bool
}

func (t *TimeBackwardDepartureFilter) GetFilterID() string {
	return "TimeBackwardDepartureFilter"
}

func NewTimeBackwardDepartureFilter() *TimeBackwardDepartureFilter {
	return &TimeBackwardDepartureFilter{}
}

func (t *TimeBackwardDepartureFilter) InitFilterResponse(
	filters *aviaAPI.SearchFiltersReq,
	snippets map[string]*aviaSearchProto.Snippet,
	reference *aviaSearchProto.Reference,
	searchContext *aviaSearchProto.SearchContext,
	filterResponse *aviaAPI.SearchFiltersRsp,
) *aviaAPI.SearchFiltersRsp {
	if filterResponse.DepartureAndArrival == nil {
		filterResponse.DepartureAndArrival = &aviaAPI.SearchFiltersRsp_DepartureAndArrivalFilter{}
	}
	filterResponse.DepartureAndArrival.BackwardDeparture = &aviaAPI.SearchFiltersRsp_DepartureOrArrivalState{
		Title: helpers.BuildDepartureTitle(searchContext.PointTo, reference),
	}

	// Заполняем пункты отправления и прибытия
	if searchContext.PointTo.Type == aviaProtoV2.PointType_POINT_TYPE_SETTLEMENT {
		filterResponse.DepartureAndArrival.BackwardDeparture.SettlementId = searchContext.PointTo.Id
	} else if searchContext.PointTo.Type == aviaProtoV2.PointType_POINT_TYPE_STATION {
		filterResponse.DepartureAndArrival.BackwardDeparture.StationId = searchContext.PointTo.Id
	}

	if len(snippets) == 0 {
		t.needToApplyFilter = false
		return filterResponse
	}

	// Проставляем максимальный интервал
	minBackwardDeparture := ""
	maxBackwardDeparture := ""
	for _, s := range snippets {
		backwardDeparture := reference.Flights[s.Backward[0]].DepartureStr
		if minBackwardDeparture == "" || minBackwardDeparture > backwardDeparture {
			minBackwardDeparture = backwardDeparture
		}
		if maxBackwardDeparture == "" || maxBackwardDeparture < backwardDeparture {
			maxBackwardDeparture = backwardDeparture
		}
	}
	filterResponse.DepartureAndArrival.BackwardDeparture.All = &aviaAPI.SearchFiltersRsp_DepartureOrArrivalInterval{
		MinDatetime: minBackwardDeparture,
		MaxDatetime: maxBackwardDeparture,
	}

	// Проставим выбранный интервал из того, что пришло в запросе
	if filters != nil && filters.DepartureAndArrival != nil {
		f := filters.DepartureAndArrival
		filterResponse.DepartureAndArrival.BackwardDeparture.Selected = copyIntervalFromRequest(f.BackwardDeparture)
	} else {
		f := filterResponse.DepartureAndArrival
		f.BackwardDeparture.Selected = copyIntervalFromResponse(f.BackwardDeparture.All)
	}

	// Выбранное должно быть не больше максимального
	fitSelectedIntoAll(filterResponse.DepartureAndArrival.BackwardDeparture)

	t.needToApplyFilter = needToApplyFilter(filterResponse.DepartureAndArrival.BackwardDeparture)

	return filterResponse
}

func (t *TimeBackwardDepartureFilter) Filter(
	filters *aviaAPI.SearchFiltersReq,
	snippets map[string]*aviaSearchProto.Snippet,
	reference *aviaSearchProto.Reference,
	filterResponse *aviaAPI.SearchFiltersRsp,
) *filterinterface.ExcludedKeys {
	excludedKeys := filterinterface.NewExcludedKeys()
	if !t.needToApplyFilter {
		return excludedKeys
	}

	for sKey, s := range snippets {
		if !inInterval(reference.Flights[s.Backward[0]].DepartureStr, filterResponse.DepartureAndArrival.BackwardDeparture.Selected) {
			excludedKeys.AddSnippetKey(sKey)
		}
	}
	return excludedKeys
}

func (t *TimeBackwardDepartureFilter) UpdateFilterResponse(
	snippets map[string]*aviaSearchProto.Snippet,
	excludedKeysByOthers *filterinterface.ExcludedKeys,
	reference *aviaSearchProto.Reference,
	filterResponse *aviaAPI.SearchFiltersRsp,
) *aviaAPI.SearchFiltersRsp {
	return filterResponse
}
