package timefilter

import (
	aviaAPI "a.yandex-team.ru/travel/app/backend/api/avia/v1"
	"a.yandex-team.ru/travel/app/backend/internal/avia/search/filtering2/filterinterface"
	"a.yandex-team.ru/travel/app/backend/internal/avia/search/filtering2/helpers"
	aviaSearchProto "a.yandex-team.ru/travel/app/backend/internal/avia/search/proto/v1"
	aviaProtoV2 "a.yandex-team.ru/travel/avia/library/proto/common/v2"
)

type TimeForwardArrivalFilter struct {
	needToApplyFilter bool
}

func (t *TimeForwardArrivalFilter) GetFilterID() string {
	return "TimeForwardArrivalFilter"
}

func NewTimeForwardArrivalFilter() *TimeForwardArrivalFilter {
	return &TimeForwardArrivalFilter{}
}

func (t *TimeForwardArrivalFilter) InitFilterResponse(
	filters *aviaAPI.SearchFiltersReq,
	snippets map[string]*aviaSearchProto.Snippet,
	reference *aviaSearchProto.Reference,
	searchContext *aviaSearchProto.SearchContext,
	filterResponse *aviaAPI.SearchFiltersRsp,
) *aviaAPI.SearchFiltersRsp {
	if filterResponse.DepartureAndArrival == nil {
		filterResponse.DepartureAndArrival = &aviaAPI.SearchFiltersRsp_DepartureAndArrivalFilter{}
	}

	filterResponse.DepartureAndArrival.ForwardArrival = &aviaAPI.SearchFiltersRsp_DepartureOrArrivalState{
		Title: helpers.BuildArrivalTitle(searchContext.PointTo, reference),
	}
	if searchContext.PointTo.Type == aviaProtoV2.PointType_POINT_TYPE_SETTLEMENT {
		filterResponse.DepartureAndArrival.ForwardArrival.SettlementId = searchContext.PointTo.Id
	} else if searchContext.PointTo.Type == aviaProtoV2.PointType_POINT_TYPE_STATION {
		filterResponse.DepartureAndArrival.ForwardArrival.StationId = searchContext.PointTo.Id
	}

	if len(snippets) == 0 {
		t.needToApplyFilter = false
		return filterResponse
	}

	// Проставляем максимальный интервал
	minForwardArrival := ""
	maxForwardArrival := ""
	for _, s := range snippets {
		forwardArrival := reference.Flights[s.Forward[len(s.Forward)-1]].ArrivalStr
		if minForwardArrival == "" || minForwardArrival > forwardArrival {
			minForwardArrival = forwardArrival
		}
		if maxForwardArrival == "" || maxForwardArrival < forwardArrival {
			maxForwardArrival = forwardArrival
		}
	}
	filterResponse.DepartureAndArrival.ForwardArrival.All = &aviaAPI.SearchFiltersRsp_DepartureOrArrivalInterval{
		MinDatetime: minForwardArrival,
		MaxDatetime: maxForwardArrival,
	}

	// Проставим выбранный интервал из того, что пришло в запросе
	if filters != nil && filters.DepartureAndArrival != nil {
		f := filters.DepartureAndArrival
		filterResponse.DepartureAndArrival.ForwardArrival.Selected = copyIntervalFromRequest(f.ForwardArrival)
	} else {
		f := filterResponse.DepartureAndArrival
		f.ForwardArrival.Selected = copyIntervalFromResponse(f.ForwardArrival.All)
	}

	// Выбранное должно быть не больше максимального
	fitSelectedIntoAll(filterResponse.DepartureAndArrival.ForwardArrival)

	t.needToApplyFilter = needToApplyFilter(filterResponse.DepartureAndArrival.ForwardArrival)

	return filterResponse
}

func (t *TimeForwardArrivalFilter) Filter(
	filters *aviaAPI.SearchFiltersReq,
	snippets map[string]*aviaSearchProto.Snippet,
	reference *aviaSearchProto.Reference,
	filterResponse *aviaAPI.SearchFiltersRsp,
) *filterinterface.ExcludedKeys {
	excludedKeys := filterinterface.NewExcludedKeys()
	if !t.needToApplyFilter {
		return excludedKeys
	}

	for sKey, s := range snippets {
		if !inInterval(reference.Flights[s.Forward[len(s.Forward)-1]].ArrivalStr, filterResponse.DepartureAndArrival.ForwardArrival.Selected) {
			excludedKeys.AddSnippetKey(sKey)
		}
	}
	return excludedKeys
}

func (t *TimeForwardArrivalFilter) UpdateFilterResponse(
	snippets map[string]*aviaSearchProto.Snippet,
	excludedKeysByOthers *filterinterface.ExcludedKeys,
	reference *aviaSearchProto.Reference,
	filterResponse *aviaAPI.SearchFiltersRsp,
) *aviaAPI.SearchFiltersRsp {
	return filterResponse
}
