package timefilter

import (
	aviaAPI "a.yandex-team.ru/travel/app/backend/api/avia/v1"
	"a.yandex-team.ru/travel/app/backend/internal/avia/search/filtering2/filterinterface"
	"a.yandex-team.ru/travel/app/backend/internal/avia/search/filtering2/helpers"
	aviaSearchProto "a.yandex-team.ru/travel/app/backend/internal/avia/search/proto/v1"
	aviaProtoV2 "a.yandex-team.ru/travel/avia/library/proto/common/v2"
)

type timeForwardDepartureFilter struct {
	needToApplyFilter bool
}

func (t *timeForwardDepartureFilter) GetFilterID() string {
	return "TimeForwardDepartureFilter"
}

func NewTimeForwardDepartureFilter() *timeForwardDepartureFilter {
	return &timeForwardDepartureFilter{}
}

func (t *timeForwardDepartureFilter) InitFilterResponse(
	filters *aviaAPI.SearchFiltersReq,
	snippets map[string]*aviaSearchProto.Snippet,
	reference *aviaSearchProto.Reference,
	searchContext *aviaSearchProto.SearchContext,
	filterResponse *aviaAPI.SearchFiltersRsp,
) *aviaAPI.SearchFiltersRsp {
	if filterResponse.DepartureAndArrival == nil {
		filterResponse.DepartureAndArrival = &aviaAPI.SearchFiltersRsp_DepartureAndArrivalFilter{}
	}
	filterResponse.DepartureAndArrival.ForwardDeparture = &aviaAPI.SearchFiltersRsp_DepartureOrArrivalState{
		Title: helpers.BuildDepartureTitle(searchContext.PointFrom, reference),
	}
	if searchContext.PointFrom.Type == aviaProtoV2.PointType_POINT_TYPE_SETTLEMENT {
		filterResponse.DepartureAndArrival.ForwardDeparture.SettlementId = searchContext.PointFrom.Id
	} else if searchContext.PointFrom.Type == aviaProtoV2.PointType_POINT_TYPE_STATION {
		filterResponse.DepartureAndArrival.ForwardDeparture.StationId = searchContext.PointFrom.Id
	}

	if len(snippets) == 0 {
		t.needToApplyFilter = false
		return filterResponse
	}

	// Проставляем максимальный интервал
	minForwardDeparture := ""
	maxForwardDeparture := ""
	for _, s := range snippets {
		forwardDeparture := reference.Flights[s.Forward[0]].DepartureStr
		if minForwardDeparture == "" || minForwardDeparture > forwardDeparture {
			minForwardDeparture = forwardDeparture
		}
		if maxForwardDeparture == "" || maxForwardDeparture < forwardDeparture {
			maxForwardDeparture = forwardDeparture
		}
	}
	filterResponse.DepartureAndArrival.ForwardDeparture.All = &aviaAPI.SearchFiltersRsp_DepartureOrArrivalInterval{
		MinDatetime: minForwardDeparture,
		MaxDatetime: maxForwardDeparture,
	}

	// Проставим выбранный интервал из того, что пришло в запросе
	if filters != nil && filters.DepartureAndArrival != nil {
		f := filters.DepartureAndArrival
		filterResponse.DepartureAndArrival.ForwardDeparture.Selected = copyIntervalFromRequest(f.ForwardDeparture)
	} else {
		f := filterResponse.DepartureAndArrival
		f.ForwardDeparture.Selected = copyIntervalFromResponse(f.ForwardDeparture.All)
	}

	// Выбранное должно быть не больше максимального
	fitSelectedIntoAll(filterResponse.DepartureAndArrival.ForwardDeparture)

	t.needToApplyFilter = needToApplyFilter(filterResponse.DepartureAndArrival.ForwardDeparture)

	return filterResponse
}

func (t *timeForwardDepartureFilter) Filter(
	filters *aviaAPI.SearchFiltersReq,
	snippets map[string]*aviaSearchProto.Snippet,
	reference *aviaSearchProto.Reference,
	filterResponse *aviaAPI.SearchFiltersRsp,
) *filterinterface.ExcludedKeys {
	excludedKeys := filterinterface.NewExcludedKeys()
	if !t.needToApplyFilter {
		return excludedKeys
	}

	for sKey, s := range snippets {
		if !inInterval(reference.Flights[s.Forward[0]].DepartureStr, filterResponse.DepartureAndArrival.ForwardDeparture.Selected) {
			excludedKeys.AddSnippetKey(sKey)
		}
	}
	return excludedKeys
}

func (t *timeForwardDepartureFilter) UpdateFilterResponse(
	snippets map[string]*aviaSearchProto.Snippet,
	excludedKeysByOthers *filterinterface.ExcludedKeys,
	reference *aviaSearchProto.Reference,
	filterResponse *aviaAPI.SearchFiltersRsp,
) *aviaAPI.SearchFiltersRsp {
	return filterResponse
}
