package transfer

import (
	aviaAPI "a.yandex-team.ru/travel/app/backend/api/avia/v1"
	"a.yandex-team.ru/travel/app/backend/internal/avia/search/filtering2/filterinterface"
	aviaSearchProto "a.yandex-team.ru/travel/app/backend/internal/avia/search/proto/v1"
	"a.yandex-team.ru/travel/library/go/containers"
)

type TransferAirportsForward struct {
	selectedAirportsForward containers.Set[uint64]
}

func (tf *TransferAirportsForward) GetFilterID() string {
	return "TransferAirportsForward"
}

func NewTransferAirportsForward(filters *aviaAPI.SearchFiltersReq) *TransferAirportsForward {
	tf := &TransferAirportsForward{}

	tf.selectedAirportsForward = containers.SetOf[uint64]()
	if filters != nil && filters.Transfer != nil && filters.Transfer.Airports != nil {
		for _, a := range filters.Transfer.Airports.Forward {
			if a.State {
				tf.selectedAirportsForward[a.StationId] = struct{}{}
			}
		}
	}

	return tf
}

func (tf *TransferAirportsForward) InitFilterResponse(
	filters *aviaAPI.SearchFiltersReq,
	snippets map[string]*aviaSearchProto.Snippet,
	reference *aviaSearchProto.Reference,
	searchContext *aviaSearchProto.SearchContext,
	filterResponse *aviaAPI.SearchFiltersRsp,
) *aviaAPI.SearchFiltersRsp {
	if filterResponse.Transfer == nil {
		filterResponse.Transfer = &aviaAPI.SearchFiltersRsp_TransferFilter{}
	}
	if filterResponse.Transfer.Airports == nil {
		filterResponse.Transfer.Airports = &aviaAPI.SearchFiltersRsp_TransferAirportsFilter{}
	}

	airportsForward := containers.SetOf[uint64]()
	for _, s := range snippets {
		for _, t := range s.Transfers.ForwardTransfers {
			airportsForward.Add(t.ArrivalStationId)
			airportsForward.Add(t.DepartureStationId)
		}
	}

	filterResponse.Transfer.Airports.Forward = buildInitialDirectionAirportsResponse(
		airportsForward,
		tf.selectedAirportsForward,
		reference,
	)

	return filterResponse
}

func (tf *TransferAirportsForward) needToSkip(transfers *aviaSearchProto.Transfers) bool {
	for _, t := range transfers.ForwardTransfers {
		if tf.selectedAirportsForward.Contains(t.ArrivalStationId) {
			return true
		}
		if tf.selectedAirportsForward.Contains(t.DepartureStationId) {
			return true
		}
	}
	return false
}

func (tf *TransferAirportsForward) Filter(
	filters *aviaAPI.SearchFiltersReq,
	snippets map[string]*aviaSearchProto.Snippet,
	reference *aviaSearchProto.Reference,
	filterResponse *aviaAPI.SearchFiltersRsp,
) *filterinterface.ExcludedKeys {
	excludedKeys := filterinterface.NewExcludedKeys()
	for sKey, s := range snippets {
		if tf.needToSkip(s.Transfers) {
			excludedKeys.AddSnippetKey(sKey)
		}
	}
	return excludedKeys
}

func (tf *TransferAirportsForward) UpdateFilterResponse(
	snippets map[string]*aviaSearchProto.Snippet,
	excludedKeysByOthers *filterinterface.ExcludedKeys,
	reference *aviaSearchProto.Reference,
	filterResponse *aviaAPI.SearchFiltersRsp,
) *aviaAPI.SearchFiltersRsp {
	airportsForward := containers.SetOf[uint64]()
	for sKey, s := range snippets {
		needToSkip := excludedKeysByOthers.ContainsSnippetKey(sKey)
		if needToSkip {
			continue
		}
		for _, t := range s.Transfers.ForwardTransfers {
			airportsForward.Add(t.ArrivalStationId)
			airportsForward.Add(t.DepartureStationId)
		}
	}
	for _, a := range filterResponse.Transfer.Airports.Forward {
		if !airportsForward.Contains(a.StationId) {
			a.State.Enabled = false
		}
	}

	return filterResponse
}
