package transfer

import (
	aviaAPI "a.yandex-team.ru/travel/app/backend/api/avia/v1"
	"a.yandex-team.ru/travel/app/backend/internal/avia/search/filtering2/filterinterface"
	aviaSearchProto "a.yandex-team.ru/travel/app/backend/internal/avia/search/proto/v1"
)

type TransferDurationFilter struct {
}

func (tf *TransferDurationFilter) GetFilterID() string {
	return "TransferDurationFilter"
}

func NewTransferDurationFilter() *TransferDurationFilter {
	return &TransferDurationFilter{}
}

func (tf *TransferDurationFilter) InitFilterResponse(
	filters *aviaAPI.SearchFiltersReq,
	snippets map[string]*aviaSearchProto.Snippet,
	reference *aviaSearchProto.Reference,
	searchContext *aviaSearchProto.SearchContext,
	filterResponse *aviaAPI.SearchFiltersRsp,
) *aviaAPI.SearchFiltersRsp {
	if filterResponse.Transfer == nil {
		filterResponse.Transfer = &aviaAPI.SearchFiltersRsp_TransferFilter{}
	}

	o := NewTransferDurationObserver2()
	for _, s := range snippets {
		o.Observe(s.Transfers)
	}

	if o.minTransferDuration == nil || o.maxTransferDuration == nil {
		return filterResponse
	}

	filterResponse.Transfer.TransferDuration = &aviaAPI.SearchFiltersRsp_TransferDurationFilter{
		All: &aviaAPI.SearchFiltersRsp_TransferDurationInterval{
			MinimumMinutes: *o.minTransferDuration,
			MaximumMinutes: *o.maxTransferDuration,
		},
		Selected: &aviaAPI.SearchFiltersRsp_TransferDurationInterval{
			MinimumMinutes: *o.minTransferDuration,
			MaximumMinutes: *o.maxTransferDuration,
		},
	}

	if filters == nil || filters.Transfer == nil || filters.Transfer.TransferDuration == nil {
		return filterResponse
	}
	minSelected := filters.Transfer.TransferDuration.MinimumMinutes
	if minSelected > *o.minTransferDuration {
		filterResponse.Transfer.TransferDuration.Selected.MinimumMinutes = minSelected
	}
	maxSelected := filters.Transfer.TransferDuration.MaximumMinutes
	if maxSelected < *o.maxTransferDuration {
		filterResponse.Transfer.TransferDuration.Selected.MaximumMinutes = maxSelected
	}

	return filterResponse
}

func (tf *TransferDurationFilter) needToSkip(filterResponse *aviaAPI.SearchFiltersRsp, transfers *aviaSearchProto.Transfers) bool {
	for _, t := range transfers.ForwardTransfers {
		if filterResponse.Transfer.TransferDuration.Selected != nil {
			if t.DurationMinutes < filterResponse.Transfer.TransferDuration.Selected.MinimumMinutes {
				return true
			}
			if t.DurationMinutes > filterResponse.Transfer.TransferDuration.Selected.MaximumMinutes {
				return true
			}
		}
	}
	for _, t := range transfers.BackwardTransfers {
		if filterResponse.Transfer.TransferDuration.Selected != nil {
			if t.DurationMinutes < filterResponse.Transfer.TransferDuration.Selected.MinimumMinutes {
				return true
			}
			if t.DurationMinutes > filterResponse.Transfer.TransferDuration.Selected.MaximumMinutes {
				return true
			}
		}
	}
	return false
}

func (tf *TransferDurationFilter) Filter(
	filters *aviaAPI.SearchFiltersReq,
	snippets map[string]*aviaSearchProto.Snippet,
	reference *aviaSearchProto.Reference,
	filterResponse *aviaAPI.SearchFiltersRsp,
) *filterinterface.ExcludedKeys {
	excludedKeys := filterinterface.NewExcludedKeys()
	for sKey, s := range snippets {
		if tf.needToSkip(filterResponse, s.Transfers) {
			excludedKeys.AddSnippetKey(sKey)
		}
	}
	return excludedKeys
}

func (tf *TransferDurationFilter) UpdateFilterResponse(
	snippets map[string]*aviaSearchProto.Snippet,
	excludedKeysByOthers *filterinterface.ExcludedKeys,
	reference *aviaSearchProto.Reference,
	filterResponse *aviaAPI.SearchFiltersRsp,
) *aviaAPI.SearchFiltersRsp {
	return filterResponse
}

type TransferDurationObserver2 struct {
	minTransferDuration *uint32
	maxTransferDuration *uint32
}

func NewTransferDurationObserver2() *TransferDurationObserver2 {
	return &TransferDurationObserver2{
		minTransferDuration: nil,
		maxTransferDuration: nil,
	}
}

func (o *TransferDurationObserver2) Observe(transfers *aviaSearchProto.Transfers) {
	for _, t := range transfers.ForwardTransfers {
		if o.minTransferDuration == nil || *o.minTransferDuration > t.DurationMinutes {
			o.minTransferDuration = &t.DurationMinutes
		}
		if o.maxTransferDuration == nil || *o.maxTransferDuration < t.DurationMinutes {
			o.maxTransferDuration = &t.DurationMinutes
		}
	}
	for _, t := range transfers.BackwardTransfers {
		if o.minTransferDuration == nil || *o.minTransferDuration > t.DurationMinutes {
			o.minTransferDuration = &t.DurationMinutes
		}
		if o.maxTransferDuration == nil || *o.maxTransferDuration < t.DurationMinutes {
			o.maxTransferDuration = &t.DurationMinutes
		}
	}
}

func (o *TransferDurationObserver2) GetMaxTransferDuration() *uint32 {
	return o.maxTransferDuration
}

func (o *TransferDurationObserver2) GetMinTransferDuration() *uint32 {
	return o.minTransferDuration
}
