package transfer

import (
	aviaAPI "a.yandex-team.ru/travel/app/backend/api/avia/v1"
	"a.yandex-team.ru/travel/app/backend/internal/avia/search/filtering2/filterinterface"
	aviaSearchProto "a.yandex-team.ru/travel/app/backend/internal/avia/search/proto/v1"
)

type NoAirportChangeFilter struct {
}

func (tf *NoAirportChangeFilter) GetFilterID() string {
	return "NoAirportChangeFilter"
}

func NewNoAirportChangeFilter() *NoAirportChangeFilter {
	return &NoAirportChangeFilter{}
}

func (tf *NoAirportChangeFilter) InitFilterResponse(
	filters *aviaAPI.SearchFiltersReq,
	snippets map[string]*aviaSearchProto.Snippet,
	reference *aviaSearchProto.Reference,
	searchContext *aviaSearchProto.SearchContext,
	filterResponse *aviaAPI.SearchFiltersRsp,
) *aviaAPI.SearchFiltersRsp {
	if filterResponse.Transfer == nil {
		filterResponse.Transfer = &aviaAPI.SearchFiltersRsp_TransferFilter{}
	}

	filterResponse.Transfer.NoAirportChange = &aviaAPI.SearchFiltersRsp_BoolFilterState{
		Enabled: true,
		Value:   false,
	}
	if filters != nil && filters.Transfer != nil {
		filterResponse.Transfer.NoAirportChange.Value = filters.Transfer.NoAirportChange
	}

	return filterResponse
}

func (tf *NoAirportChangeFilter) needToSkip(filterResponse *aviaAPI.SearchFiltersRsp, transfers *aviaSearchProto.Transfers) bool {
	for _, t := range transfers.ForwardTransfers {
		if t.AirportChange && filterResponse.Transfer.NoAirportChange.Value {
			return true
		}
	}
	for _, t := range transfers.BackwardTransfers {
		if t.AirportChange && filterResponse.Transfer.NoAirportChange.Value {
			return true
		}
	}
	return false
}

func (tf *NoAirportChangeFilter) Filter(
	filters *aviaAPI.SearchFiltersReq,
	snippets map[string]*aviaSearchProto.Snippet,
	reference *aviaSearchProto.Reference,
	filterResponse *aviaAPI.SearchFiltersRsp,
) *filterinterface.ExcludedKeys {
	excludedKeys := filterinterface.NewExcludedKeys()
	for sKey, s := range snippets {
		if tf.needToSkip(filterResponse, s.Transfers) {
			excludedKeys.AddSnippetKey(sKey)
		}
	}
	return excludedKeys
}

func (tf *NoAirportChangeFilter) UpdateFilterResponse(
	snippets map[string]*aviaSearchProto.Snippet,
	excludedKeysByOthers *filterinterface.ExcludedKeys,
	reference *aviaSearchProto.Reference,
	filterResponse *aviaAPI.SearchFiltersRsp,
) *aviaAPI.SearchFiltersRsp {
	hasNoAirportChange := false
	for sKey, s := range snippets {
		needToSkip := excludedKeysByOthers.ContainsSnippetKey(sKey)
		if needToSkip {
			continue
		}
		if withoutAirportChange(s) {
			hasNoAirportChange = true
			break
		}
	}

	filterResponse.Transfer.NoAirportChange.Enabled = hasNoAirportChange

	return filterResponse
}

func withoutAirportChange(s *aviaSearchProto.Snippet) bool {
	if len(s.Transfers.ForwardTransfers) == 0 && len(s.Transfers.BackwardTransfers) == 0 {
		// Нет пересадок - значит нет и смены аэропорта
		return true
	}

	for _, t := range s.Transfers.ForwardTransfers {
		if t.AirportChange {
			return false
		}
	}
	for _, t := range s.Transfers.BackwardTransfers {
		if t.AirportChange {
			return false
		}
	}
	return true
}
