package transfer

import (
	aviaAPI "a.yandex-team.ru/travel/app/backend/api/avia/v1"
	commonAPI "a.yandex-team.ru/travel/app/backend/api/common/v1"
	"a.yandex-team.ru/travel/app/backend/internal/avia/search/filtering2/filterinterface"
	aviaSearchProto "a.yandex-team.ru/travel/app/backend/internal/avia/search/proto/v1"
	"a.yandex-team.ru/travel/app/backend/internal/avia/search/searchcommon"
)

type NoTransferFilter struct {
}

func (tf *NoTransferFilter) GetFilterID() string {
	return "NoTransferFilter"
}

func NewNoTransferFilter() *NoTransferFilter {
	return &NoTransferFilter{}
}

func (tf *NoTransferFilter) InitFilterResponse(
	filters *aviaAPI.SearchFiltersReq,
	snippets map[string]*aviaSearchProto.Snippet,
	reference *aviaSearchProto.Reference,
	searchContext *aviaSearchProto.SearchContext,
	filterResponse *aviaAPI.SearchFiltersRsp,
) *aviaAPI.SearchFiltersRsp {
	filterResponse.QuickTransfer = &aviaAPI.SearchFiltersRsp_QuickTransferFilter{}
	if filterResponse.Transfer == nil {
		filterResponse.Transfer = &aviaAPI.SearchFiltersRsp_TransferFilter{}
	}

	f := &aviaAPI.SearchFiltersRsp_BoolFilterState{
		Enabled: true,
		Value:   false,
	}
	if filters != nil {
		f.Value = filters.QuickTransfer
	}
	filterResponse.QuickTransfer.State = f
	filterResponse.Transfer.NoTransfer = f

	return filterResponse
}

func (tf *NoTransferFilter) needToSkip(filterResponse *aviaAPI.SearchFiltersRsp, transfers *aviaSearchProto.Transfers) bool {
	if filterResponse.QuickTransfer.State.Value && (len(transfers.ForwardTransfers) != 0 || len(transfers.BackwardTransfers) != 0) {
		return true
	}
	return false
}

func (tf *NoTransferFilter) Filter(
	filters *aviaAPI.SearchFiltersReq,
	snippets map[string]*aviaSearchProto.Snippet,
	reference *aviaSearchProto.Reference,
	filterResponse *aviaAPI.SearchFiltersRsp,
) *filterinterface.ExcludedKeys {
	excludedKeys := filterinterface.NewExcludedKeys()
	for sKey, s := range snippets {
		if tf.needToSkip(filterResponse, s.Transfers) {
			excludedKeys.AddSnippetKey(sKey)
		}
	}
	return excludedKeys
}

func (tf *NoTransferFilter) UpdateFilterResponse(
	snippets map[string]*aviaSearchProto.Snippet,
	excludedKeysByOthers *filterinterface.ExcludedKeys,
	reference *aviaSearchProto.Reference,
	filterResponse *aviaAPI.SearchFiltersRsp,
) *aviaAPI.SearchFiltersRsp {
	hasDirect := false
	var withTransferMinPrice *commonAPI.Price = nil
	var noTransferMinPrice *commonAPI.Price = nil
	for sKey, s := range snippets {
		needToSkip := excludedKeysByOthers.ContainsSnippetKey(sKey)
		if needToSkip {
			continue
		}
		noTransfers := len(s.Transfers.ForwardTransfers) == 0 && len(s.Transfers.BackwardTransfers) == 0
		if noTransfers {
			hasDirect = true
		}
		for vKey, v := range s.Variant {
			needToSkipVariant := excludedKeysByOthers.ContainsVariantKey(sKey, vKey)
			if needToSkipVariant {
				continue
			}
			price := &commonAPI.Price{
				Currency: v.Price.Currency,
				Value:    v.Price.Value,
			}
			if noTransfers {
				if noTransferMinPrice == nil || searchcommon.CompareOnlyPriceAsc(price, noTransferMinPrice) {
					noTransferMinPrice = price
				}
			} else {
				if withTransferMinPrice == nil || searchcommon.CompareOnlyPriceAsc(price, withTransferMinPrice) {
					withTransferMinPrice = price
				}
			}
		}
	}
	if !hasDirect {
		filterResponse.QuickTransfer.State.Enabled = false
		filterResponse.Transfer.NoTransfer.Enabled = false
	}

	filterResponse.QuickTransfer.MinPriceNoTransfer = noTransferMinPrice
	filterResponse.QuickTransfer.MinPriceWithTransfer = withTransferMinPrice

	return filterResponse
}
