package factories

import (
	aviaSearchProto "a.yandex-team.ru/travel/app/backend/internal/avia/search/proto/v1"
)

type Reference struct {
	Flights       map[string]*aviaSearchProto.Flight
	Partners      map[string]*aviaSearchProto.Partner
	Settlements   map[uint64]*aviaSearchProto.Settlement
	Stations      map[uint64]*aviaSearchProto.Station
	AviaCompanies map[uint64]*aviaSearchProto.AviaCompany
	Alliances     map[uint64]*aviaSearchProto.Alliance
}

func NewReference() *Reference {
	return &Reference{
		Flights:       make(map[string]*aviaSearchProto.Flight),
		Partners:      make(map[string]*aviaSearchProto.Partner),
		Settlements:   make(map[uint64]*aviaSearchProto.Settlement),
		Stations:      make(map[uint64]*aviaSearchProto.Station),
		AviaCompanies: make(map[uint64]*aviaSearchProto.AviaCompany),
		Alliances:     make(map[uint64]*aviaSearchProto.Alliance),
	}
}

func (r *Reference) CreateFlight(
	ac *aviaSearchProto.AviaCompany,
	number string,
	from *aviaSearchProto.Station,
	to *aviaSearchProto.Station,
	da string, // departure & arrival, пример "9:30 - 11:40"
	opts ...CreateFlightOption,
) *aviaSearchProto.Flight {
	f := CreateFlight(ac, number, from, to, da, opts...)
	if _, found := r.AviaCompanies[ac.Id]; !found {
		r.AviaCompanies[ac.Id] = ac
	}
	if _, found := r.Stations[from.Id]; !found {
		r.Stations[from.Id] = from
	}
	if _, found := r.Stations[to.Id]; !found {
		r.Stations[to.Id] = to
	}
	r.Flights[f.Key] = f
	return f
}

func (r *Reference) CreatePartner(code string, opts ...CreatePartnerOption) *aviaSearchProto.Partner {
	p := CreatePartner(code, opts...)
	r.Partners[p.Code] = p
	return p
}

func (r *Reference) AddSettlement(s *aviaSearchProto.Settlement) {
	r.Settlements[s.Id] = s
}

func (r *Reference) CreateSettlement(id uint64, opts ...CreateSettlementOption) *aviaSearchProto.Settlement {
	s := CreateSettlement(id, opts...)
	r.AddSettlement(s)
	return s
}

func (r *Reference) AddStation(s *aviaSearchProto.Station) {
	r.Stations[s.Id] = s
}

func (r *Reference) CreateStation(id uint64, opts ...CreateStationOption) *aviaSearchProto.Station {
	s := CreateStation(id, opts...)
	r.AddStation(s)
	return s
}

func (r *Reference) CreateAviaCompany(id uint64, opts ...AviaCompanyOption) *aviaSearchProto.AviaCompany {
	ac := CreateAviaCompany(id, opts...)
	r.AviaCompanies[ac.Id] = ac
	return ac
}

func (r *Reference) CreateAlliance(id uint64, opts ...AllianceOption) *aviaSearchProto.Alliance {
	a := CreateAlliance(id, opts...)
	r.Alliances[a.Id] = a
	return a
}

func (r *Reference) GetProto() *aviaSearchProto.Reference {
	return &aviaSearchProto.Reference{
		Flights:       r.Flights,
		Partners:      r.Partners,
		Settlements:   r.Settlements,
		Stations:      r.Stations,
		AviaCompanies: r.AviaCompanies,
		Alliances:     r.Alliances,
	}
}
