package factories

import (
	"time"

	aviaSearchProto "a.yandex-team.ru/travel/app/backend/internal/avia/search/proto/v1"
)

type SnippetOption func(*aviaSearchProto.Snippet) *aviaSearchProto.Snippet

func CreateSnippet(key string, forward, backward []string, ref *Reference, variants map[string]*aviaSearchProto.Variant, opts ...SnippetOption) *aviaSearchProto.Snippet {
	s := &aviaSearchProto.Snippet{
		Key:                     key,
		Forward:                 forward,
		Backward:                backward,
		Transfers:               buildTransfers(forward, backward, ref),
		Variant:                 variants,
		ForwardDurationMinutes:  calculateDurationMinutes(forward, ref),
		BackwardDurationMinutes: calculateDurationMinutes(backward, ref),
	}

	for _, opt := range opts {
		opt(s)
	}
	return s
}

func calculateDurationMinutes(flights []string, ref *Reference) uint32 {
	if len(flights) == 0 {
		return 0
	}

	departure, _ := time.Parse(time.RFC3339, ref.Flights[flights[0]].DepartureStr)
	arrival, _ := time.Parse(time.RFC3339, ref.Flights[flights[len(flights)-1]].ArrivalStr)
	return uint32(arrival.Sub(departure).Minutes())
}

func buildTransfers(forward, backward []string, ref *Reference) *aviaSearchProto.Transfers {
	return &aviaSearchProto.Transfers{
		ForwardTransfers:  buildTransfersForDirection(forward, ref),
		BackwardTransfers: buildTransfersForDirection(backward, ref),
	}
}

func buildTransfersForDirection(flights []string, ref *Reference) []*aviaSearchProto.Transfer {
	if len(flights) == 0 {
		return make([]*aviaSearchProto.Transfer, 0)
	}
	transfers := make([]*aviaSearchProto.Transfer, len(flights)-1)
	for i := 0; i < len(flights)-1; i += 1 {
		previousFlight := ref.Flights[flights[i]]
		flight := ref.Flights[flights[i+1]]
		arrival, _ := time.Parse(time.RFC3339, previousFlight.ArrivalStr)
		departure, _ := time.Parse(time.RFC3339, flight.DepartureStr)
		transfer := &aviaSearchProto.Transfer{
			ArrivalStationId:   previousFlight.StationToId,
			DepartureStationId: flight.StationFromId,
			SelfConnect:        false,
			NightTransfer:      arrival.Day() != departure.Day(),
			AirportChange:      previousFlight.StationToId != flight.StationFromId,
			ToTrain:            false,
			ToBus:              false,
			DurationMinutes:    uint32(departure.Sub(arrival).Minutes()),
		}
		transfers[i] = transfer
	}
	return transfers
}
