package searchcommon

import (
	"fmt"
	"strconv"
	"time"

	"google.golang.org/genproto/googleapis/type/date"
	"google.golang.org/protobuf/types/known/timestamppb"

	aviaSearchProto "a.yandex-team.ru/travel/app/backend/internal/avia/search/proto/v1"
	"a.yandex-team.ru/travel/avia/library/go/searchcontext"
	commonV2 "a.yandex-team.ru/travel/avia/library/proto/common/v2"
)

func ParseQIDToProto(qid string) (*aviaSearchProto.SearchContext, error) {
	parsedQID, err := searchcontext.ParseQID(qid)
	if err != nil {
		return nil, err
	}
	pointFrom, err := parsePointKeyToProto(parsedQID.QKey.PointFromKey)
	if err != nil {
		return nil, err
	}
	pointTo, err := parsePointKeyToProto(parsedQID.QKey.PointToKey)
	if err != nil {
		return nil, err
	}
	serviceClass, err := parseServiceClassToProto(parsedQID.QKey.Class)
	if err != nil {
		return nil, err
	}

	return &aviaSearchProto.SearchContext{
		Qid:               qid,
		CreatedAt:         timestamppb.New(parsedQID.CreatedAt),
		Service:           parsedQID.Service,
		TransportTypeCode: parsedQID.TCode,
		Language:          parsedQID.Lang,
		PointFrom:         pointFrom,
		PointTo:           pointTo,
		DateForward:       dateToProto(parsedQID.QKey.DateForward),
		DateBackward:      dateToProto(parsedQID.QKey.DateBackward),
		Passengers: &commonV2.Passengers{
			Adults:   uint32(parsedQID.QKey.Adults),
			Children: uint32(parsedQID.QKey.Children),
			Infants:  uint32(parsedQID.QKey.Infants),
		},
		NationalVersion: parsedQID.QKey.NationalVersion,
		ServiceClass:    serviceClass,
	}, nil
}

func dateToProto(d time.Time) *date.Date {
	if d.IsZero() {
		return nil
	}
	year, month, day := d.Date()
	return &date.Date{
		Year:  int32(year),
		Month: int32(month),
		Day:   int32(day),
	}
}

func parsePointKeyToProto(pointKey string) (*commonV2.Point, error) {
	if len(pointKey) < 2 {
		return nil, fmt.Errorf("invalid point key %s", pointKey)
	}
	id, err := strconv.Atoi(pointKey[1:])
	if err != nil {
		return nil, fmt.Errorf("invalid point key %s: %w", pointKey, err)
	}
	var pointType commonV2.PointType
	switch pointKey[0] {
	case 'l':
		{
			pointType = commonV2.PointType_POINT_TYPE_COUNTRY
		}
	case 'r':
		{
			pointType = commonV2.PointType_POINT_TYPE_REGION
		}
	case 'c':
		{
			pointType = commonV2.PointType_POINT_TYPE_SETTLEMENT
		}
	case 's':
		{
			pointType = commonV2.PointType_POINT_TYPE_STATION
		}
	default:
		return nil, fmt.Errorf("invalid point key %s", pointKey)
	}
	return &commonV2.Point{
		Type: pointType,
		Id:   uint64(id),
	}, nil
}

func parseServiceClassToProto(serviceClass string) (commonV2.ServiceClass, error) {
	switch serviceClass {
	case "economy":
		{
			return commonV2.ServiceClass_SERVICE_CLASS_ECONOMY, nil
		}
	case "business":
		{
			return commonV2.ServiceClass_SERVICE_CLASS_BUSINESS, nil
		}
	}
	return commonV2.ServiceClass_SERVICE_CLASS_UNKNOWN, fmt.Errorf("invalid serviceclass %s", serviceClass)
}
