package search

import (
	"context"
	"fmt"
	"net/url"
	"strconv"
	"strings"
	"time"

	"github.com/mitchellh/mapstructure"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/ctxlog"
	"a.yandex-team.ru/library/go/core/xerrors"
	aviaAPI "a.yandex-team.ru/travel/app/backend/api/avia/v1"
	commonAPI "a.yandex-team.ru/travel/app/backend/api/common/v1"
	"a.yandex-team.ru/travel/app/backend/internal/common"
	"a.yandex-team.ru/travel/app/backend/internal/lib/aviatdapiclient"
	"a.yandex-team.ru/travel/app/backend/pkg/hashutils"
	"a.yandex-team.ru/travel/avia/library/go/searchcontext"
)

type ServiceSearch struct {
	logger log.Logger
}

func NewServiceSearch(logger log.Logger) *ServiceSearch {
	return &ServiceSearch{
		logger: logger,
	}
}

func BuildFlightsReference(flights []aviatdapiclient.Flight) (map[string]*aviaAPI.SearchResultReference_Flight, error) {
	result := make(map[string]*aviaAPI.SearchResultReference_Flight, len(flights))
	for _, flight := range flights {
		departureStr := flight.Departure.ISO8601()
		departure, err := time.Parse(time.RFC3339, departureStr)
		if err != nil {
			return nil, xerrors.Errorf("can't parse departure '%s'", departureStr)
		}
		arrivalStr := flight.Arrival.ISO8601()
		arrival, err := time.Parse(time.RFC3339, arrivalStr)
		if err != nil {
			return nil, xerrors.Errorf("can't parse arrival '%s'", arrivalStr)
		}
		result[flight.Key] = &aviaAPI.SearchResultReference_Flight{
			Key:             flight.Key,
			AviaCompanyId:   flight.CompanyID,
			Number:          flight.Number,
			StationFromId:   flight.StationFromID,
			StationToId:     flight.StationToID,
			Departure:       departureStr,
			Arrival:         arrivalStr,
			DurationMinutes: uint32(arrival.Sub(departure).Minutes()),
			DateChanged:     departure.Day() != arrival.Day(),
		}
	}
	return result, nil
}

func BuildPartnersReference(partners []aviatdapiclient.Partner) map[string]*aviaAPI.SearchResultReference_Partner {
	result := make(map[string]*aviaAPI.SearchResultReference_Partner, len(partners))
	for _, partner := range partners {
		result[partner.Code] = &aviaAPI.SearchResultReference_Partner{
			Code:    partner.Code,
			Title:   partner.Title,
			LogoSvg: partner.LogoSVG,
			LogoPng: partner.LogoPNG,
		}
	}
	return result
}

func BuildSettlementsReference(settlements []aviatdapiclient.Settlement) map[uint64]*aviaAPI.SearchResultReference_Settlement {
	result := make(map[uint64]*aviaAPI.SearchResultReference_Settlement, len(settlements))
	for _, city := range settlements {
		result[city.ID] = &aviaAPI.SearchResultReference_Settlement{
			Id:               city.ID,
			Title:            city.Title,
			TitleGenitive:    city.TitleGenitive,
			TitleAccusative:  city.TitleAccusative,
			TitlePreposition: city.Preposition,
			TitleLocative:    city.TitleLocative,
		}
	}
	return result
}

func BuildStationsReference(stations []aviatdapiclient.Station) map[uint64]*aviaAPI.SearchResultReference_Station {
	result := make(map[uint64]*aviaAPI.SearchResultReference_Station, len(stations))
	for _, station := range stations {
		var settlementID uint64 = 0
		if station.SettlementID != nil {
			settlementID = *station.SettlementID
		}
		result[station.ID] = &aviaAPI.SearchResultReference_Station{
			Id:               station.ID,
			AviaCode:         station.AviaCode,
			SettlementId:     settlementID,
			Title:            station.Title,
			TitleGenitive:    station.TitleGenitive,
			TitleAccusative:  station.TitleAccusative,
			TitlePreposition: station.Preposition,
			TitleLocative:    station.TitleLocative,
		}
	}
	return result
}

func BuildAviaCompaniesReference(companies []aviatdapiclient.Company) (map[uint64]*aviaAPI.SearchResultReference_AviaCompany, error) {
	result := make(map[uint64]*aviaAPI.SearchResultReference_AviaCompany, len(companies))
	// Для полей, которые отдаем наружу, достаточно company
	for _, company := range companies {
		var allianceID uint64 = 0
		if company.AllianceID != nil {
			allianceID = *company.AllianceID
		}
		result[company.ID] = &aviaAPI.SearchResultReference_AviaCompany{
			Id:         company.ID,
			AllianceId: allianceID,
			Title:      company.Title,
			LogoSvg:    company.LogoSVG,
			LogoPng:    company.LogoPNG,
			Color:      company.Color,
		}
	}
	return result, nil
}

func BuildAlliancesReference(alliances []aviatdapiclient.Alliance) map[uint64]*aviaAPI.SearchResultReference_Alliance {
	result := make(map[uint64]*aviaAPI.SearchResultReference_Alliance, len(alliances))
	for _, alliance := range alliances {
		result[alliance.ID] = &aviaAPI.SearchResultReference_Alliance{
			Id:    alliance.ID,
			Title: alliance.Title,
		}
	}
	return result
}

func (s *ServiceSearch) BuildBaggage(ctx context.Context, fareKeys [][]string, fareFamilies map[string]aviatdapiclient.FareFamily, baggage [][]string, baggageTariff map[string]aviatdapiclient.BaggageTariff) *aviaAPI.Snippet_Baggage {
	result := &aviaAPI.Snippet_Baggage{
		Included:       false,
		OptionalWeight: nil,
		OptionalPieces: nil,
	}

	baggageTerms := s.findTerms(ctx, aviatdapiclient.FFTermCodeBaggage, fareKeys, fareFamilies)
	if len(baggageTerms) == 0 {
		return result
	}

	var overallWeight *int
	var overallPieces *int

	for _, term := range baggageTerms {
		if term.Term != nil {
			var rule aviatdapiclient.FFTermRuleBaggage
			err := mapstructure.Decode(term.Term.Rule, &rule)
			if err == nil {
				if rule.Places != nil && *rule.Places > 0 {
					overallPieces = min(overallPieces, rule.Places)
					if rule.Weight != nil {
						overallWeight = min(overallWeight, rule.Weight)
					}
				}
				continue
			}
		}
		if term.Row >= len(baggage) || term.Col >= len(baggage[term.Row]) {
			continue
		}
		code := baggage[term.Row][term.Col]
		if code == "" {
			continue
		}
		if tariff, ok := baggageTariff[code]; ok {
			overallPieces = min(overallPieces, &tariff.Pieces.Count)
			if tariff.Weight.Count > 0 {
				overallWeight = min(overallWeight, &tariff.Weight.Count)
			}
		} else {
			ctxlog.Error(ctx, s.logger, fmt.Sprintf("not found code %v in baggageTariff", code))
		}
	}

	included := overallPieces != nil && *overallPieces > 0
	result.Included = included
	if included {
		if overallWeight != nil {
			result.OptionalWeight = &aviaAPI.Snippet_Baggage_Weight{Weight: uint32(*overallWeight)}
		}
		if overallPieces != nil {
			result.OptionalPieces = &aviaAPI.Snippet_Baggage_Pieces{Pieces: uint32(*overallPieces)}
		}
	}
	return result
}

func (s *ServiceSearch) BuildCarryOn(ctx context.Context, fareKeys [][]string, fareFamilies map[string]aviatdapiclient.FareFamily, route [][]string, flights []aviatdapiclient.Flight, companyTariffs []aviatdapiclient.CompanyTariff) *aviaAPI.Snippet_CarryOn {
	result := &aviaAPI.Snippet_CarryOn{
		Included:       false,
		OptionalWeight: nil,
	}
	carryOnTerms := s.findTerms(ctx, aviatdapiclient.FFTermCodeCarryOn, fareKeys, fareFamilies)
	if len(carryOnTerms) == 0 {
		return result
	}

	var places *int
	var weight *int

	for _, term := range carryOnTerms {
		if term.Term != nil {
			var rule aviatdapiclient.FFTermRuleCarryOn
			err := mapstructure.Decode(term.Term.Rule, &rule)
			if err == nil {
				if rule.Places == nil {
					continue
				}
				places = min(places, rule.Places)
				if rule.Weight != nil {
					weight = min(weight, rule.Weight)
				}
				continue
			}
		}
		if term.Row >= len(route) || term.Col >= len(route[term.Row]) {
			continue
		}
		r := route[term.Row][term.Col]
		//TODO(adurnev)  flights преобразовывать в map? https://a.yandex-team.ru/review/2468461/details#comment-3395304
		for _, flight := range flights {
			if flight.Key != r {
				continue
			}
			for _, companyTariff := range companyTariffs {
				if companyTariff.ID == flight.CompanyTariffID {
					if companyTariff.CarryOn {
						p := 1
						places = min(places, &p)
						w := int(companyTariff.CarryOnNorm)
						if w > 0 {
							weight = min(weight, &w)
						}
					} else {
						return result
					}
					break
				}
			}
			break
		}
	}

	included := places != nil && *places > 0
	result.Included = included
	if included && weight != nil {
		result.OptionalWeight = &aviaAPI.Snippet_CarryOn_Weight{Weight: uint32(*weight)}
	}
	return result
}

func (s *ServiceSearch) BuildRefund(ctx context.Context, fareKeys [][]string, fareFamilies map[string]aviatdapiclient.FareFamily) *aviaAPI.Snippet_Refund {
	notAvailable := aviaAPI.Snippet_Refund{
		Availability: aviaAPI.Snippet_REFUND_AVAILABILITY_NOT_AVAILABLE,
		Price:        nil,
	}
	var price *commonAPI.Price
	refundableTerms := s.findTerms(ctx, aviatdapiclient.FFTermCodeRefundable, fareKeys, fareFamilies)
	if len(refundableTerms) == 0 {
		return &notAvailable
	}
	for _, term := range refundableTerms {
		if term.Term == nil {
			return &notAvailable
		}
	}

	hasChargeNoPrice := false
	for _, term := range refundableTerms {
		var rule aviatdapiclient.FFTermRuleRefundable
		err := mapstructure.Decode(term.Term.Rule, &rule)
		if err != nil {
			ctxlog.Error(ctx, s.logger, fmt.Sprintf("parse error FFTermRuleRefundable %v", err))
			return &notAvailable
		}
		switch rule.Availability {
		case aviatdapiclient.AvailabilityTypeNotAvailable, aviatdapiclient.AvailabilityTypeUnknown:
			return &notAvailable
		case aviatdapiclient.AvailabilityTypeFree:

		case aviatdapiclient.AvailabilityTypeCharge:
			if price == nil {
				price = &commonAPI.Price{}
			}
			if rule.Charge == nil || rule.Charge.Value == "" {
				hasChargeNoPrice = true
				continue
			}
			v, err := strconv.ParseFloat(rule.Charge.Value, 64)
			if err != nil {
				ctxlog.Error(ctx, s.logger, fmt.Sprintf("parse float %v", rule.Charge.Value))
			}
			price.Value += v
			price.Currency = rule.Charge.Currency // Возврат может быть не в рублях, даже если билет покупаем за рубли
		}
	}

	if price != nil {
		var p *commonAPI.Price
		if !hasChargeNoPrice {
			p = price
		}
		return &aviaAPI.Snippet_Refund{
			Availability: aviaAPI.Snippet_REFUND_AVAILABILITY_CHARGE,
			Price:        p,
		}
	}

	return &aviaAPI.Snippet_Refund{
		Availability: aviaAPI.Snippet_REFUND_AVAILABILITY_FREE,
		Price:        nil,
	}
}

type TermPosition struct {
	Term *aviatdapiclient.FFTerm
	Row  int
	Col  int
}

func (s *ServiceSearch) findTerms(ctx context.Context, code string, fareKeys [][]string, fareFamilies map[string]aviatdapiclient.FareFamily) []*TermPosition {
	result := make([]*TermPosition, 0)
	for i, keys := range fareKeys {
		for j, key := range keys {
			var ffTerm *aviatdapiclient.FFTerm
			if fareFamily, ok := fareFamilies[key]; ok {
				for _, term := range fareFamily.Terms {
					if term.Code == code {
						ffTerm = &term
						break
					}
				}
			} else if key != "" {
				ctxlog.Error(ctx, s.logger, fmt.Sprintf("not found key '%v' in fareFamilies", key))
			}
			result = append(result, &TermPosition{
				Term: ffTerm,
				Row:  i,
				Col:  j,
			})
		}
	}
	return result
}

func BuildSnippetKey(route [][]string) (string, error) {
	routeKeyParts := make([]string, len(route))
	for i, direction := range route {
		routeKeyParts[i] = strings.Join(direction, ",")
	}
	routeLongKey := strings.Join(routeKeyParts, ";")
	routeKey, err := hashutils.GetSimpleHash(routeLongKey)
	if err != nil {
		return "", xerrors.Errorf("can't get hash from %s", routeLongKey)
	}
	return routeKey, nil
}

func CalcDurationMinutes(flights []string, flightReference map[string]aviatdapiclient.Flight) (uint32, error) {
	if len(flights) == 0 {
		return 0, nil
	}
	startFlight, found := flightReference[flights[0]]
	if !found {
		return 0, xerrors.Errorf("can not find flight '%s' in reference", flights[0])
	}
	finishFlight, found := flightReference[flights[len(flights)-1]]
	if !found {
		return 0, xerrors.Errorf("can not find flight '%s' in reference", flights[len(flights)-1])
	}
	departure, err := time.Parse(time.RFC3339, startFlight.Departure.ISO8601())
	if err != nil {
		return 0, xerrors.Errorf("can't parse departure '%s'", startFlight.Departure.ISO8601())
	}
	arrival, err := time.Parse(time.RFC3339, finishFlight.Arrival.ISO8601())
	if err != nil {
		return 0, xerrors.Errorf("can't parse arrival '%s'", finishFlight.Arrival.ISO8601())
	}
	return uint32(arrival.Sub(departure).Minutes()), nil
}

func BuildTransfers(
	forward []string,
	backward []string,
	selfConnect bool,
	flightReference map[string]aviatdapiclient.Flight,
	stationReference map[uint64]aviatdapiclient.Station,
) (*aviaAPI.Snippet_Transfers, error) {
	forwardTransfers, err := BuildTransfersForDirection(forward, selfConnect, flightReference, stationReference)
	if err != nil {
		return nil, xerrors.Errorf("build forward transfers error: %w", err)
	}
	backwardTransfers, err := BuildTransfersForDirection(backward, selfConnect, flightReference, stationReference)
	if err != nil {
		return nil, xerrors.Errorf("build backward transfers error: %w", err)
	}
	return &aviaAPI.Snippet_Transfers{
		ForwardTransfers:  forwardTransfers,
		BackwardTransfers: backwardTransfers,
	}, nil
}

func BuildTransfersForDirection(
	flights []string,
	selfConnect bool,
	flightReference map[string]aviatdapiclient.Flight,
	stationReference map[uint64]aviatdapiclient.Station,
) ([]*aviaAPI.Snippet_Transfer, error) {
	if len(flights) == 0 {
		return make([]*aviaAPI.Snippet_Transfer, 0), nil
	}
	transfers := make([]*aviaAPI.Snippet_Transfer, len(flights)-1)
	for i := 0; i < len(flights)-1; i += 1 {
		previousFlight, found := flightReference[flights[i]]
		if !found {
			return nil, xerrors.Errorf("can not find flight '%s' in reference", flights[i])
		}
		flight, found := flightReference[flights[i+1]]
		if !found {
			return nil, xerrors.Errorf("can not find flight '%s' in reference", flights[i+1])
		}
		arrival, err := time.Parse(time.RFC3339, previousFlight.Arrival.ISO8601())
		if err != nil {
			return nil, xerrors.Errorf("can't parse arrival '%s'", previousFlight.Arrival.ISO8601())
		}
		departure, err := time.Parse(time.RFC3339, flight.Departure.ISO8601())
		if err != nil {
			return nil, xerrors.Errorf("can't parse departure '%s'", flight.Departure.ISO8601())
		}
		stationForTType, found := stationReference[flight.StationToID]
		if !found {
			return nil, xerrors.Errorf("can not find station '%d' in reference", flight.StationToID)
		}
		transfer := aviaAPI.Snippet_Transfer{
			ArrivalStationId:   previousFlight.StationToID,
			DepartureStationId: flight.StationFromID,
			SelfConnect:        selfConnect,
			NightTransfer:      arrival.Day() != departure.Day(),
			AirportChange:      previousFlight.StationToID != flight.StationFromID,
			ToTrain:            stationForTType.TransportType == aviatdapiclient.TrainTransportType,
			ToBus:              stationForTType.TransportType == aviatdapiclient.BusTransportType,
			DurationMinutes:    uint32(departure.Sub(arrival).Minutes()),
			DateChanged:        arrival.Day() != departure.Day(),
		}
		transfers[i] = &transfer
	}
	return transfers, nil
}

func BuildOrderURL(
	qid string,
	forward []string,
	backward []string,
	flightReference map[string]aviatdapiclient.Flight,
	baggageIncluded bool,
	refundAvailability aviaAPI.Snippet_RefundAvailability,
) (string, error) {
	parsedQID, err := searchcontext.ParseQID(qid)
	if err != nil {
		return "", xerrors.Errorf("can't parse qid '%s'", qid)
	}

	forwardParam, err := buildDirectionURLPart(forward, flightReference)
	if err != nil {
		return "", xerrors.Errorf("error building forward param: %w", err)
	}

	isTwoWay := parsedQID.QKey.DateBackward != time.Time{}
	oneway := "1"
	if isTwoWay {
		oneway = "2"
	}
	params := url.Values{
		"adult_seats":    {strconv.Itoa(int(parsedQID.QKey.Adults))},
		"children_seats": {strconv.Itoa(int(parsedQID.QKey.Children))},
		"infant_seats":   {strconv.Itoa(int(parsedQID.QKey.Infants))},
		"fromId":         {parsedQID.QKey.PointFromKey},
		"toId":           {parsedQID.QKey.PointToKey},
		"klass":          {parsedQID.QKey.Class},
		"oneway":         {oneway},
		"when":           {common.FormatDate(parsedQID.QKey.DateForward)},
		"forward":        {forwardParam},
	}
	if isTwoWay {
		backwardParam, err := buildDirectionURLPart(backward, flightReference)
		if err != nil {
			return "", xerrors.Errorf("error building backward param: %w", err)
		}
		params.Add("return_date", common.FormatDate(parsedQID.QKey.DateBackward))
		params.Add("backward", backwardParam)
	}
	if baggageIncluded {
		params.Add("baggage", "1")
	}
	switch refundAvailability {
	case aviaAPI.Snippet_REFUND_AVAILABILITY_FREE:
		params.Add("free_refund", "1")
	case aviaAPI.Snippet_REFUND_AVAILABILITY_CHARGE:
		params.Add("charge_refund", "1")
	}
	return fmt.Sprintf("/avia/order/?%s", params.Encode()), nil
}

func buildDirectionURLPart(flights []string, flightReference map[string]aviatdapiclient.Flight) (string, error) {
	parts := make([]string, len(flights))
	for i, flightKey := range flights {
		flight, found := flightReference[flightKey]
		if !found {
			return "", xerrors.Errorf("can not find flight '%s' in reference", flightKey)
		}
		departure, err := time.Parse(time.RFC3339, flight.Departure.ISO8601())
		if err != nil {
			return "", xerrors.Errorf("can't parse departure '%s'", flight.Departure.ISO8601())
		}
		part := fmt.Sprintf("%s.%s", flight.Number, departure.Format("2006-01-02T15:04"))
		parts[i] = part
	}
	return strings.Join(parts, ","), nil
}

func min(overallValue, value *int) *int {
	if overallValue == nil {
		overallValue = value
	} else {
		overallValue = minPtr(overallValue, value)
	}
	return overallValue
}

func minPtr(a, b *int) *int {
	if a == nil || b == nil {
		return nil
	}
	if *a < *b {
		return a
	}
	return b
}
