package sorting

import (
	"sort"

	aviaApi "a.yandex-team.ru/travel/app/backend/api/avia/v1"
	v1 "a.yandex-team.ru/travel/app/backend/internal/avia/search/proto/v1"
	"a.yandex-team.ru/travel/app/backend/internal/avia/search/searchcommon"
	"a.yandex-team.ru/travel/avia/library/go/services/featureflag"
)

type InterestingSorterConfig struct {
	BadgesOrder             []aviaApi.Snippet_BadgeType
	BadgesToFlagCodeMapping map[aviaApi.Snippet_BadgeType]string
}

var InterestingSorterDefaultConfig = InterestingSorterConfig{
	BadgesOrder: []aviaApi.Snippet_BadgeType{
		aviaApi.Snippet_BADGE_TYPE_BEST_PRICE,
		aviaApi.Snippet_BADGE_TYPE_POPULAR,
		aviaApi.Snippet_BADGE_TYPE_COMFY,
		aviaApi.Snippet_BADGE_TYPE_AVIACOMPANY_DIRECT_SELLING,
		aviaApi.Snippet_BADGE_TYPE_BOOK_ON_YANDEX,
	},
	BadgesToFlagCodeMapping: map[aviaApi.Snippet_BadgeType]string{
		aviaApi.Snippet_BADGE_TYPE_BEST_PRICE:                 "TRAVEL_APP_RECOMMENDED_SORT_ENABLE_BEST_PRICE",
		aviaApi.Snippet_BADGE_TYPE_POPULAR:                    "TRAVEL_APP_RECOMMENDED_SORT_ENABLE_POPULAR",
		aviaApi.Snippet_BADGE_TYPE_COMFY:                      "TRAVEL_APP_RECOMMENDED_SORT_ENABLE_COMFY",
		aviaApi.Snippet_BADGE_TYPE_AVIACOMPANY_DIRECT_SELLING: "TRAVEL_APP_RECOMMENDED_SORT_ENABLE_AVIACOMPANY_DIRECT_SELLING",
		aviaApi.Snippet_BADGE_TYPE_BOOK_ON_YANDEX:             "TRAVEL_APP_RECOMMENDED_SORT_ENABLE_BOOK_ON_YANDEX",
	},
}

type InterestingSorter struct {
	cfg                *InterestingSorterConfig
	featureFlagStorage featureflag.StorageInterface
}

func NewInterestingSorter(cfg *InterestingSorterConfig, featureFlagStorage featureflag.StorageInterface) *InterestingSorter {
	return &InterestingSorter{
		cfg:                cfg,
		featureFlagStorage: featureFlagStorage,
	}
}

func (s *InterestingSorter) GetActualBadgesOrder() []aviaApi.Snippet_BadgeType {
	flags := s.featureFlagStorage.GetFlags()
	resultFlags := make([]aviaApi.Snippet_BadgeType, 0)
	for _, flag := range s.cfg.BadgesOrder {
		if flags.IsFlagEnabled(s.cfg.BadgesToFlagCodeMapping[flag]) {
			resultFlags = append(resultFlags, flag)
		}
	}
	return resultFlags
}

func removeElem(array []*aviaApi.Snippet, index int) []*aviaApi.Snippet {
	return append(array[:index], array[index+1:]...)
}

func (s *InterestingSorter) Sort(
	snippets []*aviaApi.Snippet,
	flights map[string]*v1.Flight,
	_ aviaApi.SearchSort,
) []*aviaApi.Snippet {
	if len(snippets) == 0 {
		return snippets
	}
	sort.SliceStable(snippets, func(i, j int) bool {
		return searchcommon.CompareExtendedPriceAsc(
			snippets[i].Variant.Price, snippets[j].Variant.Price,
			snippets[i].Forward, snippets[j].Forward,
			snippets[i].Backward, snippets[j].Backward,
			flights,
		)
	})

	actualBadgesOrder := s.GetActualBadgesOrder()
	foundIds := make(map[aviaApi.Snippet_BadgeType]int)

	for i, elem := range snippets {
		for _, badge := range elem.Badges {
			for _, badgeType := range actualBadgesOrder {
				if badge.Type == badgeType {
					if _, found := foundIds[badgeType]; !found {
						foundIds[badge.Type] = i
					}
				}
			}
		}
	}

	newPart := make([]*aviaApi.Snippet, 0)
	toRemove := make([]int, 0)

	for i, badgeType := range actualBadgesOrder {
		badgeElemID, exists := foundIds[badgeType]
		if !exists {
			continue
		}

		isAlreadySelected := true
		// check if the same card is already selected
		for j := 0; j < i; j++ {
			prevElemID, prevExists := foundIds[actualBadgesOrder[j]]
			if prevExists && prevElemID == badgeElemID {
				isAlreadySelected = false
				break
			}
		}

		if isAlreadySelected {
			newPart = append(newPart, snippets[badgeElemID])
			toRemove = append(toRemove, badgeElemID)
		}
	}

	sort.Slice(toRemove, func(i, j int) bool {
		return toRemove[i] > toRemove[j]
	})
	for _, elem := range toRemove {
		snippets = removeElem(snippets, elem)
	}

	return append(newPart, snippets...)
}
