package sorting

import (
	"sort"

	aviaAPI "a.yandex-team.ru/travel/app/backend/api/avia/v1"
	aviaSearchProto "a.yandex-team.ru/travel/app/backend/internal/avia/search/proto/v1"
	"a.yandex-team.ru/travel/app/backend/internal/avia/search/searchcommon"
	"a.yandex-team.ru/travel/avia/library/go/services/featureflag"
)

type ManagerSorterConfig struct {
	InterestingConfig InterestingSorterConfig
	DefaultSort       aviaAPI.SearchSort
}

var ManagerSorterDefaultConfig = ManagerSorterConfig{
	InterestingConfig: InterestingSorterDefaultConfig,
	DefaultSort:       aviaAPI.SearchSort_SEARCH_SORT_RECOMMENDED_FIRST,
}

type ManagerSorter struct {
	cfg               *ManagerSorterConfig
	interestingSorter *InterestingSorter
}

func NewManagerSorter(cfg *ManagerSorterConfig, featureFlagStorage featureflag.StorageInterface) *ManagerSorter {
	return &ManagerSorter{
		cfg:               cfg,
		interestingSorter: NewInterestingSorter(&cfg.InterestingConfig, featureFlagStorage),
	}
}

func (s *ManagerSorter) Sort(
	snippets []*aviaAPI.Snippet,
	flightReference map[string]*aviaSearchProto.Flight,
	sortBy aviaAPI.SearchSort,
) []*aviaAPI.Snippet {
	if sortBy == aviaAPI.SearchSort_SEARCH_SORT_UNKNOWN {
		sortBy = s.cfg.DefaultSort
	}

	if sortBy == aviaAPI.SearchSort_SEARCH_SORT_RECOMMENDED_FIRST {
		snippets = s.interestingSorter.Sort(snippets, flightReference, sortBy)
		return snippets
	}

	var sortFunc func(i, j int) bool
	switch sortBy {
	case aviaAPI.SearchSort_SEARCH_SORT_CHEAPEST_FIRST:
		sortFunc = func(i, j int) bool {
			return compareSnippetByPrice(snippets[i], snippets[j], flightReference)
		}
	case aviaAPI.SearchSort_SEARCH_SORT_EXPENSIVE_FIRST:
		sortFunc = func(i, j int) bool {
			return compareSnippetsByPriceDesc(snippets[i], snippets[j], flightReference)
		}
	case aviaAPI.SearchSort_SEARCH_SORT_BY_DEPARTURE:
		sortFunc = func(i, j int) bool {
			return compareSnippetsByDeparture(flightReference, snippets[i], snippets[j])
		}
	case aviaAPI.SearchSort_SEARCH_SORT_BY_ARRIVAL:
		sortFunc = func(i, j int) bool {
			return compareSnippetsByArrival(flightReference, snippets[i], snippets[j])
		}
	}
	sort.SliceStable(snippets, sortFunc)

	return snippets
}

func compareSnippetByPrice(left, right *aviaAPI.Snippet, flights map[string]*aviaSearchProto.Flight) bool {
	return searchcommon.CompareExtendedPriceAsc(left.Variant.Price, right.Variant.Price, left.Forward, right.Forward, left.Backward, right.Backward, flights)
}

func compareSnippetsByPriceDesc(left, right *aviaAPI.Snippet, flights map[string]*aviaSearchProto.Flight) bool {
	return searchcommon.CompareExtendedPriceDesc(left.Variant.Price, right.Variant.Price, left.Forward, right.Forward, left.Backward, right.Backward, flights)
}

func compareSnippetsByDeparture(flights map[string]*aviaSearchProto.Flight, left, right *aviaAPI.Snippet) bool {
	return flights[left.Forward[0]].Departure.Seconds < flights[right.Forward[0]].Departure.Seconds
}

func compareSnippetsByArrival(flights map[string]*aviaSearchProto.Flight, left, right *aviaAPI.Snippet) bool {
	return flights[left.Forward[len(left.Forward)-1]].Arrival.Seconds < flights[right.Forward[len(right.Forward)-1]].Arrival.Seconds
}
