package handler

import (
	"context"

	"google.golang.org/grpc"

	"a.yandex-team.ru/library/go/core/xerrors"
	calendarAPI "a.yandex-team.ru/travel/app/backend/api/calendar/v1"
	"a.yandex-team.ru/travel/app/backend/internal/calendar"
	"a.yandex-team.ru/travel/app/backend/internal/common"
)

type GRPCCalendarHandler struct {
	Service *calendar.Service
}

func NewGRPCCalendarHandler(service *calendar.Service) *GRPCCalendarHandler {
	return &GRPCCalendarHandler{
		Service: service,
	}
}

func (h *GRPCCalendarHandler) GetDays(ctx context.Context, req *calendarAPI.DaysReq) (*calendarAPI.DaysRsp, error) {
	days, err := h.Service.GetDays(req.DateFrom, req.DateTo)
	if err != nil {
		return nil, xerrors.Errorf("GRPCCalendarHandler.GetDays error: %w", err)
	}
	result := calendarAPI.DaysRsp{
		Days: make(map[string]calendarAPI.DayType, 0),
	}

	for day, dayType := range days {
		dt := calendarAPI.DayType_DAY_TYPE_UNKNOWN

		switch dayType {
		case "holiday":
			dt = calendarAPI.DayType_DAY_TYPE_HOLIDAY
		case "weekday":
			dt = calendarAPI.DayType_DAY_TYPE_WEEKDAY
		case "weekend":
			dt = calendarAPI.DayType_DAY_TYPE_WEEKEND
		}
		result.Days[common.FormatDate(day.Time)] = dt
	}

	return &result, nil
}

func (h *GRPCCalendarHandler) GetServiceRegisterer() func(*grpc.Server) {
	return func(server *grpc.Server) {
		calendarAPI.RegisterCalendarAPIServer(server, h)
	}
}
