package calendar

import (
	"fmt"
	"sync"
	"sync/atomic"
	"time"

	"golang.org/x/net/context"
	"google.golang.org/genproto/googleapis/type/date"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/app/backend/internal/common"
	"a.yandex-team.ru/travel/app/backend/internal/lib/calendarclient"
	"a.yandex-team.ru/travel/library/go/geobase/consts"
)

const servicePath = "calendar.Service"

type Service struct {
	logger     log.Logger
	config     Config
	client     *calendarclient.HTTPClient
	cacheValue atomic.Value
	Mutex      sync.Mutex
}

func NewService(
	logger log.Logger,
	config Config,
	calendarClient *calendarclient.HTTPClient,
) *Service {
	return &Service{
		logger: logger.WithName(servicePath),
		config: config,
		client: calendarClient,
	}
}

func (s *Service) GetDays(from *date.Date, to *date.Date) (map[calendarclient.Date]string, error) {
	load := s.cacheValue.Load()
	if load == nil {
		return nil, xerrors.Errorf("cache empty")
	}
	cache := load.(map[calendarclient.Date]string)

	if from == nil || to == nil {
		return cache, nil
	}
	dateFrom := calendarclient.Date{Time: time.Date(int(from.Year), time.Month(from.Month), int(from.Day), 0, 0, 0, 0, time.UTC)}
	dateTo := calendarclient.Date{Time: time.Date(int(to.Year), time.Month(to.Month), int(to.Day), 0, 0, 0, 0, time.UTC)}

	result := make(map[calendarclient.Date]string, 0)
	temp := dateFrom
	for temp.Before(dateTo.Time) || temp == dateTo {
		d := calendarclient.Date{Time: temp.Time}
		if t, ok := cache[d]; ok {
			result[d] = t
		}

		temp.Time = temp.Time.Add(consts.Day)
	}

	return result, nil
}

func (s *Service) RunCaching(ctx context.Context) {
	for {
		s.logger.Info("start calendar caching")
		cached := s.cachingImpl(ctx)
		s.logger.Infof("finish calendar caching with result = %v", cached)

		nextRunDelay := s.config.CachingInterval
		if !cached {
			nextRunDelay = s.config.RetryInterval
		}

		select {
		case <-ctx.Done():
			return
		case <-time.After(nextRunDelay):
		}
	}
}

func (s *Service) cachingImpl(ctx context.Context) bool {
	defer func() {
		if r := recover(); r != nil {
			s.logger.Error(fmt.Sprintf("an error is caught: %+v", r))
		}
	}()

	data := s.fetch()
	if data == nil {
		return false
	}
	cache := make(map[calendarclient.Date]string, len(data.Holidays))
	for _, holiday := range data.Holidays {
		t, _ := common.ParseDate(holiday.Date)
		cache[calendarclient.Date{Time: t}] = holiday.Type
	}

	s.cacheValue.Store(cache)
	return true
}

func (s *Service) fetch() *calendarclient.HolidaysRsp {
	ctx, cancel := context.WithTimeout(context.Background(), s.config.RequestTimeout)
	defer cancel()

	now := time.Now()
	dateFrom := calendarclient.Date{Time: time.Date(now.Year(), now.Month()-1, 1, 0, 0, 0, 0, time.UTC)}
	dateTo := calendarclient.Date{Time: time.Date(now.Year()+1, now.Month()+1, 0, 0, 0, 0, 0, time.UTC)}
	res, err := s.client.GetHolidays(ctx, dateFrom, dateTo)

	if err != nil {
		s.logger.Error(
			"failed to cache holidays",
			log.Error(err),
			log.Time("from", dateFrom.Time),
			log.Time("to", dateTo.Time),
		)
	}
	return res
}
