package cityimages

import (
	"context"
	"fmt"
	"strings"
	"sync"
	"sync/atomic"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	contentadmin "a.yandex-team.ru/travel/app/backend/internal/lib/contentadminclient"
)

const servicePath = "cityimages.Service"

type Service struct {
	logger             log.Logger
	Config             Config
	contentAdminClient contentadmin.Client
	cacheValue         atomic.Value
	promotedIDs        map[int]struct{}
	promotedMutex      sync.Mutex
}

func NewService(
	logger log.Logger,
	config Config,
	contentAdminClient contentadmin.Client,
) *Service {
	return (&Service{
		logger:             logger.WithName(servicePath),
		Config:             config,
		contentAdminClient: contentAdminClient,
		promotedIDs:        make(map[int]struct{}),
	}).init()
}

func (s *Service) init() *Service {
	cache := make(map[int]string, len(initImages))
	for key, value := range initImages {
		cache[key] = value
	}
	s.cacheValue.Store(cache)
	return s
}

func (s *Service) GetURL(cityID int) string {
	s.promoteID(cityID)
	url := s.cacheValue.Load().(map[int]string)[cityID]
	if url == "" {
		s.logger.Warn(
			"cached city image not found",
			log.Int("city_id", cityID),
		)
	}
	return url
}

func (s *Service) RunCaching(ctx context.Context) {
	s.logger.Info("start images caching")
	s.cachingImpl(ctx)
	s.logger.Info("finish images caching")

	select {
	case <-ctx.Done():
		return
	case <-time.After(s.Config.CachingInterval):
	}

	go s.RunCaching(ctx)
}

func (s *Service) cachingImpl(ctx context.Context) {
	defer func() {
		if r := recover(); r != nil {
			s.logger.Error(fmt.Sprintf("an error is caught: %+v", r))
		}
	}()

	cache := make(map[int]string, 0)
	for id, url := range s.cacheValue.Load().(map[int]string) {
		cache[id] = url
	}

	promotedIDs := s.promotedIDs
	s.cleanPromoted()

	for id := range cache {
		promotedIDs[id] = struct{}{}
	}

	for id := range promotedIDs {
		url := s.fetchURL(id)
		if url != "" {
			cache[id] = url
		}
		select {
		case <-ctx.Done():
			return
		case <-time.After(s.Config.FetchingInterval):
		}
	}
	s.cacheValue.Store(cache)
}

func (s *Service) fetchURL(cityID int) string {
	ctx, cancel := context.WithTimeout(context.Background(), s.Config.RequestTimeout)
	defer cancel()

	res, err := s.contentAdminClient.GetCityImageByID(ctx, cityID)

	if err != nil {
		if strings.Contains(err.Error(), "NotFound") {
			s.logger.Warn(
				"city image not found",
				log.Int("city_id", cityID),
			)
		} else {
			s.logger.Error(
				"failed to cache city image",
				log.Error(err),
				log.Int("city_id", cityID),
			)
		}
	}
	return res.GetImageUrl()
}

func (s *Service) promoteID(id int) {
	s.promotedMutex.Lock()
	defer s.promotedMutex.Unlock()
	s.promotedIDs[id] = struct{}{}
}

func (s *Service) cleanPromoted() {
	s.promotedMutex.Lock()
	defer s.promotedMutex.Unlock()
	s.promotedIDs = make(map[int]struct{})
}
