package common

import (
	"context"
	"strings"

	"google.golang.org/grpc/metadata"

	"a.yandex-team.ru/travel/library/go/grpcgateway"
)

var SessionIDHTTPHeader = grpcgateway.HeaderSpec{
	Name:        "X-Travel-Session-Id",
	Description: "Идентификатор сессии",
}
var SessionIDGrpcHeader = strings.ToLower(SessionIDHTTPHeader.Name)

var UserAgentHTTPHeader = grpcgateway.HeaderSpec{
	Name:        "User-Agent",
	Description: "Информация о приложении-клиенте",
	Default:     "ru.yandex.travel.app/0.1 (Apple iPhone10,3; iOS 14.7.1)",
}
var UserAgentGrpcHeader = "grpcgateway-user-agent"

var AcceptLanguageHTTPHeader = grpcgateway.HeaderSpec{
	Name:        "Accept-Language",
	Description: "Ожидаемый язык ответа",
	Default:     "ru-RU",
	Enum:        []string{"ru-RU", "en-RU", "ru-UA"},
}
var acceptLanguageGrpcHeader = "grpcgateway-accept-language"

var RealIPHeader = grpcgateway.HeaderSpec{
	Name:        "X-Real-Ip",
	Description: "IP адрес пользователя",
}
var realIPGrpcHeader = strings.ToLower(RealIPHeader.Name)

var DeviceIDHTTPHeader = grpcgateway.HeaderSpec{
	Name:        "X-Travel-Device-Id",
	Description: "ID девайса",
}
var DeviceIDGrpcHeader = strings.ToLower(DeviceIDHTTPHeader.Name)

var UUIDHTTPHeader = grpcgateway.HeaderSpec{
	Name:        "X-Travel-Uuid",
	Description: "uuid установки приложения",
	Default:     "b59507fc-974a-41df-87cb-44bb0d2bd6eb",
}
var UUIDGrpcHeader = strings.ToLower(UUIDHTTPHeader.Name)

var RequestIDHTTPHeader = grpcgateway.HeaderSpec{
	Name:        "X-Request-Id",
	Description: "ID запроса",
	Default:     "b59507fc-974a-41df-87cb-44bb0d2b0000",
}
var requestIDGrpcHeader = strings.ToLower(RequestIDHTTPHeader.Name)

var ABExperimentsHeader = grpcgateway.HeaderSpec{
	Name:        "X-Travel-Ab-Experiments",
	Description: "AB эксперименты",
	Default:     "{\"TRAVELAPP_AVIA_FILTERS_DISABLE_ONLY\": \"0\"}",
}
var ABExperimentsGrpcHeader = strings.ToLower(ABExperimentsHeader.Name)

func GetSessionID(ctx context.Context) string {
	if md, ok := metadata.FromIncomingContext(ctx); ok {
		if sID, ok := md[SessionIDGrpcHeader]; ok {
			res := strings.Join(sID, ",")
			return res
		}
	}
	return ""
}

func GetRawUserAgent(ctx context.Context) string {
	if md, ok := metadata.FromIncomingContext(ctx); ok {
		if sID, ok := md[UserAgentGrpcHeader]; ok {
			res := strings.Join(sID, ",")
			return res
		}
	}
	return ""
}

func GetUserAgent(ctx context.Context) UserAgentData {
	rawUserAgent := GetRawUserAgent(ctx)
	if rawUserAgent != "" {
		return ParseUserAgent(rawUserAgent)
	}
	return UserAgentData{}
}

func GetRealIP(ctx context.Context) *string {
	if md, ok := metadata.FromIncomingContext(ctx); ok {
		if h, ok := md[realIPGrpcHeader]; ok {
			realIP := h[0]
			return &realIP
		}
	}
	return nil
}

func GetDeviceID(ctx context.Context) string {
	if md, ok := metadata.FromIncomingContext(ctx); ok {
		if h, ok := md[DeviceIDGrpcHeader]; ok {
			deviceID := h[0]
			return deviceID
		}
	}
	return ""
}

func GetUUID(ctx context.Context) string {
	if md, ok := metadata.FromIncomingContext(ctx); ok {
		if h, ok := md[UUIDGrpcHeader]; ok {
			uuid := h[0]
			return uuid
		}
	}
	return ""
}

func GetRequestID(ctx context.Context) string {
	if md, ok := metadata.FromIncomingContext(ctx); ok {
		if h, ok := md[requestIDGrpcHeader]; ok {
			uuid := h[0]
			return uuid
		}
	}
	return ""
}

func NewTracingHeader(name string) grpcgateway.HeaderSpec {
	return grpcgateway.HeaderSpec{
		Name:        name,
		Description: "Информация о трассировке",
	}
}

type HeaderProvider func(ctx context.Context) (map[string]string, error)
