package common

import (
	"context"
	"regexp"

	"google.golang.org/grpc"
	"google.golang.org/grpc/metadata"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/xerrors"
)

var contentLanguageRegExp = regexp.MustCompile(`^\s*(?P<lang>[a-z]{2})-(?P<countryCodeAlpha2>[A-Z]{2})\s*$`)

const (
	defaultAcceptLanguageHeaderValue = "ru-RU"
)

var fallbackAcceptedLang = map[string]string{
	acceptLanguageGrpcHeader: defaultAcceptLanguageHeaderValue,
}

var defaultMetadata = metadata.New(fallbackAcceptedLang)

type Locale struct {
	Language          string
	CountryCodeAlpha2 string
}

func getAcceptLanguage(ctx context.Context) *string {
	if md, ok := metadata.FromIncomingContext(ctx); ok {
		if als, ok := md[acceptLanguageGrpcHeader]; ok {
			al := als[0]
			return &al
		}
	}
	return nil
}

func parseAcceptLanguage(al *string) (*Locale, error) {
	if al == nil {
		return nil, xerrors.Errorf("no Accept-Language header")
	}
	matched := contentLanguageRegExp.FindStringSubmatch(*al)
	if len(matched) != 3 {
		return nil, xerrors.Errorf("invalid Accept-Language header %s", *al)
	}
	return &Locale{
		Language:          matched[1],
		CountryCodeAlpha2: matched[2],
	}, nil
}

func GetLocale(ctx context.Context) Locale {
	al := getAcceptLanguage(ctx)
	locale, err := parseAcceptLanguage(al)
	if err != nil {
		panic("TRAVELAPP-310 We validate Accept-Language header, and replace invalid with ru-RU, so this situation is impossible")
	}
	return *locale
}

func GetLanguage(ctx context.Context) string {
	return GetLocale(ctx).Language
}

func GetCountryCode(ctx context.Context) string {
	return GetLocale(ctx).CountryCodeAlpha2
}

func NewGrpcL10NInterceptor(logger log.Logger, supportedLanguages []string, supportedCountries []string) grpc.UnaryServerInterceptor {
	supportedLanguagesMap := make(map[string]struct{}, len(supportedLanguages))
	for _, val := range supportedLanguages {
		supportedLanguagesMap[val] = struct{}{}
	}
	supportedCountriesMap := make(map[string]struct{}, len(supportedCountries))
	for _, val := range supportedCountries {
		supportedCountriesMap[val] = struct{}{}
	}
	return func(
		ctx context.Context,
		req interface{},
		info *grpc.UnaryServerInfo,
		handler grpc.UnaryHandler,
	) (interface{}, error) {
		err := validateGrpcAcceptLanguageHeader(ctx, supportedLanguagesMap, supportedCountriesMap)
		if err != nil {
			logger.Debug("reset Accept Language header to default", log.Error(err))
			ctx = setDefaultGrpcAcceptLanguageHeader(ctx)
		}

		resp, err := handler(ctx, req)

		return resp, err
	}
}

func validateGrpcAcceptLanguageHeader(ctx context.Context, supportedLanguages, supportedCountries map[string]struct{}) error {
	md, ok := metadata.FromIncomingContext(ctx)
	if !ok {
		return xerrors.Errorf("empty metadata")
	}

	acceptedLanguages, ok := md[acceptLanguageGrpcHeader]
	if !ok {
		return xerrors.Errorf("missing Accept-Language header")
	}

	al := acceptedLanguages[0]
	locale, err := parseAcceptLanguage(&al)
	if err != nil {
		return err
	}

	_, okLanguage := supportedLanguages[locale.Language]
	_, okCountry := supportedCountries[locale.CountryCodeAlpha2]
	if okLanguage && okCountry {
		return nil
	}

	return xerrors.Errorf("unsupported Accept-Language header value %s", al)
}

func setDefaultGrpcAcceptLanguageHeader(ctx context.Context) context.Context {
	md, ok := metadata.FromIncomingContext(ctx)
	if !ok {
		metadata.NewIncomingContext(ctx, defaultMetadata)
	}

	newMd := md.Copy()
	newMd.Set(acceptLanguageGrpcHeader, "ru-RU")
	return metadata.NewIncomingContext(ctx, newMd)
}
