package common

import (
	"context"
	"strconv"
	"strings"

	grpc_auth "github.com/grpc-ecosystem/go-grpc-middleware/auth"
	"github.com/grpc-ecosystem/go-grpc-middleware/util/metautils"
	"google.golang.org/grpc/codes"
	"google.golang.org/grpc/status"

	"a.yandex-team.ru/library/go/yandex/blackbox"
	"a.yandex-team.ru/library/go/yandex/blackbox/httpbb"
)

const xForwardedFor = "X-Forwarded-For"
const bearerSchema = "Bearer"

const (
	userTicketHeader = "X-Ya-User-Ticket"
	userIPHeader     = "X-Ya-User-Ip"
	passportIDHeader = "X-Ya-PassportId"
	loginHeader      = "X-Ya-Login"
)

type contextMarker string

const AuthMarker contextMarker = "bb_auth"

type AuthInfo struct {
	IP            string
	Authenticated bool
	User          *blackbox.User
	OAuthToken    string
	UserTicket    string
}

func (ai *AuthInfo) IsPlus() bool {
	user := ai.User
	if user == nil {
		return false
	}
	if user.Attributes[blackbox.UserAttributeAccountHavePlus] == "1" {
		return true
	}
	return false
}

func (ai *AuthInfo) IsStaff() bool {
	user := ai.User
	if user == nil {
		return false
	}
	_, isStaff := user.Aliases[blackbox.UserAliasYandexoid]
	return isStaff
}

type AuthConfig struct {
	AllowedClientIDs []string
	BlackBoxEnv      *httpbb.Environment
}

func BuildBlackBoxAuthFunction(cfg AuthConfig, client blackbox.Client) func(ctx context.Context) (context.Context, error) {
	allowedClients := make(map[string]bool)
	for _, id := range cfg.AllowedClientIDs {
		allowedClients[id] = true
	}

	return func(ctx context.Context) (context.Context, error) {
		ips := metautils.ExtractIncoming(ctx).Get(xForwardedFor)
		ip := strings.Split(ips, ",")[0]

		token, err := grpc_auth.AuthFromMD(ctx, bearerSchema)
		if err != nil {
			a := AuthInfo{
				Authenticated: false,
				User:          nil,
				IP:            ip,
			}
			return context.WithValue(ctx, AuthMarker, a), nil
		}
		request := blackbox.OAuthRequest{
			OAuthToken:    token,
			UserIP:        ip,
			GetPublicName: true,
			RegName:       true,
			GetUserTicket: true,
			Emails:        blackbox.EmailsGetDefault,
			Attributes: []blackbox.UserAttribute{
				blackbox.UserAttributeAccountHavePlus,
			},
			Aliases: []blackbox.UserAlias{
				blackbox.UserAliasYandexoid,
			},
		}
		res, err := client.OAuth(ctx, request)
		if err != nil {
			return ctx, status.Error(codes.Unauthenticated, err.Error())
		}
		if _, exists := allowedClients[res.ClientID]; !exists {
			return ctx, status.Error(codes.Unauthenticated, "client_id is not allowed")
		}
		a := AuthInfo{
			Authenticated: true,
			User:          &res.User,
			OAuthToken:    token,
			IP:            ip,
			UserTicket:    res.UserTicket,
		}
		return context.WithValue(ctx, AuthMarker, a), nil
	}
}

func GetUser(ctx context.Context) *blackbox.User {
	v := ctx.Value(AuthMarker)
	if v == nil {
		return nil
	} else {
		return v.(AuthInfo).User
	}
}

func GetAuth(ctx context.Context) *AuthInfo {
	v := ctx.Value(AuthMarker)
	if v == nil {
		return nil
	} else {
		auth := v.(AuthInfo)
		return &auth
	}
}

func RequireAuth(ctx context.Context) (*AuthInfo, error) {
	v := ctx.Value(AuthMarker)
	if v == nil {
		return nil, status.Error(codes.Unauthenticated, "authentication expected")
	} else {
		auth := v.(AuthInfo)
		if !auth.Authenticated || auth.User == nil {
			return nil, status.Error(codes.Unauthenticated, "authentication required")
		}
		if auth.UserTicket == "" {
			return nil, status.Error(codes.Unauthenticated, "no user ticket available")
		}
		return &auth, nil
	}
}

func GetAuthHeaders(ctx context.Context) (map[string]string, error) {
	res := make(map[string]string, 4)
	auth := GetAuth(ctx)
	if auth != nil {
		res[userIPHeader] = auth.IP
		if auth.Authenticated && auth.User != nil {
			res[userTicketHeader] = auth.UserTicket
			res[passportIDHeader] = strconv.FormatUint(auth.User.ID, 10)
			res[loginHeader] = auth.User.Login
		}
	}
	return res, nil
}
