package common

import (
	"regexp"
	"strconv"
	"strings"
)

var uaRegExp = regexp.MustCompile(`(?P<name>\w[\w.]*)/(?P<version>\d+(\.\d+)*)\s*(\((?P<version_code>\d*)\))?\s*\((?P<manufacturer>[^\s]+)\s+(?P<device>[^;]+);\s+(?P<os_name>[^\s]+)\s+(?P<os_version>[^)]+)\)`)

type UserAgentData struct {
	RawUA              string
	IsStandardFormat   bool
	KnownOS            OSType
	AppName            string
	AppVersion         string
	AppVersionCode     uint64
	DeviceManufacturer string
	DeviceName         string
	OSName             string
	OSVersion          string
}

func ParseUserAgent(userAgentString string) UserAgentData {
	res := UserAgentData{RawUA: userAgentString}
	matched := uaRegExp.FindStringSubmatch(userAgentString)
	if len(matched) == 10 {
		res.IsStandardFormat = true
		res.AppName = matched[1]
		res.AppVersion = matched[2]
		res.AppVersionCode, _ = strconv.ParseUint(matched[5], 10, 64)
		res.DeviceManufacturer = matched[6]
		res.DeviceName = matched[7]
		res.OSName = matched[8]
		res.OSVersion = matched[9]

		if osType, known := osTypeByName[strings.ToLower(res.OSName)]; known {
			res.KnownOS = osType
		}
	}
	return res
}
