package handler

import (
	"context"
	"fmt"

	"google.golang.org/grpc"
	"google.golang.org/grpc/metadata"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/ctxlog"
	"a.yandex-team.ru/library/go/core/xerrors"
	geobaseLib "a.yandex-team.ru/library/go/yandex/geobase"
	commonAPI "a.yandex-team.ru/travel/app/backend/api/common/v1"
	geolocationAPI "a.yandex-team.ru/travel/app/backend/api/geolocation/v1"
	"a.yandex-team.ru/travel/app/backend/internal/aviacommon"
	"a.yandex-team.ru/travel/app/backend/internal/common"
	"a.yandex-team.ru/travel/app/backend/internal/lib/aviabackendclient"
	"a.yandex-team.ru/travel/app/backend/internal/references"
	"a.yandex-team.ru/travel/library/go/geobase"
	"a.yandex-team.ru/travel/library/go/geobase/consts"
	"a.yandex-team.ru/travel/proto/dicts/rasp"
)

type GRPCGeolocationHandler struct {
	logger            log.Logger
	geoBase           geobase.Geobase
	registry          *references.Registry
	aviaBackendClient *aviabackendclient.HTTPClient
}

func NewGRPCGeolocationHandler(
	logger log.Logger,
	geoBase geobase.Geobase,
	registry *references.Registry,
	aviaBackendClient *aviabackendclient.HTTPClient,
) *GRPCGeolocationHandler {
	return &GRPCGeolocationHandler{
		logger:            logger,
		geoBase:           geoBase,
		registry:          registry,
		aviaBackendClient: aviaBackendClient,
	}
}

func (h *GRPCGeolocationHandler) GetAviaLocation(ctx context.Context, req *geolocationAPI.GetAviaLocationReq) (*geolocationAPI.GetAviaLocationRsp, error) {
	headerNameToLogName := map[string]string{
		"grpcgateway-user-agent": "userAgent",
		"x-real-ip":              "ipHeader",
		"x-request-id":           "requestId",
		"x-travel-session-id":    "travelSessionId",
		"x-travel-device-id":     "travelDeviceId",
		"x-travel-uuid":          "travelUuid",
	}
	ctxOptions := make([]log.Field, 0, len(headerNameToLogName))
	if md, ok := metadata.FromIncomingContext(ctx); ok {
		for headerName, logName := range headerNameToLogName {
			if headerValue, exists := md[headerName]; exists && len(headerValue) != 0 {
				logName = fmt.Sprintf("headers.%s", logName)
				ctxOptions = append(ctxOptions, log.String(logName, headerValue[0]))
			}
		}
	}
	ctx = ctxlog.WithFields(ctx, ctxOptions...)

	locale := common.GetLocale(ctx)
	originalGeoID := h.getGeobaseGeoID(ctx, req.Latitude, req.Longitude, common.GetRealIP(ctx))

	// Пробуем получить город из geolookup
	aviaCityFromGeolookup, err := h.getAviaCity(ctx, locale, originalGeoID)
	var aviaGeoID int32 = consts.MoscowGeoID
	if err != nil {
		ctxlog.Error(ctx, h.logger, "ошибка geolookup:", log.Error(err))
	} else {
		aviaGeoID = aviaCityFromGeolookup.GeoID
	}

	// Пробуем найти авиа город и оригинальный город в общих справочниках
	var originalCityGeoID int32
	aviaCity, found := h.registry.Settlements.GetByGeoID(int(aviaGeoID))
	if found {
		_, foundOriginal := h.registry.Settlements.GetByGeoID(int(originalGeoID))
		if foundOriginal {
			originalCityGeoID = originalGeoID
		} else {
			originalCityGeoID = aviaGeoID
		}
	} else {
		originalCityGeoID = consts.MoscowGeoID
		aviaCityFromGeolookup = nil
		aviaCity, found = h.registry.Settlements.GetByGeoID(consts.MoscowGeoID)
		if !found {
			return nil, xerrors.Errorf("не нашли Москву в общих справочниках")
		}
	}

	// Заполняем ответ
	aviaPoint := h.buildPointFromSettlement(aviaCity, locale)

	// Из geolookup приходят не только коды города, но и коды аэропорта, если у города нет соответствующего кода,
	// а у аэропорта есть. Поэтому используем коды из geolookup
	if aviaCityFromGeolookup != nil {
		aviaPoint.AviaCode = aviaCityFromGeolookup.Code
		if aviaCityFromGeolookup.IATACode != "" {
			aviaPoint.Iata = aviaCityFromGeolookup.IATACode
		}
		if aviaCityFromGeolookup.SirenaCode != "" {
			aviaPoint.Sirena = aviaCityFromGeolookup.SirenaCode
		}
	}

	return &geolocationAPI.GetAviaLocationRsp{
		OriginalCityGeoId: originalCityGeoID,
		AviaPoint:         aviaPoint,
	}, nil
}

func (h *GRPCGeolocationHandler) GetAviaLocationPost(ctx context.Context, req *geolocationAPI.GetAviaLocationReq) (*geolocationAPI.GetAviaLocationRsp, error) {
	return h.GetAviaLocation(ctx, req)
}

func (h *GRPCGeolocationHandler) getGeobaseGeoID(ctx context.Context, latitude, longitude float64, ip *string) int32 {
	region, err := h.geoBase.GetRegionByLocation(latitude, longitude)
	if err != nil {
		ctxlog.Error(ctx, h.logger, "ошибка получения geoID по координатам из геобазы", log.Float64("lat", latitude), log.Float64("lon", longitude), log.Error(err))
	} else {
		ctxlog.Debug(ctx, h.logger, "получили region по координатам из геобазы", log.Float64("lat", latitude), log.Float64("lon", longitude), log.Any("region", region))
		cityID := int32(region.CityID)
		if cityID > 0 {
			return cityID
		}
		geoID := int32(region.ID)
		if geoID > 0 {
			return geoID
		}
	}

	if ip != nil {
		region, err := h.geoBase.GetRegionByIP(*ip)
		if err != nil {
			ctxlog.Error(ctx, h.logger, "ошибка получения geoID по ip из геобазы", log.Error(err))
		} else {
			ctxlog.Debug(ctx, h.logger, "получили region по ip из геобазы", log.Any("region", region))
			cityID := int32(region.CityID)
			if cityID > 0 {
				return cityID
			}
			geoID := int32(region.ID)
			if geoID > 0 {
				return geoID
			}
		}
	}

	ctxlog.Warn(ctx, h.logger, "не удалось получить geoID, берем значение по-умолчанию")
	return consts.MoscowGeoID
}

func (h *GRPCGeolocationHandler) getAviaCity(ctx context.Context, locale common.Locale, originalGeoID int32) (*aviabackendclient.SearchCityResult, error) {
	nationalVersion := aviacommon.GetNationalVersionByCountryCode(locale.CountryCodeAlpha2)
	rsp, err := h.aviaBackendClient.GeoLookup(ctx, nationalVersion, locale.Language, originalGeoID)
	if err != nil {
		return nil, xerrors.Errorf("ошибка geolookup %w", err)
	} else if rsp.Status != aviabackendclient.SuccessResponseStatus {
		return nil, xerrors.Errorf("ошибка geolookup: status %s", string(rsp.Status))
	} else if len(rsp.Data) < 1 {
		return nil, xerrors.Errorf("ошибка geolookup: пустой ответ")
	} else {
		return &rsp.Data[0].SearchCity, nil
	}
}

func (h *GRPCGeolocationHandler) buildPointFromSettlement(city *rasp.TSettlement, locale common.Locale) *geolocationAPI.GeoPoint {
	title := getCityTitle(city, locale.Language)

	var regionTitle = ""
	region, foundRegion := h.registry.Regions.Get(int(city.RegionId))
	if foundRegion {
		regionTitle = getRegionTitle(region, locale.Language)
	}

	var countryTitle = ""
	countryGeoID, err := h.geoBase.GetCountryID(int(city.GeoId), getCrimeaStatus(locale.CountryCodeAlpha2))
	if err == nil {
		country, foundCountry := h.registry.Countries.GetByGeoID(countryGeoID)
		if foundCountry {
			countryTitle = getCountryTitle(country, locale.Language)
		}
	} else {
		country, foundCountry := h.registry.Countries.Get(int(city.CountryId))
		if foundCountry {
			countryTitle = getCountryTitle(country, locale.Language)
		}
	}

	return &geolocationAPI.GeoPoint{
		GeoId:        city.GeoId,
		Id:           uint64(city.Id),
		Type:         commonAPI.GeoPointType_GEO_POINT_TYPE_SETTLEMENT,
		Title:        title,
		PointKey:     fmt.Sprintf("c%d", city.Id),
		Slug:         city.Slug,
		Iata:         city.Iata,
		Icao:         "",
		Sirena:       city.SirenaId,
		RegionTitle:  regionTitle,
		CountryTitle: countryTitle,
	}
}

func getCrimeaStatus(countryCode string) geobaseLib.CrimeaStatus {
	if countryCode == "" || countryCode == "RU" {
		return geobaseLib.CrimeaInRU
	}
	return geobaseLib.CrimeaInUA
}

func (h *GRPCGeolocationHandler) GetServiceRegisterer() func(*grpc.Server) {
	return func(server *grpc.Server) {
		geolocationAPI.RegisterGeolocationAPIServer(server, h)
	}
}

func getCityTitle(city *rasp.TSettlement, lang string) string {
	var title string
	switch lang {
	case "ru":
		title = city.Title.Ru.Nominative
	case "uk":
		title = city.Title.Uk.Nominative
	case "tr":
		title = city.Title.Tr.Nominative
	case "en":
		title = city.Title.En.Nominative
	default:
		title = city.Title.Ru.Nominative
	}
	return title
}

func getRegionTitle(region *rasp.TRegion, lang string) string {
	var title string
	switch lang {
	case "ru":
		title = region.TitleNominative.Ru
	case "uk":
		title = region.TitleNominative.Uk
	case "tr":
		title = region.TitleNominative.Tr
	case "en":
		title = region.TitleNominative.En
	default:
		title = region.TitleNominative.Ru
	}
	return title
}

func getCountryTitle(country *rasp.TCountry, lang string) string {
	var title string
	switch lang {
	case "ru":
		title = country.Title.Ru.Nominative
	case "uk":
		title = country.Title.Uk.Nominative
	case "tr":
		title = country.Title.Tr.Nominative
	case "en":
		title = country.Title.En.Nominative
	default:
		title = country.Title.Ru.Nominative
	}
	return title
}
