package hotels

import (
	"fmt"
	"strconv"
	"strings"

	"github.com/golang/protobuf/ptypes/wrappers"
	"google.golang.org/protobuf/types/known/wrapperspb"

	"a.yandex-team.ru/library/go/core/xerrors"
	v1 "a.yandex-team.ru/travel/app/backend/api/common/v1"
	hotelsAPI "a.yandex-team.ru/travel/app/backend/api/hotels/v1"
	"a.yandex-team.ru/travel/app/backend/internal/common"
	"a.yandex-team.ru/travel/app/backend/internal/lib/travelapiclient/models"
)

const badgeYandexPlusID = "yandex_plus"

func ConvertAnalyticsData(analytics *hotelsAPI.AnalyticsData) *models.AnalyticsParams {
	if analytics == nil {
		return nil
	}
	result := models.AnalyticsParams{}
	if analytics.GetUtmContent() != nil {
		result.UtmContent = &analytics.GetUtmContent().Value
	}
	if analytics.GetUtmContent() != nil {
		result.UtmCampaign = &analytics.GetUtmContent().Value
	}
	if analytics.GetUtmTerm() != nil {
		result.UtmTerm = &analytics.GetUtmTerm().Value
	}
	if analytics.GetUtmMedium() != nil {
		result.UtmMedium = &analytics.GetUtmMedium().Value
	}
	if analytics.GetUtmSource() != nil {
		result.UtmSource = &analytics.GetUtmSource().Value
	}
	return &result
}

//Permalink и Slug вместе передавать нельзя
func convertHotelQueryData(hotelID *hotelsAPI.HotelID) *models.GetHotelInfoQueryData {
	res := models.GetHotelInfoQueryData{}
	if hotelID.GetPermalink() > 0 {
		res.Permalink = hotelID.GetPermalink()
	} else {
		res.HotelSlug = hotelID.GetHotelSlug()
	}
	return &res
}

//Permalink и Slug вместе передавать нельзя
func convertHotelQueryData2(data *hotelsAPI.HotelSuggestData) *models.GetHotelInfoQueryData {
	res := models.GetHotelInfoQueryData{}
	if data.Permalink > 0 {
		res.Permalink = data.Permalink
	} else {
		res.HotelSlug = data.Slug
	}
	return &res
}

func ConvertSearchParams(params *hotelsAPI.OfferSearchParams) *models.SearchParams {
	if params == nil {
		return nil
	}
	p := params
	searchParams := models.SearchParams{}
	if p.CheckinDate != nil {
		searchParams.CheckinDate = fmt.Sprintf("%4d-%02d-%02d", p.CheckinDate.Year, p.CheckinDate.Month, p.CheckinDate.Day)
	}
	if p.CheckoutDate != nil {
		searchParams.CheckoutDate = fmt.Sprintf("%4d-%02d-%02d", p.CheckoutDate.Year, p.CheckoutDate.Month, p.CheckoutDate.Day)
	}
	if p.Adults > 0 {
		searchParams.Adults = int(p.Adults)
	}
	if len(p.ChildrenAges) > 0 {
		searchParams.ChildrenAges = make([]int, len(p.ChildrenAges))
		for k, v := range p.ChildrenAges {
			searchParams.ChildrenAges[k] = int(v)
		}
	}
	return &searchParams
}

func ConvertFromSearchParams(params *models.SearchParams) *hotelsAPI.OfferSearchParams {
	if params == nil {
		return nil
	}
	p := params
	searchParams := hotelsAPI.OfferSearchParams{}
	searchParams.CheckinDate, _ = common.DateStringToProto(p.CheckinDate)
	searchParams.CheckoutDate, _ = common.DateStringToProto(p.CheckoutDate)
	if p.Adults > 0 {
		searchParams.Adults = uint32(p.Adults)
	}
	if len(p.ChildrenAges) > 0 {
		searchParams.ChildrenAges = make([]uint32, len(p.ChildrenAges))
		for k, v := range p.ChildrenAges {
			searchParams.ChildrenAges[k] = uint32(v)
		}
	}
	return &searchParams
}

func convertImages(images []models.Image, imageSizeStr *string) []*hotelsAPI.Image {
	res := make([]*hotelsAPI.Image, len(images))
	for i, image := range images {
		sizes := make([]*hotelsAPI.Image_Size, len(image.Sizes))
		var sizeOrig *hotelsAPI.Image_Size
		for j, size := range image.Sizes {
			sizes[j] = &hotelsAPI.Image_Size{Identifier: size.Identifier, Height: size.Height, Width: size.Width}
			if size.Identifier == ImageSizeOrig {
				sizeOrig = sizes[j]
			}
		}
		res[i] = &hotelsAPI.Image{
			Sizes:       sizes,
			SizeOrig:    sizeOrig,
			UrlTemplate: image.URLTemplate,
			Url:         buildImageURL(&image.URLTemplate, imageSizeStr),
		}
	}
	return res
}

func convertFeatureIcon(id string) hotelsAPI.FeatureIcon {
	switch id {
	case "gym":
		return hotelsAPI.FeatureIcon_FEATURE_ICON_GYM
	case "wi_fi":
		return hotelsAPI.FeatureIcon_FEATURE_ICON_WI_FI
	case "car_park":
		return hotelsAPI.FeatureIcon_FEATURE_ICON_CAR_PARK
	case "air_conditioning":
		return hotelsAPI.FeatureIcon_FEATURE_ICON_AIR_CONDITIONING
	case "payment_by_credit_card":
		return hotelsAPI.FeatureIcon_FEATURE_ICON_PAYMENT_BY_CREDIT_CARD
	case "pool":
		return hotelsAPI.FeatureIcon_FEATURE_ICON_POOL
	case "beach_line":
		return hotelsAPI.FeatureIcon_FEATURE_ICON_BEACH_LINE
	default:
		return hotelsAPI.FeatureIcon_FEATURE_ICON_UNKNOWN
	}
}

func ConvertHotel(h *models.Hotel, imageSizeStr *string) *hotelsAPI.Hotel {
	result := hotelsAPI.Hotel{}
	permalink, err := ConvertPermalink(h.Permalink)
	if err == nil {
		result.Permalink = int64(permalink)
	}

	result.Name = h.Name
	result.Address = wrapperspb.String(h.Address)
	result.Rating = wrapperspb.Double(h.Rating)
	var stars *wrapperspb.UInt32Value
	if h.Stars != nil {
		stars = wrapperspb.UInt32(uint32(*h.Stars))
	}
	result.Stars = stars
	result.Coordinates = &hotelsAPI.Coordinates{
		Latitude:  h.Coordinates.Latitude,
		Longitude: h.Coordinates.Longitude,
	}

	result.Images = convertImages(h.Images, imageSizeStr)

	result.TotalImageCount = &wrappers.UInt32Value{Value: h.TotalImageCount}
	result.TotalTextReviewCount = &wrappers.Int32Value{Value: h.TotalTextReviewCount}

	result.GeoInfo = &hotelsAPI.Hotel_GeoInfo{
		GeoFeatures: []*hotelsAPI.Hotel_GeoFeature{},
	}
	if h.DistanceText != "" {
		result.GeoInfo.UserDistanceText = &wrappers.StringValue{Value: h.DistanceText}
	}
	if h.GeoInfo != nil {
		result.GeoInfo.GeoFeatures = append(result.GeoInfo.GeoFeatures, &hotelsAPI.Hotel_GeoFeature{
			Name: h.GeoInfo.Name,
			Icon: hotelsAPI.Hotel_GEO_FEATURE_ICON_MAIN_DISTANCE,
		})
	}
	if len(h.NearestStations) > 0 {
		// TODO(ganintsev): узнать какие станции надо показывать на самом деле
		st := h.NearestStations[0]
		result.GeoInfo.GeoFeatures = append(result.GeoInfo.GeoFeatures, &hotelsAPI.Hotel_GeoFeature{
			Name:         st.Name,
			DistanceText: st.DistanceText,
			Icon:         hotelsAPI.Hotel_GEO_FEATURE_ICON_TRANSPORT_DISTANCE,
		})
	}

	features := make([]*hotelsAPI.Feature, len(h.Features))
	for i, f := range h.Features {
		features[i] = &hotelsAPI.Feature{
			Name: &wrappers.StringValue{Value: f.Name},
			Icon: convertFeatureIcon(f.ID),
		}
	}
	result.Features = features

	result.IsYandexHotel = &wrappers.BoolValue{Value: h.IsYandexHotel}
	result.IsFavorite = &wrappers.BoolValue{Value: h.IsFavorite}
	result.HotelUrl = h.HotelURL
	return &result
}

func ConvertHotelWithOffers(h *models.HotelWithOffers, imageSizeStr *string, getTranslation func(key TranslationKey) string) *hotelsAPI.Hotel {
	result := ConvertHotel(&h.Hotel, imageSizeStr)
	if len(h.OffersInfo) > 0 {
		result.OfferInfo = convertOffer(h.OffersInfo[0], getTranslation)
	}

	result.PollingFinished = &wrappers.BoolValue{Value: h.PollingFinished}
	result.IsPlusAvailable = isPlusAvailable(h.Badges)

	return result
}

func isPlusAvailable(badges []models.HotelBadge) *wrapperspb.BoolValue {
	for _, badge := range badges {
		if badge.ID == "yandex_plus" {
			return wrapperspb.Bool(true)
		}
	}
	return wrapperspb.Bool(false)
}

func convertOffers(offers []models.OfferInfo, getTranslation func(key TranslationKey) string) []*hotelsAPI.OfferInfo {
	result := make([]*hotelsAPI.OfferInfo, 0, len(offers))
	for _, o := range offers {
		result = append(result, convertOffer(o, getTranslation))
	}
	return result
}

func convertOffer(offer models.OfferInfo, getTranslation func(key TranslationKey) string) *hotelsAPI.OfferInfo {
	o := hotelsAPI.OfferInfo{
		Name:       offer.Name,
		Price:      &wrappers.UInt64Value{Value: uint64(offer.Price.Value)},
		RoomId:     offer.RoomID,
		OperatorId: offer.OperatorID,
		Badges:     convertHotelBadges(offer.Badges),
		RedirectInfo: &hotelsAPI.RedirectInfo{
			XRedirect: offer.LandingURL,
			IsBoy:     offer.YandexOffer,
		},
	}
	if offer.YandexPlusInfo != nil {
		o.YandexPlusInfo = &hotelsAPI.YandexPlusInfo{
			Eligible: offer.YandexPlusInfo.Eligible,
			Points:   offer.YandexPlusInfo.Points,
		}
	}
	o.PansionInfo = convertPansionInfo(offer.PansionInfo)
	o.CancellationInfo = convertCancellationInfo(offer.CancellationInfo, getTranslation)
	if offer.DiscountInfo != nil && offer.DiscountInfo.ShowDiscountInfo {
		o.StrikeThroughPrice = &wrapperspb.UInt64Value{Value: uint64(offer.DiscountInfo.StrikethroughPrice.Value)}
	}
	return &o
}

func convertHotelBadges(badges []models.HotelBadge) []*hotelsAPI.HotelBadge {
	res := make([]*hotelsAPI.HotelBadge, 0, len(badges))
	for _, b := range badges {
		t := hotelsAPI.HotelBadgeType_HOTEL_BADGE_TYPE_UNKNOWN
		if b.ID == badgeYandexPlusID {
			t = hotelsAPI.HotelBadgeType_HOTEL_BADGE_TYPE_YANDEX_PLUS
		}
		res = append(res, &hotelsAPI.HotelBadge{
			Type: t,
			Text: &wrapperspb.StringValue{Value: b.Text},
		})
	}
	return res
}

func convertHotelDescription(info *models.HotelDescription) string {
	if info == nil {
		return ""
	}
	return info.Text
}

func convertRatingInfo(info *models.RatingsInfo) *hotelsAPI.RatingsInfo {
	if info == nil {
		return nil
	}
	featureRatings := make([]*hotelsAPI.FeatureRatingInfo, 0, len(info.FeatureRatings))
	for _, fr := range info.FeatureRatings {
		featureRatings = append(featureRatings, &hotelsAPI.FeatureRatingInfo{
			Name:            fr.Name,
			PositivePercent: uint32(fr.PositivePercent),
		})
	}
	return &hotelsAPI.RatingsInfo{
		Teaser:         info.Teaser,
		FeatureRatings: featureRatings,
	}
}

func convertToHotelOffersRsp(rsp *models.GetHotelOffersResponse, getTranslation func(key TranslationKey) string) *hotelsAPI.GetHotelOffersRsp {
	if rsp == nil {
		return nil
	}
	return &hotelsAPI.GetHotelOffersRsp{
		OffersInfo: convertOffersInfo(&rsp.OffersInfo, getTranslation),
	}
}

func convertCancellationInfo(info *models.CancellationInfo, getTranslation func(key TranslationKey) string) *hotelsAPI.CancellationInfo {
	if info == nil {
		return nil
	}
	rules := make([]*hotelsAPI.RefundRule, 0, len(info.RefundRules))
	for _, r := range info.RefundRules {
		rule := hotelsAPI.RefundRule{
			Type: convertRefundType(r.Type),
		}
		if r.Penalty != nil {
			rule.Penalty = &wrapperspb.UInt32Value{Value: uint32(r.Penalty.Value)}
		}
		if r.StartsAt != nil {
			rule.StartsAt = &wrapperspb.StringValue{Value: r.StartsAt.String()}
		}
		if r.EndsAt != nil {
			rule.EndsAt = &wrapperspb.StringValue{Value: r.EndsAt.String()}
		}
		rules = append(rules, &rule)
	}
	refundType := convertRefundType(info.RefundType)
	return &hotelsAPI.CancellationInfo{
		HasFreeCancellation: &wrapperspb.BoolValue{Value: info.HasFreeCancellation},
		RefundType:          refundType,
		RefundRules:         rules,
		Name:                getNameRefund(refundType, getTranslation),
	}
}

func convertPansionInfo(info *models.PansionInfo) *hotelsAPI.PansionInfo {
	if info == nil {
		return nil
	}
	return &hotelsAPI.PansionInfo{
		PansionType: convertPansionType(info.ID),
		PansionName: &wrappers.StringValue{Value: info.Name},
	}
}

func ConvertPermalink(v models.Permalink) (int, error) {
	switch v := v.(type) {
	case int:
		return v, nil
	case string:
		c, err := strconv.Atoi(v)
		if err != nil {
			return 0, err
		}
		return int(c), nil
	default:
		return 0, fmt.Errorf("conversion to int from %T not supported", v)
	}
}

func convertOffersInfo(info *models.OffersInfo, getTranslate func(key TranslationKey) string) *hotelsAPI.OffersInfo {
	if info == nil {
		return nil
	}
	return &hotelsAPI.OffersInfo{
		PollingFinished:           info.OfferSearchProgress.Finished,
		OfferSearchProgress:       convertOfferSearchProgress(&info.OfferSearchProgress),
		AggregatedOfferInfo:       convertAggregatedOfferInfo(&info.AggregatedOfferInfo),
		NextPollingRequestDelayMs: &wrapperspb.Int32Value{Value: int32(info.NextPollingRequestDelayMs)},
		PartnerOffers:             convertPartnerOffers(info.PartnerOffers, getTranslate),
		MainOffers:                convertOffers(info.MainOffers, getTranslate),
		OfferCount:                int32(info.OfferCount),
		Rooms:                     convertRooms(info.Rooms),
		// TODO(adurnev) унести в справочники?
		Operators:      convertOperators(info.OperatorByID),
		OperatorsCount: int32(info.OperatorCount),
	}
}

func convertOperators(operatorByID models.OperatorByID) map[string]*hotelsAPI.HotelsOperator {
	res := make(map[string]*hotelsAPI.HotelsOperator, len(operatorByID))
	for _, o := range operatorByID {
		res[o.ID] = &hotelsAPI.HotelsOperator{
			Id:      o.ID,
			Name:    o.Name,
			IconUrl: o.IconURL,
		}
	}
	return res
}

func convertRooms(rooms []models.Room) []*hotelsAPI.HotelRoom {
	res := make([]*hotelsAPI.HotelRoom, 0, len(rooms))
	for _, r := range rooms {
		res = append(res, convertRoom(&r))
	}
	return res
}

func convertRoom(room *models.Room) *hotelsAPI.HotelRoom {
	amenityGroups := make([]*hotelsAPI.RoomAmenityGroup, 0, len(room.AmenityGroups)+1)
	for _, ag := range room.AmenityGroups {
		amenityGroups = append(amenityGroups, &hotelsAPI.RoomAmenityGroup{
			Name:      ag.Name,
			Amenities: convertHotelRoomFeature(ag.Features),
			// TODO(adurnev) icon?
		})
	}
	bedGroups := make([]*hotelsAPI.RoomBedGroup, 0, len(room.BedGroups))
	for _, bg := range room.BedGroups {
		items := make([]*hotelsAPI.RoomBedItem, 0, len(bg.Configuration))
		for _, c := range bg.Configuration {
			items = append(items, &hotelsAPI.RoomBedItem{
				Quantity: uint32(c.Quantity),
				Icon:     convertHotelBedIcon(c.ID),
			})
		}
		bedGroups = append(bedGroups, &hotelsAPI.RoomBedGroup{
			Items: items,
		})
	}
	//TODO(adurenv) delete after https://st.yandex-team.ru/TRAVELAPP-1583
	fix := ""
	return &hotelsAPI.HotelRoom{
		Id:            room.ID,
		Name:          room.Name,
		Description:   room.Description,
		Images:        convertImages(room.Images, &fix),
		Area:          convertHotelArea(&room.Area),
		MainAmenities: convertHotelRoomFeature(room.MainAmenities),
		AmenityGroups: amenityGroups,
		BedGroups:     bedGroups,
	}
}

const bedsTypesIcon = "beds-types"

// TODO(adurnev) выбрать название из amenities и features
func convertHotelRoomFeature(features []models.Feature) []*hotelsAPI.RoomAmenity {
	amenities := make([]*hotelsAPI.RoomAmenity, 0, len(features))
	for _, a := range features {
		amenity := hotelsAPI.RoomAmenity{
			Name: a.Name,
		}
		if a.Icon != nil && *a.Icon == bedsTypesIcon {
			icon := convertHotelRoomAmenityBedIcon(a.ID)
			amenity.IconValue = &icon
		} else {
			icon := convertHotelRoomAmenityIcon(a.Icon)
			amenity.IconValue = &icon
		}
		amenities = append(amenities, &amenity)
	}
	return amenities
}

func convertHotelRoomAmenityIcon(icon *string) hotelsAPI.RoomAmenity_Icon {
	res := hotelsAPI.RoomAmenity_Icon{}
	if icon == nil {
		return res
	}
	switch *icon {
	case "wifi":
		res.Icon = hotelsAPI.RoomAmenityIcon_ROOM_AMENITY_WIFI
	case "air-conditioning":
		res.Icon = hotelsAPI.RoomAmenityIcon_ROOM_AMENITY_AIR_CONDITIONING
	case "flat-panel-tv":
		res.Icon = hotelsAPI.RoomAmenityIcon_ROOM_AMENITY_FLAT_PANEL_TV
	case "bathroom":
		res.Icon = hotelsAPI.RoomAmenityIcon_ROOM_AMENITY_BATHROOM
	case "television-yes":
		res.Icon = hotelsAPI.RoomAmenityIcon_ROOM_AMENITY_TELEVISION
	case "view-city-view":
		res.Icon = hotelsAPI.RoomAmenityIcon_ROOM_AMENITY_CITY_VIEW
	case "minibar":
		res.Icon = hotelsAPI.RoomAmenityIcon_ROOM_AMENITY_MINIBAR
	case "view-court-view":
		res.Icon = hotelsAPI.RoomAmenityIcon_ROOM_AMENITY_COURT_VIEW
	case "bathtub":
		res.Icon = hotelsAPI.RoomAmenityIcon_ROOM_AMENITY_BATHTUB
	case "balcony":
		res.Icon = hotelsAPI.RoomAmenityIcon_ROOM_AMENITY_BALCONY
	case "coffee-machine":
		res.Icon = hotelsAPI.RoomAmenityIcon_ROOM_AMENITY_COFFEE_MACHINE
	case "soundproofed-rooms":
		res.Icon = hotelsAPI.RoomAmenityIcon_ROOM_AMENITY_SOUNDPROOFED_ROOMS
	case "balcony-with-view":
		res.Icon = hotelsAPI.RoomAmenityIcon_ROOM_AMENITY_BALCONY_WITH_VIEW
	case "view-sea-view":
		res.Icon = hotelsAPI.RoomAmenityIcon_ROOM_AMENITY_SEA_VIEW
	case "view-mountain-view":
		res.Icon = hotelsAPI.RoomAmenityIcon_ROOM_AMENITY_MOUNTAIN_VIEW
	case "view-river-view":
		res.Icon = hotelsAPI.RoomAmenityIcon_ROOM_AMENITY_RIVER_VIEW
	case "view-park-view":
		res.Icon = hotelsAPI.RoomAmenityIcon_ROOM_AMENITY_PARK_VIEW
	case "lanai":
		res.Icon = hotelsAPI.RoomAmenityIcon_ROOM_AMENITY_LANAI
	case "view-garden-view":
		res.Icon = hotelsAPI.RoomAmenityIcon_ROOM_AMENITY_GARDEN_VIEW
	case "view-pool-view":
		res.Icon = hotelsAPI.RoomAmenityIcon_ROOM_AMENITY_POOL_VIEW
	case "view-forest-view":
		res.Icon = hotelsAPI.RoomAmenityIcon_ROOM_AMENITY_FOREST_VIEW
	case "jacuzzi-bathroom":
		res.Icon = hotelsAPI.RoomAmenityIcon_ROOM_AMENITY_JACUZZI_BATHROOM
	case "view-lake-view":
		res.Icon = hotelsAPI.RoomAmenityIcon_ROOM_AMENITY_LAKE_VIEW
	case "view-partial-seaview":
		res.Icon = hotelsAPI.RoomAmenityIcon_ROOM_AMENITY_PARTIAL_SEA_VIEW
	case "jetted-bathtub":
		res.Icon = hotelsAPI.RoomAmenityIcon_ROOM_AMENITY_JETTED_BATHTUB
	case "barbecue-facilities":
		res.Icon = hotelsAPI.RoomAmenityIcon_ROOM_AMENITY_BARBECUE_FACILITIES
	case "balcony-with-deck-chairs":
		res.Icon = hotelsAPI.RoomAmenityIcon_ROOM_AMENITY_BALCONY_WITH_DECK_CHAIRS
	case "three-tv-sets":
		res.Icon = hotelsAPI.RoomAmenityIcon_ROOM_AMENITY_THREE_TV_SETS
	case "balcony-with-side-sea-view":
		res.Icon = hotelsAPI.RoomAmenityIcon_ROOM_AMENITY_BALCONY_WITH_SIDE_SEA_VIEW
	case "private-pool":
		res.Icon = hotelsAPI.RoomAmenityIcon_ROOM_AMENITY_PRIVATE_POOL
	case "view-view-to-stadium":
		res.Icon = hotelsAPI.RoomAmenityIcon_ROOM_AMENITY_VIEW_TO_STADIUM
	case "aeromassage-bath":
		res.Icon = hotelsAPI.RoomAmenityIcon_ROOM_AMENITY_AERO_MASSAGE_BATH
	case "view-bay-view":
		res.Icon = hotelsAPI.RoomAmenityIcon_ROOM_AMENITY_BAY_VIEW
	case "view-view-race-track":
		res.Icon = hotelsAPI.RoomAmenityIcon_ROOM_AMENITY_VIEW_RACE_TRACK
	}
	return res
}

func convertHotelRoomAmenityBedIcon(id string) hotelsAPI.RoomAmenity_BedItem {
	res := hotelsAPI.RoomAmenity_BedItem{}
	str := strings.Split(id, ":")
	if len(str) != 2 {
		return res
	}
	name := str[1]
	icon := convertHotelBedIcon(name)
	if icon == hotelsAPI.RoomBedIcon_ROOM_BED_ICON_UNKNOWN {
		return res
	}

	quantity := uint32(1)
	switch {
	case strings.HasPrefix(name, "two"):
		quantity = 2
	case strings.HasPrefix(name, "three"):
		quantity = 3
	case strings.HasPrefix(name, "four"):
		quantity = 4
	case strings.HasPrefix(name, "five"):
		quantity = 5
	case strings.HasPrefix(name, "six"):
		quantity = 6
	case strings.HasPrefix(name, "seven"):
		quantity = 7
	case strings.HasPrefix(name, "eight"):
		quantity = 8
	case strings.HasPrefix(name, "nine"):
		quantity = 8
	case strings.HasPrefix(name, "ten"):
		quantity = 10
	}

	res.BedItem = &hotelsAPI.RoomBedItem{
		Icon:     icon,
		Quantity: quantity,
	}
	return res
}

// TODO(adurnev) тут много видов кроватей "Трехъярусная кровать"...
func convertHotelBedIcon(str string) hotelsAPI.RoomBedIcon {
	switch {
	case strings.Contains(str, "single_bed"):
		return hotelsAPI.RoomBedIcon_ROOM_BED_ICON_SINGLE
	case strings.Contains(str, "double_bed"):
		return hotelsAPI.RoomBedIcon_ROOM_BED_ICON_DOUBLE
	default:
		return hotelsAPI.RoomBedIcon_ROOM_BED_ICON_UNKNOWN
	}
}

func convertHotelArea(area *models.Area) *hotelsAPI.RoomArea {
	res := hotelsAPI.RoomArea{
		Value: float32(area.Value),
		Unit:  hotelsAPI.RoomAreaUnits_ROOM_AREA_UNITS_UNKNOWN,
	}
	if area.Unit == "SQUARE_METERS" {
		res.Unit = hotelsAPI.RoomAreaUnits_ROOM_AREA_UNITS_SQUARE_METERS
	}
	return &res
}

func convertPartnerOffers(partnerOffers []models.PartnerOffers, getTranslate func(key TranslationKey) string) []*hotelsAPI.HotelPartnerOffersInfo {
	result := make([]*hotelsAPI.HotelPartnerOffersInfo, 0, len(partnerOffers))
	for _, p := range partnerOffers {
		result = append(result, &hotelsAPI.HotelPartnerOffersInfo{
			PansionAggregate:             convertPansionAggregate(p.PansionAggregate),
			CancellationInfoAggregate:    convertCancellationInfoAggregate(p.CancellationInfoAggregate),
			DefaultOffer:                 convertOffer(p.DefaultOffer, getTranslate),
			DefaultOfferPansion:          convertPansionInfo(p.DefaultOfferPansion),
			DefaultOfferCancellationInfo: convertCancellationInfo(p.DefaultOfferCancellationInfo, getTranslate),
			OperatorId:                   p.OperatorID,
		})
	}
	return result
}

func convertAggregatedOfferInfo(info *models.AggregatedOfferInfo) *hotelsAPI.AggregatedOfferInfo {
	return &hotelsAPI.AggregatedOfferInfo{
		MinPrice:                  convertPrice(&info.MinPrice.Price),
		MaxPrice:                  convertPrice(&info.MaxPrice.Price),
		CancellationInfoAggregate: convertCancellationInfoAggregate(info.CancellationInfoAggregate),
		PansionAggregate:          convertPansionAggregate(info.PansionAggregate),
	}
}

func convertPansionAggregate(info *string) hotelsAPI.PansionAggregate {
	if info == nil {
		return hotelsAPI.PansionAggregate_PANSION_AGGREGATE_UNKNOWN
	}
	switch *info {
	case "NO_PANSION_AVAILABLE":
		return hotelsAPI.PansionAggregate_PANSION_AGGREGATE_NO_PANSION_AVAILABLE
	case "PANSION_AVAILABLE":
		return hotelsAPI.PansionAggregate_PANSION_AGGREGATE_PANSION_AVAILABLE
	default:
		return hotelsAPI.PansionAggregate_PANSION_AGGREGATE_UNKNOWN
	}
}

func convertCancellationInfoAggregate(info *string) hotelsAPI.CancellationInfoAggregate {
	if info == nil {
		return hotelsAPI.CancellationInfoAggregate_CANCELLATION_INFO_AGGREGATE_UNKNOWN
	}
	switch *info {
	case "FULLY_REFUNDABLE_AVAILABLE":
		return hotelsAPI.CancellationInfoAggregate_CANCELLATION_INFO_AGGREGATE_FULLY_REFUNDABLE_AVAILABLE
	case "REFUNDABLE_WITH_PENALTY_AVAILABLE":
		return hotelsAPI.CancellationInfoAggregate_CANCELLATION_INFO_AGGREGATE_REFUNDABLE_WITH_PENALTY_AVAILABLE
	case "NON_REFUNDABLE_AVAILABLE":
		return hotelsAPI.CancellationInfoAggregate_CANCELLATION_INFO_AGGREGATE_NON_REFUNDABLE_AVAILABLE
	default:
		return hotelsAPI.CancellationInfoAggregate_CANCELLATION_INFO_AGGREGATE_UNKNOWN
	}
}

func convertPrice(info *models.Price) *v1.Price {
	if info == nil {
		return nil
	}
	return &v1.Price{
		Currency: info.Currency,
		Value:    info.Value,
	}
}

func convertOfferSearchProgress(info *models.OfferSearchProgress) *hotelsAPI.OfferSearchProgress {
	finishedPartners := make([]hotelsAPI.PartnerId, 0, len(info.FinishedPartners))
	for _, p := range info.FinishedPartners {
		finishedPartners = append(finishedPartners, convertPartnerID(p))
	}
	pendingPartners := make([]hotelsAPI.PartnerId, 0, len(info.PendingPartners))
	for _, p := range info.PendingPartners {
		pendingPartners = append(pendingPartners, convertPartnerID(p))
	}
	return &hotelsAPI.OfferSearchProgress{
		Finished:         info.Finished,
		PartnersComplete: info.PartnersComplete,
		PartnersTotal:    info.PartnersTotal,
		FinishedPartners: finishedPartners,
		PendingPartners:  pendingPartners,
	}
}

func convertPartnerID(id string) hotelsAPI.PartnerId {
	switch id {
	case "PI_BOOKING":
		return hotelsAPI.PartnerId_PARTNER_ID_BOOKING
	case "PI_HOTELS101":
		return hotelsAPI.PartnerId_PARTNER_ID_HOTELS101
	case "PI_TRAVELLINE":
		return hotelsAPI.PartnerId_PARTNER_ID_TRAVELLINE
	case "PI_HOTELSCOMBINED":
		return hotelsAPI.PartnerId_PARTNER_ID_HOTELSCOMBINED
	case "PI_OSTROVOK":
		return hotelsAPI.PartnerId_PARTNER_ID_OSTROVOK
	case "PI_EXPEDIA":
		return hotelsAPI.PartnerId_PARTNER_ID_EXPEDIA
	case "PI_DOLPHIN":
		return hotelsAPI.PartnerId_PARTNER_ID_DOLPHIN
	case "PI_BNOVO":
		return hotelsAPI.PartnerId_PARTNER_ID_BNOVO
	case "PI_TVIL":
		return hotelsAPI.PartnerId_PARTNER_ID_TVIL
	case "PI_BRONEVIK":
		return hotelsAPI.PartnerId_PARTNER_ID_BRONEVIK
	default:
		return hotelsAPI.PartnerId_PARTNER_ID_UNUSED
	}
}

func convertHotelImagesRsp(response *models.GetHotelImagesResponse) (*hotelsAPI.GetHotelImagesRsp, error) {
	fix := ""
	return &hotelsAPI.GetHotelImagesRsp{
		Images: convertImages(response.Images, &fix),
	}, nil
}

func convertHotelsCountersRsp(response models.GetCountersResponse) *hotelsAPI.GetHotelCountersRsp {
	rsp := hotelsAPI.GetHotelCountersRsp{}
	rsp.TotalHotelsCount = int32(response.TotalHotels)

	detailedFilters := response.FilterInfo.DetailedFilter

	detailedFiltersRsp := hotelsAPI.Filters{}
	detailed := make([]*hotelsAPI.Filter, len(detailedFilters))
	for i, f := range detailedFilters {
		filter := convertDetailedFilter(f)
		detailed[i] = filter
	}
	detailedFiltersRsp.DetailedFilters = detailed

	filtersBatches := response.FilterInfo.DetailedFilterBatches
	batch := make([]*hotelsAPI.FilterBatch, len(filtersBatches))
	for i, f := range filtersBatches {
		res := make([]*hotelsAPI.Filter, len(f.Filters))
		for j, ff := range f.Filters {
			res[j] = convertDetailedFilter(ff)
		}
		batch[i] = &hotelsAPI.FilterBatch{
			DetailedFilter: res,
		}
	}
	detailedFiltersRsp.DetailedFiltersBatches = batch

	rsp.DetailedFilters = &detailedFiltersRsp

	return &rsp
}

func convertToGetCountersReq(req *hotelsAPI.GetHotelCountersReq) (*models.GetCountersRequest, error) {
	if req == nil || req.QueryData == nil {
		return nil, xerrors.Errorf("queryData is require")
	}

	main := req.QueryData
	request := models.GetCountersRequest{}

	request.AnalyticsParams = ConvertAnalyticsData(req.AnalyticsData)

	data := &request.QueryData
	data.Adults = int(main.Adults)
	for _, val := range main.ChildrenAges {
		data.ChildrenAge = append(data.ChildrenAge, int(val))
	}

	if len(req.QueryData.FilterAtoms) > 0 {
		data.Atoms = req.QueryData.FilterAtoms
	}
	checkinDate := main.CheckinDate
	if checkinDate == nil {
		return nil, xerrors.Errorf("checkin is require")
	}
	data.CheckinDate = fmt.Sprintf("%4d-%02d-%02d", checkinDate.Year, checkinDate.Month, checkinDate.Day)
	checkoutDate := main.CheckoutDate
	if checkoutDate == nil {
		return nil, xerrors.Errorf("checkout is require")
	}
	data.CheckoutDate = fmt.Sprintf("%4d-%02d-%02d", checkoutDate.Year, checkoutDate.Month, checkoutDate.Day)

	if req.QueryData.Bbox == nil || req.QueryData.Bbox.LeftDown == nil || req.QueryData.Bbox.UpRight == nil {
		return nil, xerrors.Errorf("bbox is require")
	}
	bbox := models.BoundingBox{
		Coordinates: []models.Coordinates{
			{
				Latitude:  req.QueryData.Bbox.LeftDown.Latitude,
				Longitude: req.QueryData.Bbox.LeftDown.Longitude,
			},
			{
				Latitude:  req.QueryData.Bbox.UpRight.Latitude,
				Longitude: req.QueryData.Bbox.UpRight.Longitude,
			},
		},
	}
	data.Bbox = &bbox
	if req.QueryData.Price != nil {
		if req.QueryData.Price.GetFrom() != nil {
			data.PriceFrom = req.QueryData.Price.GetFrom().GetValue()
		}
		if req.QueryData.Price.GetTo() != nil {
			data.PriceTo = req.QueryData.Price.GetTo().GetValue()
		}
	}
	return &request, nil
}

func convertToHotelInfoRequest(req *hotelsAPI.GetHotelInfoReq, config HotelInfo) (*models.GetHotelInfoRequest, error) {
	converted := models.GetHotelInfoRequest{
		ImageParams: convertImageParams(int(config.ImageLimit), 0),
		ReviewPagingParams: &models.PagingParams{
			Limit:  int(config.ReviewLimit),
			Offset: 0,
		},
		ReviewPhrase: &models.PhraseReq{
			Limit: int(config.PhraseLimit),
		},
		SimilarLimit:    int(config.SimilarLimit),
		ReviewSort:      ConvertSort(hotelsAPI.ReviewSort_REVIEW_SORT_RELEVANT_FIRST),
		QueryData:       convertHotelQueryData2(req.QueryData),
		AnalyticsParams: ConvertAnalyticsData(req.AnalyticsData),
	}

	if req.Params == nil {
		return &converted, nil
	}
	converted.Params = ConvertSearchParams(req.Params)

	return &converted, nil
}

func convertImageParams(limit, offset int) *models.HotelImageParams {
	return &models.HotelImageParams{
		OnlyTop: true,
		PagingParams: &models.PagingParams{
			Limit:  limit,
			Offset: offset,
		},
		Sizes: []string{ImageSizeOrig},
	}
}

func convertToHotelOffersRequest(req *hotelsAPI.GetHotelOffersReq) (*models.GetHotelOffersRequest, error) {
	converted := models.GetHotelOffersRequest{}
	if req == nil {
		return &converted, nil
	}
	converted.AnalyticsParams = ConvertAnalyticsData(req.AnalyticsData)
	converted.QueryData = convertHotelQueryData(req.HotelId)
	converted.Params = ConvertSearchParams(req.Params)
	if req.SearchPagePollingId != nil {
		converted.SearchPagePollingID = req.SearchPagePollingId.Value
	}
	return &converted, nil
}

func convertToHotelImagesRequest(req *hotelsAPI.GetHotelImagesReq, cfg Config) (*models.GetHotelImagesRequest, error) {
	converted := models.GetHotelImagesRequest{}
	if req == nil {
		return &converted, nil
	}
	converted.QueryData = convertHotelQueryData(req.HotelId)
	limit := int(cfg.ImagesLimit)
	offset := 0
	if req.PagingParams != nil {
		limit = int(req.PagingParams.Limit)
		offset = int(req.PagingParams.Offset)
	}
	converted.ImageParams = convertImageParams(limit, offset)
	if req.ParentRequestId != nil {
		converted.ParentRequestID = req.ParentRequestId.GetValue()
	}

	return &converted, nil
}

func convertToSimilarHotelsRequest(req *hotelsAPI.GetSimilarHotelsReq) (*models.GetSimilarHotelsRequest, error) {
	converted := models.GetSimilarHotelsRequest{}
	if req == nil {
		return &converted, nil
	}
	converted.SimilarHotelLimit = req.Limit
	converted.QueryData = convertHotelQueryData(req.HotelId)
	converted.AnalyticsParams = ConvertAnalyticsData(req.AnalyticsData)
	converted.Params = ConvertSearchParams(req.Params)

	if req.ParentRequestId != nil {
		converted.ParentRequestID = req.ParentRequestId.GetValue()
	}
	if req.SearchPagePollingId != nil {
		converted.SearchPagePollingID = req.SearchPagePollingId.Value
	}
	return &converted, nil
}

func convertSorter(sorter hotelsAPI.Sort) string {
	switch sorter {
	case hotelsAPI.Sort_SORT_HIGH_RATING_FIRST:
		return "high-rating-first"
	case hotelsAPI.Sort_SORT_PRICE_ASC:
		return "cheap-first"
	case hotelsAPI.Sort_SORT_PRICE_DESC:
		return "expensive-first"
	case hotelsAPI.Sort_SORT_RELEVANT_FIRST:
		return "relevant-first"

	}
	return "relevant-first"
}

func convert(reason *hotelsAPI.StartSearchReason) models.HotelSearchStartReason {
	r := *reason
	switch r {
	case hotelsAPI.StartSearchReason_START_SEARCH_REASON_QUERY_BY_LOCATION:
		return models.StartSearchReasonQueryByLocation
	case hotelsAPI.StartSearchReason_START_SEARCH_REASON_MAP_BOUNDS:
		return models.StartSearchReasonMapBounds
	case hotelsAPI.StartSearchReason_START_SEARCH_REASON_NAVIGATION_TOKEN:
		return models.StartSearchReasonNavigationToken
	case hotelsAPI.StartSearchReason_START_SEARCH_REASON_FILTERS_TYPE_SEARCH_TEXT:
		return models.StartSearchReasonFiltersTypeSearchText
	case hotelsAPI.StartSearchReason_START_SEARCH_REASON_SORT:
		return models.StartSearchReasonSort
	case hotelsAPI.StartSearchReason_START_SEARCH_REASON_FILTERS:
		return models.StartSearchReasonFilters
	case hotelsAPI.StartSearchReason_START_SEARCH_REASON_MOUNT:
		return models.StartSearchReasonMount
	}
	return models.StartSearchReasonUnknown
}

func convertDetailedFilter(f models.FilterAggregation) *hotelsAPI.Filter {
	return &hotelsAPI.Filter{
		Type:           f.Type,
		DetailedFilter: convDetailedFilters(f.DetailedFilter),
	}
}

func convDetailedFilters(f *models.Filter) *hotelsAPI.FilterInfo {
	if f == nil {
		return nil
	}
	return &hotelsAPI.FilterInfo{
		Id:    &wrappers.StringValue{Value: f.ID},
		Name:  &wrappers.StringValue{Value: f.Name},
		Type:  &wrappers.StringValue{Value: f.Type},
		Items: convertFilterItems(f.Items),
	}
}

func convertFilterItems(items []models.DetailedFilter) []*hotelsAPI.DetailedFilter {
	result := make([]*hotelsAPI.DetailedFilter, len(items))
	for i, f := range items {
		result[i] = &hotelsAPI.DetailedFilter{
			Id:   f.ID,
			Name: f.Name,
			Hint: f.Hint,
			//Effect: f.Effect,
			Enabled: filterEnabled(f.Enabled, f.Hint),
			Atoms:   f.Atoms,
		}
	}
	return result
}

func convertRefundType(refund string) hotelsAPI.RefundType {
	switch refund {
	case "FULLY_REFUNDABLE":
		return hotelsAPI.RefundType_REFUND_TYPE_FULLY_REFUNDABLE
	case "REFUNDABLE_WITH_PENALTY":
		return hotelsAPI.RefundType_REFUND_TYPE_REFUNDABLE_WITH_PENALTY
	case "NON_REFUNDABLE":
		return hotelsAPI.RefundType_REFUND_TYPE_NON_REFUNDABLE
	}
	return hotelsAPI.RefundType_REFUND_TYPE_UNKNOWN
}

func getNameRefund(t hotelsAPI.RefundType, getTranslation func(key TranslationKey) string) string {
	switch t {
	case hotelsAPI.RefundType_REFUND_TYPE_FULLY_REFUNDABLE:
		return getTranslation(RefundFullyRefundableNameKey)
	case hotelsAPI.RefundType_REFUND_TYPE_REFUNDABLE_WITH_PENALTY:
		return getTranslation(RefundRefundableWithPenaltyNameKey)
	case hotelsAPI.RefundType_REFUND_TYPE_NON_REFUNDABLE:
		return getTranslation(RefundNonRefundableNameKey)
	}
	return getTranslation(RefundUnknownNameKey)
}

func convertPansionType(pt string) hotelsAPI.PansionType {
	switch pt {
	case "AI", "PT_AI":
		return hotelsAPI.PansionType_PANSION_TYPE_ALL_INCLUSIVE
	case "BB", "PT_BB":
		return hotelsAPI.PansionType_PANSION_TYPE_BED_AND_BREAKFAST
	case "FB", "PT_FB":
		return hotelsAPI.PansionType_PANSION_TYPE_FULL_BOARD
	case "HB", "PT_HB":
		return hotelsAPI.PansionType_PANSION_TYPE_HALF_BOARD
	case "RO", "PT_RO":
		return hotelsAPI.PansionType_PANSION_TYPE_ROOM_ONLY
	case "UAI", "PT_UAI":
		return hotelsAPI.PansionType_PANSION_TYPE_ULTRA_ALL_INCLUSIVE
	case "LAI", "PT_LAI":
		return hotelsAPI.PansionType_PANSION_TYPE_LIGHT_ALL_INCLUSIVE
	case "DO", "PT_DO":
		return hotelsAPI.PansionType_PANSION_TYPE_DINNER_ONLY
	}
	return hotelsAPI.PansionType_PANSION_TYPE_UNKNOWN
}

func buildImageURL(imageURLTemplate, imageSize *string) string {
	return strings.TrimSuffix(*imageURLTemplate, ImageSuffix) + *imageSize
}

func ConvertSort(sort hotelsAPI.ReviewSort) string {
	switch sort {
	case hotelsAPI.ReviewSort_REVIEW_SORT_RELEVANT_FIRST:
		return "byRelevanceOrg"
	case hotelsAPI.ReviewSort_REVIEW_SORT_TIME_DESC:
		return "byTime"
	case hotelsAPI.ReviewSort_REVIEW_SORT_RATING_ASC:
		return "byRatingAsc"
	case hotelsAPI.ReviewSort_REVIEW_SORT_RATING_DESC:
		return "byRatingDesc"
	case hotelsAPI.ReviewSort_REVIEW_SORT_LIKES_COUNT_DESC:
		return "byLikesCountDesc"
	default:
		return "byRelevanceOrg"
	}
}

func ConvertReviewsRsp(info *models.ReviewsInfo, getTranslation func(key TranslationKey) string, config *ReviewsConfig) *hotelsAPI.ReviewsRsp {
	if info == nil {
		return nil
	}
	var reviews []*hotelsAPI.ReviewRsp
	if info.TextReviews != nil {
		reviews = make([]*hotelsAPI.ReviewRsp, len(info.TextReviews))
		for i := 0; i < len(info.TextReviews); i++ {
			reviews[i] = ConvertReview(&info.TextReviews[i], config)
		}
	}

	var phraseAvailable []*hotelsAPI.Keyphrase
	if info.KeyPhrases != nil {
		phraseAvailable = make([]*hotelsAPI.Keyphrase, len(info.KeyPhrases))
		for i := 0; i < len(info.KeyPhrases); i++ {
			p := info.KeyPhrases[i]
			phraseAvailable[i] = &hotelsAPI.Keyphrase{
				Name:        p.Name,
				ReviewCount: uint32(p.ReviewCount),
			}
		}
	}
	phrases := hotelsAPI.ReviewQuickFilterPhrase{
		Available:  phraseAvailable,
		TotalCount: uint32(info.TotalKeyPhraseCount),
	}

	userReview := ConvertReview(info.UserTextReview, config)
	sort := hotelsAPI.ReviewQuickFilterSort{
		Name:     getTranslation(HotelReviewSortNameKey),
		Selected: hotelsAPI.ReviewSort_REVIEW_SORT_RELEVANT_FIRST,
		Available: []*hotelsAPI.ReviewQuickSorter{
			{Id: hotelsAPI.ReviewSort_REVIEW_SORT_RELEVANT_FIRST, Title: getTranslation(SortRelevantFirstTitleKey), ShortTitle: getTranslation(SortRelevantFirstShortTitleKey)},
			{Id: hotelsAPI.ReviewSort_REVIEW_SORT_TIME_DESC, Title: getTranslation(SortTimeDescTitleKey), ShortTitle: getTranslation(SortTimeDescShortTitleKey)},
			{Id: hotelsAPI.ReviewSort_REVIEW_SORT_RATING_ASC, Title: getTranslation(SortRatingAscTitleKey), ShortTitle: getTranslation(SortRatingAscShortTitleKey)},
			{Id: hotelsAPI.ReviewSort_REVIEW_SORT_RATING_DESC, Title: getTranslation(SortRatingDescTitleKey), ShortTitle: getTranslation(SortRatingDescShortTitleKey)},
			//{Id: v1.ReviewSort_REVIEW_SORT_LIKES_COUNT_DESC, Title: "", ShortTitle: ""},//TODO(adurnev) на портале такой нет, нужно ли нам?
		},
	}
	return &hotelsAPI.ReviewsRsp{
		HasMore:          info.HasMore,
		Sort:             &sort,
		TotalReviewCount: uint32(info.TotalTextReviewCount),
		Reviews:          reviews,
		Phrases:          &phrases,
		UserReview:       userReview,
	}
}

func ConvertReview(review *models.TextReview, config *ReviewsConfig) *hotelsAPI.ReviewRsp {
	if review == nil {
		return nil
	}
	var phraseMatch *hotelsAPI.Keyphrasematch
	if review.KeyPhraseMatch != nil {
		fragments := make([]*hotelsAPI.Keyphrasematch_Fragments, len(review.KeyPhraseMatch.Fragments))
		for i := 0; i < len(review.KeyPhraseMatch.Fragments); i++ {
			f := review.KeyPhraseMatch.Fragments[i]
			fragments[i] = &hotelsAPI.Keyphrasematch_Fragments{
				Size:     uint32(f.Size),
				Position: uint32(f.Position),
			}
		}
		phraseMatch = &hotelsAPI.Keyphrasematch{
			Fragments: fragments,
		}
	}
	var images []*hotelsAPI.ImageReview
	if review.Images != nil {
		images = make([]*hotelsAPI.ImageReview, len(review.Images))
		for i := 0; i < len(review.Images); i++ {
			image := review.Images[i]
			var moderation *hotelsAPI.Moderation
			if image.Moderation != nil {
				moderation = &hotelsAPI.Moderation{
					Status: buildModerationStatus(image.Moderation.Status),
				}
			}
			images[i] = &hotelsAPI.ImageReview{
				Id: image.ID,
				Image: &hotelsAPI.Image{
					UrlTemplate: fixImageURLTemplate(image.URLTemplate, config.AvatarSuffix),
					//TODO(adurnev) orig нет в ответе отеелей, вобщ енет sizes
				},
				Moderation: moderation,
				Tags:       image.Tags,
			}
		}
	}
	var moderation *hotelsAPI.Moderation
	if review.Moderation != nil {
		moderation = &hotelsAPI.Moderation{
			Status: buildModerationStatus(review.Moderation.Status),
		}
	}
	return &hotelsAPI.ReviewRsp{
		Author: &hotelsAPI.Author{
			Name:       review.Author.Name,
			Level:      review.Author.Level,
			AvatarUrl:  buildAvatarURL(review.Author.AvatarURLTemplate, config),
			ProfileUrl: review.Author.ProfileURL,
		},
		Id:                review.ID,
		UpdatedAt:         review.UpdatedAt,
		Rating:            uint32(review.Rating),
		Text:              review.Text,
		Snippet:           review.Snippet,
		Images:            images,
		UserReaction:      buildUserReaction(review.UserReaction),
		TotalDislikeCount: uint32(review.TotalDislikeCount),
		TotalLikeCount:    uint32(review.TotalLikeCount),
		BusinessComment:   review.BusinessComment,
		CommentCount:      uint32(review.CommentCount),
		Moderation:        moderation,
		PhraseMatch:       phraseMatch,
	}
}

func fixImageURLTemplate(imageURLTemplate string, suffix string) string {
	if len(imageURLTemplate) == 0 {
		return ""
	}
	return strings.TrimSuffix(imageURLTemplate, suffix) + ImageSuffix
}

func buildAvatarURL(avatarURLTemplate string, config *ReviewsConfig) string {
	if len(avatarURLTemplate) == 0 {
		return config.AvatarDefaultURL
	}
	return strings.TrimSuffix(avatarURLTemplate, config.AvatarSuffix) + config.AvatarSizeDefault
}

func buildUserReaction(reaction string) hotelsAPI.ReviewReactionType {
	switch reaction {
	case "NONE":
		return hotelsAPI.ReviewReactionType_REVIEW_REACTION_NONE
	case "LIKE":
		return hotelsAPI.ReviewReactionType_REVIEW_REACTION_LIKE
	case "DISLIKE":
		return hotelsAPI.ReviewReactionType_REVIEW_REACTION_DISLIKE
	default:
		return hotelsAPI.ReviewReactionType_REVIEW_REACTION_UNKNOWN
	}
}

func buildModerationStatus(status string) hotelsAPI.ModerationStatus {
	switch status {
	case "IN_PROGRESS":
		return hotelsAPI.ModerationStatus_MODERATION_STATUS_IN_PROGRESS
	case "ACCEPTED":
		return hotelsAPI.ModerationStatus_MODERATION_STATUS_ACCEPTED
	case "DECLINED":
		return hotelsAPI.ModerationStatus_MODERATION_STATUS_DECLINED
	default:
		return hotelsAPI.ModerationStatus_MODERATION_STATUS_UNKNOWN
	}
}
