package handler

import (
	"context"
	"strconv"
	"strings"

	"google.golang.org/grpc"

	"a.yandex-team.ru/library/go/core/log"
	hotelFavoriteAPI "a.yandex-team.ru/travel/app/backend/api/hotelfavorites/v1"
	v1 "a.yandex-team.ru/travel/app/backend/api/hotels/v1"
	"a.yandex-team.ru/travel/app/backend/internal/cityimages"
	hotels "a.yandex-team.ru/travel/app/backend/internal/hotels"
	"a.yandex-team.ru/travel/app/backend/internal/lib/clientscommon"
	"a.yandex-team.ru/travel/app/backend/internal/lib/travelapiclient/models"
)

const serviceName = "hotel favorites"

type GRPCHandler struct {
	config          *hotels.FavoritesConfig
	logger          log.Logger
	travelAPIClient TravelAPIClient
	cityImages      *cityimages.Service
	hotelService    *hotels.Service
}

func NewGRPCHotelFavoritesHandler(config *hotels.FavoritesConfig, logger log.Logger, travelAPIClient TravelAPIClient, cityImages *cityimages.Service, hotelService *hotels.Service) *GRPCHandler {
	return &GRPCHandler{
		config:          config,
		logger:          logger.WithName("hotels.favorites.GRPCHandler"),
		travelAPIClient: travelAPIClient,
		cityImages:      cityImages,
		hotelService:    hotelService,
	}
}

const (
	origImageSuffix = "/orig"
	newImageSuffix  = "/travel-avia-desktop" // https://mds.yandex-team.ru/avatars/namespaces/rasp/versions/9
)

func (h *GRPCHandler) GetCategories(ctx context.Context, req *v1.GetFavoritesCategoriesReq) (*v1.GetFavoritesCategoriesRsp, error) {
	request := models.GetHotelFavoritesReq{
		AnalyticsParams: hotels.ConvertAnalyticsData(req.AnalyticsData),
		PagingParams: &models.PagingParams{
			Limit:  0,
			Offset: 0,
		},
	}

	favorite, err := h.travelAPIClient.GetHotelFavorites(ctx, &request)
	if err != nil {
		return nil, clientscommon.ConvertHTTPErrorToGRPCError(ctx, h.logger, err, serviceName)
	}

	response := v1.GetFavoritesCategoriesRsp{}

	if len(favorite.Categories) > 0 {
		response.Categories = make([]*v1.Categories, 0)

		for _, cat := range favorite.Categories {
			if cat.ID == "all" {
				continue
			}
			catID, _ := strconv.Atoi(cat.ID)
			imageURL := h.cityImages.GetURL(catID)
			if imageURL == "" {
				imageURL = h.cityImages.Config.DefaultCityImage
			} else {
				imageURL = strings.TrimSuffix(imageURL, origImageSuffix) + newImageSuffix
			}
			response.Categories = append(response.Categories, &v1.Categories{
				Name:       cat.Name,
				Id:         cat.ID,
				HotelCount: uint32(cat.HotelCount),
				ImageUrl:   imageURL,
			})
		}
	}

	return &response, nil
}
func (h *GRPCHandler) Get(ctx context.Context, req *v1.GetFavoritesReq) (*v1.GetFavoritesRsp, error) {
	request := models.GetHotelFavoritesReq{
		SearchParams:    hotels.ConvertSearchParams(req.SearchParams),
		AnalyticsParams: hotels.ConvertAnalyticsData(req.AnalyticsData),
		CategoryID:      req.CategoryId,
	}

	limit := h.config.Limit
	if req.Limit > 0 {
		limit = req.Limit
	}
	request.PagingParams = hotels.ConvertPagingData(&v1.PagingParams{
		Limit:  limit,
		Offset: req.Offset,
	})

	favorite, err := h.travelAPIClient.GetHotelFavorites(ctx, &request)
	if err != nil {
		return nil, clientscommon.ConvertHTTPErrorToGRPCError(ctx, h.logger, err, serviceName)
	}

	response := v1.GetFavoritesRsp{}

	if favorite.SelectedCategoryID != "" {
		response.SelectedCategoryId = favorite.SelectedCategoryID
	}
	if favorite.TotalHotelCount > 0 {
		response.TotalHotelCount = uint32(favorite.TotalHotelCount)
	}
	if len(favorite.Hotels) > 0 {
		response.Hotels, err = h.hotelService.ConvertHotels(ctx, favorite.Hotels, req.ImageParams)
		if err != nil {
			return nil, err
		}
	}
	response.SearchParams = hotels.ConvertFromSearchParams(favorite.OfferSearchParams)

	return &response, nil
}
func (h *GRPCHandler) Add(context context.Context, req *v1.AddFavoritesReq) (*v1.AddFavoritesRsp, error) {
	request := models.AddHotelFavoritesReq{}

	request.AnalyticsParams = hotels.ConvertAnalyticsData(req.AnalyticsData)
	request.Permalink = &req.Permalink

	err := h.travelAPIClient.AddHotelFavorites(context, &request)
	if err != nil {
		return nil, clientscommon.ConvertHTTPErrorToGRPCError(context, h.logger, err, serviceName)
	}

	return &v1.AddFavoritesRsp{}, nil
}
func (h *GRPCHandler) Remove(context context.Context, req *v1.RemoveFavoritesReq) (*v1.RemoveFavoritesRsp, error) {
	request := models.RemoveHotelFavoritesReq{}

	request.AnalyticsParams = hotels.ConvertAnalyticsData(req.AnalyticsData)
	if req.GetPermalink() > 0 {
		p := req.GetPermalink()
		request.Permalink = &p
	} else if req.GetCategoryId() != "" {
		c := req.GetCategoryId()
		request.CategoryID = &c
	}

	err := h.travelAPIClient.RemoveHotelFavorites(context, &request)
	if err != nil {
		return nil, clientscommon.ConvertHTTPErrorToGRPCError(context, h.logger, err, serviceName)
	}

	return &v1.RemoveFavoritesRsp{}, nil
}

func (h *GRPCHandler) GetServiceRegisterer() func(*grpc.Server) {
	return func(server *grpc.Server) {
		hotelFavoriteAPI.RegisterHotelFavoritesAPIServer(server, h)
	}
}

type TravelAPIClient interface {
	GetHotelFavorites(ctx context.Context, req *models.GetHotelFavoritesReq) (*models.GetHotelFavoritesRsp, error)
	AddHotelFavorites(ctx context.Context, req *models.AddHotelFavoritesReq) error
	RemoveHotelFavorites(ctx context.Context, req *models.RemoveHotelFavoritesReq) error
}
