package handler

import (
	"context"
	"errors"
	"fmt"
	"net/http"
	"strconv"
	"strings"

	"github.com/golang/protobuf/ptypes/wrappers"
	"golang.org/x/exp/slices"
	"google.golang.org/grpc"
	"google.golang.org/grpc/codes"
	"google.golang.org/grpc/status"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/ctxlog"
	"a.yandex-team.ru/library/go/core/xerrors"
	commonAPI "a.yandex-team.ru/travel/app/backend/api/commonorder/v1"
	hotelOrderAPI "a.yandex-team.ru/travel/app/backend/api/hotelorder/v1"
	"a.yandex-team.ru/travel/app/backend/internal/common"
	"a.yandex-team.ru/travel/app/backend/internal/hotels"
	"a.yandex-team.ru/travel/app/backend/internal/l10n"
	"a.yandex-team.ru/travel/app/backend/internal/lib/clientscommon"
	"a.yandex-team.ru/travel/app/backend/internal/lib/travelapiclient/models"
)

const (
	serviceName      = "hotel order"
	purchaseTokenKey = "purchase_token"
	l10nKeyset       = "backend_hotel_order"
)

var waitingForStatuses = []int{http.StatusRequestTimeout, http.StatusInternalServerError, http.StatusBadGateway, http.StatusServiceUnavailable, http.StatusGatewayTimeout}

type translationKey string

const (
	defaultWaitingTitleKey        translationKey = "workflow_default_waiting_title"
	defaultWaitingMessageKey      translationKey = "workflow_default_waiting_message"
	reservedWaitingTitleKey       translationKey = "workflow_reserved_waiting_title"
	reservedWaitingMessageKey     translationKey = "workflow_reserved_waiting_message"
	promocodeErrorAlertTitleKey   translationKey = "workflow_promocode_error_alert_title"
	promocodeErrorAlertMessageKey translationKey = "workflow_promocode_error_alert_message"
	paymentErrorAlertTitleKey     translationKey = "workflow_payment_error_alert_title"
	paymentErrorAlertMessageKey   translationKey = "workflow_payment_error_alert_message"
	buttonExitTitleKey            translationKey = "workflow_button_exit_title"
	buttonPayTitleKey             translationKey = "workflow_button_pay_title"

	hpButtonTextKey translationKey = "workflow_hp_button_text"

	hpResultCellPrimaryTextSuccessKey          translationKey = "workflow_hp_result_cell_primary_text_success"
	hpResultCellPrimaryTextPaymentFailedKey    translationKey = "workflow_hp_result_cell_primary_text_payment_failed"
	hpResultCellPrimaryTextBookingCancelledKey translationKey = "workflow_hp_result_cell_primary_text_booking_cancelled"
	hpResultCellPrimaryTextUnknownKey          translationKey = "workflow_hp_result_cell_primary_text_unknown"

	hpResultCellSecondaryTextSuccessKey          translationKey = "workflow_hp_result_cell_secondary_text_success"
	hpResultCellSecondaryTextPaymentFailedKey    translationKey = "workflow_hp_result_cell_secondary_text_payment_failed"
	hpResultCellSecondaryTextBookingCancelledKey translationKey = "workflow_hp_result_cell_secondary_text_booking_cancelled"
	hpResultCellSecondaryTextUnknownKey          translationKey = "workflow_hp_result_cell_secondary_text_unknown"

	hpResultCellWhatsappTextKey              translationKey = "workflow_hp_result_cell_whatsapp_text"
	hpResultCellCallDataTextKey              translationKey = "workflow_hp_result_cell_call_data_text"
	hpResultCellPaymentInfoPlusTextKey       translationKey = "workflow_hp_result_cell_payment_info_plus_text"
	hpResultCellPaymentInfoAdditionalTextKey translationKey = "workflow_hp_result_cell_payment_info_additional_text"

	hpSupportCellCallDataTextKey  translationKey = "workflow_hp_support_cell_call_data_text"
	hpSupportCellCallDataPhoneKey translationKey = "workflow_hp_support_cell_call_data_phone"
	hpSupportCellPrimaryTextKey   translationKey = "workflow_hp_support_cell_primary_text"
	hpSupportCellSecondaryTextKey translationKey = "workflow_hp_support_cell_secondary_text"
)

type GRPCHandler struct {
	config          *hotels.OrderConfig
	logger          log.Logger
	env             common.EnvType
	travelAPIClient TravelAPIClient
	l10nService     L10nService
}

func NewGRPCHotelOrderHandler(logger log.Logger, env common.EnvType, travelAPIClient TravelAPIClient, l10nService L10nService, config *hotels.OrderConfig) *GRPCHandler {
	return &GRPCHandler{
		config:          config,
		logger:          logger,
		env:             env,
		travelAPIClient: travelAPIClient,
		l10nService:     l10nService,
	}
}

func (h *GRPCHandler) GetL10NKeysetName() string {
	return l10nKeyset
}

func (h *GRPCHandler) GetL10NKeys() []string {
	return []string{
		string(defaultWaitingTitleKey),
		string(defaultWaitingMessageKey),
		string(reservedWaitingTitleKey),
		string(reservedWaitingMessageKey),
		string(promocodeErrorAlertTitleKey),
		string(promocodeErrorAlertMessageKey),
		string(paymentErrorAlertTitleKey),
		string(paymentErrorAlertMessageKey),
		string(buttonPayTitleKey),
		string(buttonExitTitleKey),
		string(hpButtonTextKey),
		string(hpResultCellPrimaryTextSuccessKey),
		string(hpResultCellPrimaryTextPaymentFailedKey),
		string(hpResultCellPrimaryTextBookingCancelledKey),
		string(hpResultCellPrimaryTextUnknownKey),
		string(hpResultCellSecondaryTextSuccessKey),
		string(hpResultCellSecondaryTextPaymentFailedKey),
		string(hpResultCellSecondaryTextBookingCancelledKey),
		string(hpResultCellSecondaryTextUnknownKey),
		string(hpResultCellWhatsappTextKey),
		string(hpResultCellCallDataTextKey),
		string(hpSupportCellCallDataTextKey),
		string(hpSupportCellCallDataPhoneKey),
		string(hpSupportCellPrimaryTextKey),
		string(hpSupportCellSecondaryTextKey),
		string(hpResultCellPaymentInfoPlusTextKey),
		string(hpResultCellPaymentInfoAdditionalTextKey),
	}
}

func (h *GRPCHandler) getTranslations(ctx context.Context) (*l10n.Keyset, error) {
	lang := common.GetLanguage(ctx)
	keyset, err := h.l10nService.Get(l10nKeyset, lang)
	if err != nil {
		if errors.Is(err, l10n.UnsupportedLanguage) {
			return nil, status.Error(codes.InvalidArgument, fmt.Sprintf("unsupported language '%s'", lang))
		}
		msg := fmt.Sprintf("no keyset for %s, %s", l10nKeyset, lang)
		ctxlog.Error(ctx, h.logger, msg, log.Error(err))
		return nil, status.Error(codes.Unknown, msg)
	}
	return keyset, nil
}

func (h *GRPCHandler) getTranslation(keyset *l10n.Keyset, key translationKey) string {
	// Не проверяем нашелся или нет ключ,
	// потому что проверка ключей происходит на старте приложения и при обновлении данных из танкера.
	// Проверяется наличие ключей, которые мы возвращаем в GetL10NKeys()
	return keyset.Keys[string(key)]
}

//TODO(adurenv) delete?
func (h *GRPCHandler) GetOrderStatus(ctx context.Context, req *commonAPI.GetOrderReq) (*commonAPI.GetOrderRsp, error) {
	// В get_order_status не прорастает purchase_token, хоть он и есть в спеке, поэтому ходим в get_order
	order, err := h.travelAPIClient.GetOrder(ctx, req.OrderId)
	if err != nil {
		return nil, clientscommon.ConvertHTTPErrorToGRPCError(ctx, h.logger, err, serviceName)
	}

	if !req.ForceFullOrderInfo {
		currentStatus := convertOrderStatusToProto(order.Status)
		if _, ok := common.FinalStatuses[currentStatus]; !ok {
			var purchaseToken *wrappers.StringValue = nil
			if order.Payment != nil && order.Payment.Current != nil {
				purchaseToken, err = common.GetTokenFromURL(order.Payment.Current.PaymentURL, purchaseTokenKey)
				if err != nil {
					ctxlog.Error(ctx, h.logger, "invalid get_order_status response", log.Error(err))
					return nil, status.Error(codes.Unknown, "invalid order info")
				}
			}
			return &commonAPI.GetOrderRsp{
				Status:        currentStatus,
				PurchaseToken: purchaseToken,
			}, nil
		}
	}

	hp, err := h.travelAPIClient.GetHotelHappyPage(ctx, req.OrderId)
	if err != nil {
		return nil, clientscommon.ConvertHTTPErrorToGRPCError(ctx, h.logger, err, serviceName)
	}
	orderStatus := convertOrderStatusToProto(hp.Order.Status)

	keyset, err := h.getTranslations(ctx)
	if err != nil {
		return nil, err
	}
	getTranslation := func(key translationKey) string {
		return h.getTranslation(keyset, key)
	}

	return &commonAPI.GetOrderRsp{
		Status:        orderStatus,
		HappyPageData: h.buildHappyPage(hp, order, &orderStatus, getTranslation),
	}, nil
}

func (h *GRPCHandler) GetExpectedAction(ctx context.Context, req *commonAPI.GetExpectedActionReq) (*commonAPI.GetExpectedActionRsp, error) {
	keyset, err := h.getTranslations(ctx)
	if err != nil {
		return nil, err
	}
	getTranslation := func(key translationKey) string {
		return h.getTranslation(keyset, key)
	}
	waiting := commonAPI.GetExpectedActionRsp{
		ExpectedAction: &commonAPI.GetExpectedActionRsp_Waiting_{
			Waiting: &commonAPI.GetExpectedActionRsp_Waiting{
				Title:    getTranslation(reservedWaitingTitleKey),
				Subtitle: getTranslation(reservedWaitingMessageKey),
				DelayMs:  h.config.ExpectedActionDefaultDelayMs,
			},
		},
	}

	// В get_order_status не прорастает purchase_token, хоть он и есть в спеке, поэтому ходим в get_order
	order, err := h.travelAPIClient.GetOrder(ctx, req.OrderId)
	if err != nil {
		var httpErr clientscommon.StatusError
		if xerrors.As(err, &httpErr) {
			if slices.Contains(waitingForStatuses, httpErr.Status) {
				ctxlog.Warn(ctx, h.logger, "error get order, waiting", log.Error(err))
				return &waiting, nil
			}
		}
		ctxlog.Error(ctx, h.logger, "error get order", log.Error(err))
		return nil, clientscommon.ConvertHTTPErrorToGRPCError(ctx, h.logger, err, serviceName)
	}
	currentStatus := convertOrderStatusToProto(order.Status)

	// Init payment
	// https://a.yandex-team.ru/arc_vcs/travel/frontend/portal/src/redux/sagas/hotels/bookAndPay/pollingOrder.ts?rev=r9331608#L111
	if order.Status == models.OrderStatusReserved ||
		(order.Status == models.OrderStatusReservedWithRestrictions && req.GetPayment() != nil && req.GetPayment().Command == commonAPI.Command_COMMAND_START_PAYMENT_WITH_RESTRICTIONS) ||
		(order.Status == models.OrderStatusPaymentFailed && (order.Payment == nil || order.Payment.Error == nil || order.Payment.Error.Code != models.PaymentErrorCodePaymentTimeout)) {
		var token *string
		if req.GetPayment() != nil && req.GetPayment().PaymentTestContextToken != nil && h.env != common.ProductionEnv {
			t := req.GetPayment().PaymentTestContextToken.GetValue()
			token = &t
		}
		err = h.travelAPIClient.StartPayment(ctx, req.OrderId, token)
		if err != nil {
			var httpErr clientscommon.StatusError
			if xerrors.As(err, &httpErr) {
				if httpErr.Status == http.StatusConflict || slices.Contains(waitingForStatuses, httpErr.Status) {
					ctxlog.Warn(ctx, h.logger, "error start payment, waiting", log.Error(err))
					return &waiting, nil
				}
			}
			ctxlog.Error(ctx, h.logger, "error start payment", log.Error(err))
			return nil, clientscommon.ConvertHTTPErrorToGRPCError(ctx, h.logger, err, serviceName)
		}

		return &waiting, nil
	}

	// в случаи OrderStatusPaymentFailed и PaymentErrorCodePaymentTimeout надо начинать с начала
	if _, isFinalStatus := common.FinalStatuses[currentStatus]; isFinalStatus || order.Status == models.OrderStatusPaymentFailed {
		hp, err := h.travelAPIClient.GetHotelHappyPage(ctx, req.OrderId)
		if err != nil {
			var httpErr clientscommon.StatusError
			if xerrors.As(err, &httpErr) {
				if slices.Contains(waitingForStatuses, httpErr.Status) {
					ctxlog.Warn(ctx, h.logger, "error get happy page, waiting", log.Error(err))
					return &waiting, nil
				}
			}
			ctxlog.Error(ctx, h.logger, "error get happy page", log.Error(err))
			return nil, clientscommon.ConvertHTTPErrorToGRPCError(ctx, h.logger, err, serviceName)
		}
		hpOrderStatus := convertOrderStatusToProto(hp.Order.Status)

		response := commonAPI.GetExpectedActionRsp{}
		response.ExpectedAction = &commonAPI.GetExpectedActionRsp_HappyPage{
			HappyPage: h.buildHappyPage(hp, order, &hpOrderStatus, getTranslation),
		}
		return &response, nil
	}

	var purchaseToken *wrappers.StringValue
	if order.Payment != nil && order.Payment.Current != nil {
		purchaseToken, err = common.GetTokenFromURL(order.Payment.Current.PaymentURL, purchaseTokenKey)
		if err != nil {
			ctxlog.Error(ctx, h.logger, "invalid get_order response", log.Error(err))
			return nil, status.Error(codes.Unknown, "invalid order info")
		}
	}

	switch order.Status {
	case models.OrderStatusInProgress:
		if purchaseToken != nil && len(purchaseToken.GetValue()) > 0 {
			response := commonAPI.GetExpectedActionRsp{}
			response.ExpectedAction = &commonAPI.GetExpectedActionRsp_Payment_{
				Payment: &commonAPI.GetExpectedActionRsp_Payment{
					Value: &commonAPI.GetExpectedActionRsp_Payment_PurchaseToken{
						PurchaseToken: purchaseToken.GetValue(),
					},
				},
			}
			return &response, nil
		} else {
			response := commonAPI.GetExpectedActionRsp{}
			response.ExpectedAction = &commonAPI.GetExpectedActionRsp_Waiting_{
				Waiting: &commonAPI.GetExpectedActionRsp_Waiting{
					Title:    getTranslation(defaultWaitingTitleKey),
					Subtitle: getTranslation(defaultWaitingMessageKey),
					DelayMs:  h.config.ExpectedActionDefaultDelayMs,
				},
			}
			return &response, nil
		}
	case models.OrderStatusReserved:
		response := commonAPI.GetExpectedActionRsp{}
		response.ExpectedAction = &commonAPI.GetExpectedActionRsp_Waiting_{
			Waiting: &commonAPI.GetExpectedActionRsp_Waiting{
				Title:    getTranslation(reservedWaitingTitleKey),
				Subtitle: getTranslation(reservedWaitingMessageKey),
				DelayMs:  h.config.ExpectedActionDefaultDelayMs,
			},
		}
		return &response, nil
	case models.OrderStatusReservedWithRestrictions:
		response := commonAPI.GetExpectedActionRsp{}
		response.ExpectedAction = &commonAPI.GetExpectedActionRsp_Payment_{
			Payment: &commonAPI.GetExpectedActionRsp_Payment{
				Value: &commonAPI.GetExpectedActionRsp_Payment_Warning{
					Warning: &commonAPI.AlertMessage{
						Title:    getTranslation(promocodeErrorAlertTitleKey),
						Subtitle: getTranslation(promocodeErrorAlertMessageKey),
						Buttons: []*commonAPI.Button{
							{
								Name: getTranslation(buttonExitTitleKey),
							},
							{
								Name:    getTranslation(buttonPayTitleKey),
								Command: commonAPI.Command_COMMAND_START_PAYMENT_WITH_RESTRICTIONS,
							},
						},
					},
				},
			},
		}
		return &response, nil
	case models.OrderStatusAwaitsPayment:
		if purchaseToken != nil && len(purchaseToken.GetValue()) > 0 {
			response := commonAPI.GetExpectedActionRsp{}
			response.ExpectedAction = &commonAPI.GetExpectedActionRsp_Payment_{
				Payment: &commonAPI.GetExpectedActionRsp_Payment{
					Value: &commonAPI.GetExpectedActionRsp_Payment_PurchaseToken{
						PurchaseToken: purchaseToken.GetValue(),
					},
				},
			}
			return &response, nil
		} else {
			response := commonAPI.GetExpectedActionRsp{}
			response.ExpectedAction = &commonAPI.GetExpectedActionRsp_Waiting_{
				Waiting: &commonAPI.GetExpectedActionRsp_Waiting{
					Title:    getTranslation(defaultWaitingTitleKey),
					Subtitle: getTranslation(defaultWaitingMessageKey),
					DelayMs:  h.config.ExpectedActionDefaultDelayMs,
				},
			}
			return &response, nil
		}
	}

	return nil, status.Error(codes.Unknown, serviceName+" unexpected error")
}

//TODO(adurnev) дописать
func (h *GRPCHandler) CreateOrder(ctx context.Context, req *hotelOrderAPI.CreateOrderReq) (*hotelOrderAPI.CreateOrderRsp, error) {
	/*var whiteLabel *models.WhiteLabel
	var yandexPlus *models.YandexPlus
	if req.PromoCampaignsApplied != nil {
		if req.PromoCampaignsApplied.WhiteLabel != nil {
			whiteLabel = &models.WhiteLabel{
				CustomerNumber: "", //req.PromoCampaignsApplied.WhiteLabel.CustomerNumber,
			}
		}
		if req.PromoCampaignsApplied.YandexPlus != nil {
			yandexPlus = &models.YandexPlus{
				Mode:   "", //req.PromoCampaignsApplied.YandexPlus.Mode,
				Points: 0,
			}
		}
	}
	request := models.CreateOrderRequest{
		PromoCampaignsApplied: models.PromoCampaignsApplied{
			WhiteLabel: whiteLabel,
			YandexPlus: yandexPlus,
		},
		Checksum:                "",
		CustomerEmail:           "",
		CustomerIP:              "",
		CustomerLogin:           "",
		CustomerPassportID:      "",
		CustomerPhone:           "",
		CustomerUserAgent:       "",
		CustomerYandexUID:       "",
		Guests:                  nil,
		Label:                   "",
		PaymentTestContextToken: "",
		PromoCodes:              nil,
		SelectedBedGroupIndex:   0,
		SessionKey:              "",
		SubscriptionParams:      models.SubscriptionParams{},
		Token:                   "",
		UseDeferredPayments:     false,
	}
	response, err := h.travelAPIClient.CreateOrder(ctx, &request)
	if err != nil {
		return nil, xerrors.Errorf("enable create order", err)
	}
	return &hotelOrderAPI.CreateOrderRsp{
		OrderId: response.Id,
	}, nil*/
	panic("not implemented")
}

func (h *GRPCHandler) buildHappyPage(hp *models.GetHotelHappyPageRsp, orderOrig *models.GetOrderRsp, status *commonAPI.OrderStatus, getTranslation func(key translationKey) string) *commonAPI.HappyPageData {
	order := hp.Order
	amount, _ := strconv.ParseFloat(order.OrderInfo.RateInfo.HotelCharges.Totals.Base.Amount, 64)
	if orderOrig.Payment != nil {
		amount = orderOrig.Payment.AmountPaid.Value
	}

	finalStatus := common.BuildFinalStatus(status)

	var plusData *commonAPI.PlusHappyPageData
	yp := order.PromoCampaignsApplied.YandexPlus
	if yp != nil && yp.Mode == models.YandexPlusModeTopup {
		plusData = &commonAPI.PlusHappyPageData{
			Text:   getTranslation(hpResultCellPaymentInfoPlusTextKey),
			Amount: float64(yp.Points),
		}
	}
	return &commonAPI.HappyPageData{
		Status:     finalStatus,
		ImageUrl:   h.buildImageURL(&finalStatus), //todo(adurnev) дозаполнить
		ButtonText: getTranslation(hpButtonTextKey),
		ResultCell: &commonAPI.ResultCellData{
			PrimaryText:         buildPrimaryText(&finalStatus, getTranslation),
			SecondaryText:       buildSecondaryText(&finalStatus, getTranslation),
			Email:               order.GuestInfo.CustomerEmail,
			WhatsappSectionText: getTranslation(hpResultCellWhatsappTextKey),
			HotelsPaymentInfoData: &commonAPI.HotelsPaymentInfoData{
				Amount:        amount,
				PersonCount:   int32(len(order.GuestInfo.Guests)),
				NightsCount:   int32(len(order.OrderInfo.RateInfo.HotelCharges.Nightly)),
				IsInstallment: order.Payment.UsesDeferredPayments,
				//Deadline:       "",
				AdditionalText: getTranslation(hpResultCellPaymentInfoAdditionalTextKey),
				PlusData:       plusData,
			},
			CallData: &commonAPI.CallData{
				Phone: order.OrderInfo.BasicHotelInfo.Phone,
				Text:  getTranslation(hpResultCellCallDataTextKey),
			},
		},
		HotelBookedCell: &commonAPI.HotelBookedCellData{
			Name:             order.OrderInfo.BasicHotelInfo.Name,
			ImageUrl:         strings.ReplaceAll(order.OrderInfo.BasicHotelInfo.ImageURLTemplate, "%s", h.config.HappyPageHotelImageSize),
			ImageUrlTemplate: order.OrderInfo.BasicHotelInfo.ImageURLTemplate,
			DateStart:        order.OrderInfo.RequestInfo.CheckinDate,
			DateFinish:       order.OrderInfo.RequestInfo.CheckoutDate,
			Address:          order.OrderInfo.BasicHotelInfo.Address,
			Coordinates: &commonAPI.Coordinates{
				Latitude:  order.OrderInfo.BasicHotelInfo.Coordinates.Latitude,
				Longitude: order.OrderInfo.BasicHotelInfo.Coordinates.Longitude,
			},
		},
		SupportCell: &commonAPI.SupportCellData{
			CallData: &commonAPI.CallData{
				Phone: getTranslation(hpSupportCellCallDataPhoneKey),
				Text:  getTranslation(hpSupportCellCallDataTextKey),
			},
			PrimaryText:   getTranslation(hpSupportCellPrimaryTextKey),
			SecondaryText: getTranslation(hpSupportCellSecondaryTextKey),
		},
	}
}

func (h *GRPCHandler) buildImageURL(status *commonAPI.HappyPageFinalStatus) string {
	if status != nil && *status == commonAPI.HappyPageFinalStatus_HAPPY_PAGE_FINAL_STATUS_SUCCESS {
		return h.config.HappyPageImageSuccessURL
	}
	return h.config.HappyPageImageFailedURL
}

func buildPrimaryText(status *commonAPI.HappyPageFinalStatus, getTranslation func(key translationKey) string) string {
	switch *status {
	case commonAPI.HappyPageFinalStatus_HAPPY_PAGE_FINAL_STATUS_SUCCESS:
		return getTranslation(hpResultCellPrimaryTextSuccessKey)
	case commonAPI.HappyPageFinalStatus_HAPPY_PAGE_FINAL_STATUS_PAYMENT_FAILED:
		return getTranslation(hpResultCellPrimaryTextPaymentFailedKey)
	case commonAPI.HappyPageFinalStatus_HAPPY_PAGE_FINAL_STATUS_BOOKING_CANCELLED:
		return getTranslation(hpResultCellPrimaryTextBookingCancelledKey)
		//"Оплачено! Заказ в обработке"
	}
	return getTranslation(hpResultCellPrimaryTextUnknownKey)
}

func buildSecondaryText(status *commonAPI.HappyPageFinalStatus, getTranslation func(key translationKey) string) string {
	switch *status {
	case commonAPI.HappyPageFinalStatus_HAPPY_PAGE_FINAL_STATUS_SUCCESS:
		return getTranslation(hpResultCellSecondaryTextSuccessKey)
	case commonAPI.HappyPageFinalStatus_HAPPY_PAGE_FINAL_STATUS_PAYMENT_FAILED:
		return getTranslation(hpResultCellSecondaryTextPaymentFailedKey)
	case commonAPI.HappyPageFinalStatus_HAPPY_PAGE_FINAL_STATUS_BOOKING_CANCELLED:
		return getTranslation(hpResultCellSecondaryTextBookingCancelledKey)
	}
	return getTranslation(hpResultCellSecondaryTextUnknownKey)
}

func (h *GRPCHandler) GetServiceRegisterer() func(*grpc.Server) {
	return func(server *grpc.Server) {
		hotelOrderAPI.RegisterHotelOrderAPIServer(server, h)
	}
}

func convertOrderStatusToProto(s models.OrderStatus) commonAPI.OrderStatus {
	switch s {
	case models.OrderStatusInProgress, models.OrderStatusAwaitsPayment:
		return commonAPI.OrderStatus_ORDER_STATUS_IN_PROGRESS

	case models.OrderStatusReserved:
		return commonAPI.OrderStatus_ORDER_STATUS_WAIT_START_PAYMENT

	case models.OrderStatusReservedWithRestrictions:
		return commonAPI.OrderStatus_ORDER_STATUS_WAIT_START_PAYMENT_WITH_RESTRICTIONS

	case models.OrderStatusConfirmed, models.OrderStatusRefunded:
		return commonAPI.OrderStatus_ORDER_STATUS_SUCCESS

	case models.OrderStatusCancelled, models.OrderStatusCancelledWithRefund:
		return commonAPI.OrderStatus_ORDER_STATUS_CANCELLED

	case models.OrderStatusPaymentFailed:
		return commonAPI.OrderStatus_ORDER_STATUS_PAYMENT_FAILED

	case models.OrderStatusRefundFailed, models.OrderStatusFailed:
		return commonAPI.OrderStatus_ORDER_STATUS_FAILED
	}
	return commonAPI.OrderStatus_ORDER_STATUS_UNKNOWN
}

type TravelAPIClient interface {
	GetOrder(ctx context.Context, orderID string) (*models.GetOrderRsp, error)
	GetOrderByToken(ctx context.Context, request *models.GetOrderByTokenRequest) (*models.GetOrderByTokenResponse, error)
	GetHotelHappyPage(ctx context.Context, orderID string) (*models.GetHotelHappyPageRsp, error)
	StartPayment(ctx context.Context, orderID string, paymentTestContextToken *string) error
	CreateOrder(ctx context.Context, req *models.CreateOrderRequest) (*models.CreateOrderResponse, error)
}

type L10nService interface {
	Get(keysetName string, language string) (*l10n.Keyset, error)
}
