package handler

import (
	"context"

	"google.golang.org/grpc"

	"a.yandex-team.ru/library/go/core/log"
	hotelReviewsAPI "a.yandex-team.ru/travel/app/backend/api/hotelreviews/v1"
	v1 "a.yandex-team.ru/travel/app/backend/api/hotels/v1"
	"a.yandex-team.ru/travel/app/backend/internal/hotels"
	"a.yandex-team.ru/travel/app/backend/internal/lib/clientscommon"
	"a.yandex-team.ru/travel/app/backend/internal/lib/travelapiclient/models"
)

const (
	serviceName = "hotel reviews"
)

type GRPCHandler struct {
	config           *hotels.ReviewsConfig
	logger           log.Logger
	travelAPIClient  TravelAPIClient
	translateService hotels.Translation
}

func NewGRPCHandler(config *hotels.ReviewsConfig, logger log.Logger, travelAPIClient TravelAPIClient, translateService hotels.Translation) *GRPCHandler {
	return &GRPCHandler{
		config:           config,
		logger:           logger,
		travelAPIClient:  travelAPIClient,
		translateService: translateService,
	}
}

func (h *GRPCHandler) GetReviews(ctx context.Context, req *v1.GetReviewsReq) (*v1.ReviewsRsp, error) {
	pagingParams := hotels.ConvertPagingData(req.PagingParams)
	if pagingParams != nil && pagingParams.Limit == 0 {
		pagingParams.Limit = int(h.config.ReviewsLimit)
	}

	selectedSort := req.Sort
	if selectedSort == v1.ReviewSort_REVIEW_SORT_UNKNOWN {
		selectedSort = v1.ReviewSort_REVIEW_SORT_RELEVANT_FIRST
	}
	request := models.GetHotelReviewsReq{
		AnalyticsParams: hotels.ConvertAnalyticsData(req.AnalyticsData),
		QueryData:       hotels.ConvertHotelID(req.HotelId),
		PagingParams:    pagingParams,
		Sort:            hotels.ConvertSort(selectedSort),
	}

	selectedPhrase := ""
	if req.PhraseParams != nil {
		request.Phrase = &models.PhraseReq{
			Filter: req.PhraseParams.Filter,
			Limit:  int(req.PhraseParams.Limit),
		}
		if request.Phrase.Limit == 0 {
			request.Phrase.Limit = int(h.config.PhraseLimit)
		}
		selectedPhrase = req.PhraseParams.Filter
	}

	response, err := h.travelAPIClient.GetHotelReviews(ctx, &request)
	if err != nil {
		return nil, clientscommon.ConvertHTTPErrorToGRPCError(ctx, h.logger, err, serviceName)
	}

	res := hotels.ConvertReviewsRsp(&response.ReviewsInfo, h.translateService.GetFunc(ctx), h.config)
	res.Phrases.Selected = selectedPhrase
	res.Sort.Selected = selectedSort

	return res, nil
}

func buildUserReactionStr(reaction v1.ReviewReactionType) string {
	switch reaction {
	case v1.ReviewReactionType_REVIEW_REACTION_NONE:
		return "NONE"
	case v1.ReviewReactionType_REVIEW_REACTION_LIKE:
		return "LIKE"
	case v1.ReviewReactionType_REVIEW_REACTION_DISLIKE:
		return "DISLIKE"
	default:
		return "NONE"
	}
}

func (h *GRPCHandler) AddReview(ctx context.Context, req *v1.AddReviewReq) (*v1.ReviewRsp, error) {
	request := models.AddHotelReviewReq{
		AnalyticsParams: hotels.ConvertAnalyticsData(req.AnalyticsData),
		QueryData:       hotels.ConvertHotelID(req.HotelId),
		Body: models.HotelReviewReq{
			Text:     req.Body.Text,
			Rating:   int(req.Body.Rating),
			ImageIds: req.Body.ImageIds,
		},
	}

	response, err := h.travelAPIClient.AddHotelReview(ctx, &request)
	if err != nil {
		return nil, err
	}

	res := hotels.ConvertReview(response.TextReview, h.config)

	return res, nil
}

func (h *GRPCHandler) DeleteReview(ctx context.Context, req *v1.DeleteReviewReq) (*v1.DeleteReviewRsp, error) {
	request := models.DeleteHotelReviewReq{
		AnalyticsParams: hotels.ConvertAnalyticsData(req.AnalyticsData),
		QueryData:       hotels.ConvertHotelID(req.HotelId),
		ReviewID:        req.ReviewId,
	}

	err := h.travelAPIClient.DeleteHotelReview(ctx, &request)
	if err != nil {
		return nil, err
	}

	return &v1.DeleteReviewRsp{}, nil
}

func (h *GRPCHandler) EditReview(ctx context.Context, req *v1.EditReviewReq) (*v1.ReviewRsp, error) {
	request := models.EditHotelReviewReq{
		ReviewID: req.ReviewId,
		AddHotelReviewReq: models.AddHotelReviewReq{
			AnalyticsParams: hotels.ConvertAnalyticsData(req.AnalyticsData),
			QueryData:       hotels.ConvertHotelID(req.HotelId),
			Body: models.HotelReviewReq{
				Text:     req.Body.Text,
				Rating:   int(req.Body.Rating),
				ImageIds: req.Body.ImageIds,
			},
		},
	}

	response, err := h.travelAPIClient.EditHotelReview(ctx, &request)
	if err != nil {
		return nil, err
	}

	res := hotels.ConvertReview(response.TextReview, h.config)

	return res, nil
}

func (h *GRPCHandler) SetReaction(ctx context.Context, req *v1.SetReactionReq) (*v1.SetReactionRsp, error) {
	request := models.SetReactionHotelReviewReq{
		ReviewID:        req.ReviewId,
		Reaction:        buildUserReactionStr(req.Reaction),
		AnalyticsParams: hotels.ConvertAnalyticsData(req.AnalyticsData),
		QueryData:       hotels.ConvertHotelID(req.HotelId),
	}

	err := h.travelAPIClient.SetReactionHotelReview(ctx, &request)
	if err != nil {
		return nil, err
	}

	return &v1.SetReactionRsp{}, nil
}

func (h *GRPCHandler) DeleteImages(ctx context.Context, req *v1.DeleteImagesReq) (*v1.DeleteImagesRsp, error) {
	request := models.DeleteImagesHotelReviewReq{
		AnalyticsParams: hotels.ConvertAnalyticsData(req.AnalyticsData),
		QueryData:       hotels.ConvertHotelID(req.HotelId),
		Body: models.DeleteImagesHotelReviewReqBody{
			ImageIds: req.ImageIds,
		},
	}

	err := h.travelAPIClient.DeleteImagesHotelReview(ctx, &request)
	if err != nil {
		return nil, err
	}

	return &v1.DeleteImagesRsp{}, nil
}

func (h *GRPCHandler) GetServiceRegisterer() func(*grpc.Server) {
	return func(server *grpc.Server) {
		hotelReviewsAPI.RegisterHotelReviewsAPIServer(server, h)
	}
}

type TravelAPIClient interface {
	GetHotelReviews(ctx context.Context, req *models.GetHotelReviewsReq) (*models.GetHotelReviewsRsp, error)

	AddHotelReview(ctx context.Context, request *models.AddHotelReviewReq) (*models.HotelReviewRsp, error)
	EditHotelReview(ctx context.Context, request *models.EditHotelReviewReq) (*models.HotelReviewRsp, error)
	DeleteHotelReview(ctx context.Context, request *models.DeleteHotelReviewReq) error

	SetReactionHotelReview(ctx context.Context, request *models.SetReactionHotelReviewReq) error

	UploadImageHotelReview(ctx context.Context, request *models.UploadImageHotelReviewReq) (*models.UploadImageHotelReviewRsp, error)
	DeleteImagesHotelReview(ctx context.Context, request *models.DeleteImagesHotelReviewReq) error
}
