package hotels

import (
	"context"
	"strings"

	"github.com/golang/protobuf/ptypes/wrappers"
	"google.golang.org/genproto/googleapis/type/date"
	"google.golang.org/grpc/codes"
	"google.golang.org/grpc/status"
	"google.golang.org/protobuf/types/known/wrapperspb"

	"a.yandex-team.ru/library/go/core/log"
	v1 "a.yandex-team.ru/travel/app/backend/api/common/v1"
	hotelsAPI "a.yandex-team.ru/travel/app/backend/api/hotels/v1"
	"a.yandex-team.ru/travel/app/backend/internal/common"
	"a.yandex-team.ru/travel/app/backend/internal/lib/clientscommon"
	"a.yandex-team.ru/travel/app/backend/internal/lib/travelapiclient/models"
)

const (
	serviceTitle  = "hotels"
	ImageSuffix   = "%s"
	ImageSizeOrig = "orig"
)

type Service struct {
	logger           log.Logger
	client           TravelAPIClient
	config           Config
	translateService Translation
}

func New(logger log.Logger, client TravelAPIClient, config Config, translateService Translation) *Service {
	return &Service{logger: logger, client: client, config: config, translateService: translateService}
}

var typeMap = map[models.SuggestType]hotelsAPI.SuggestItemType{
	models.SuggestTypeRegion:    hotelsAPI.SuggestItemType_SUGGEST_ITEM_TYPE_REGION,
	models.SuggestTypeHotel:     hotelsAPI.SuggestItemType_SUGGEST_ITEM_TYPE_HOTEL,
	models.SuggestTypeHistory:   hotelsAPI.SuggestItemType_SUGGEST_ITEM_TYPE_HISTORY,
	models.SuggestTypeCrossSale: hotelsAPI.SuggestItemType_SUGGEST_ITEM_TYPE_TRIP,
}

var domainMap = map[string]string{
	"RU": "ru",
	"UA": "ua",
}

const defaultDomain = "ru"

func mapDate(source models.Date) *date.Date {
	return &date.Date{
		Year:  int32(source.Year()),
		Month: int32(source.Month()),
		Day:   int32(source.Day()),
	}
}

func (s *Service) Suggest(ctx context.Context, req *hotelsAPI.HotelsSuggestReq) (*hotelsAPI.HotelsSuggestRsp, error) {
	if req.Limit > s.config.SuggestLimit {
		req.Limit = s.config.SuggestLimit
	}

	lang := common.GetLanguage(ctx)
	domain := defaultDomain
	country := common.GetCountryCode(ctx)
	if d, found := domainMap[country]; found {
		domain = d
	}
	resp, err := s.client.Suggest(ctx, req.Query, int(req.Limit), lang, domain, common.GetSessionID(ctx), int(req.RequestIndex))
	if err != nil {
		return nil, clientscommon.ConvertHTTPErrorToGRPCError(ctx, s.logger, err, serviceTitle)
	}
	return mapClientResponseToProto(resp), nil
}

func (s *Service) LogSuggest(ctx context.Context, req *hotelsAPI.LogHotelsSuggestSelectedReq) error {
	if err := s.client.LogSuggest(ctx, req.SelectedId, common.GetSessionID(ctx), int(req.RequestIndex), true, true); err != nil {
		return clientscommon.ConvertHTTPErrorToGRPCError(ctx, s.logger, err, serviceTitle)
	}
	return nil
}

func (s *Service) SearchHotels(ctx context.Context, req *hotelsAPI.SearchHotelsReq) (*hotelsAPI.SearchHotelsRsp, error) {
	request, err := s.convertToSearchHotelsRequest(req)
	if err != nil {
		return nil, status.Error(codes.InvalidArgument, "bad request")
	}
	response, err := s.client.SearchHotels(ctx, request)
	if err != nil {
		return nil, clientscommon.ConvertHTTPErrorToGRPCError(ctx, s.logger, err, serviceTitle)
	}
	converted, err := s.convertToSearchHotelRsp(ctx, response, req.ImageParams)
	if err != nil {
		return nil, status.Error(codes.Internal, "bad convert")
	}
	return converted, nil
}

func (s *Service) GetHotelOffers(ctx context.Context, req *hotelsAPI.GetHotelOffersReq) (*hotelsAPI.GetHotelOffersRsp, error) {
	request, err := convertToHotelOffersRequest(req)
	if err != nil {
		return nil, status.Error(codes.InvalidArgument, "bad request")
	}
	response, err := s.client.GetHotelOffers(ctx, request)
	if err != nil {
		return nil, clientscommon.ConvertHTTPErrorToGRPCError(ctx, s.logger, err, serviceTitle)
	}
	return convertToHotelOffersRsp(response, s.translateService.GetFunc(ctx)), nil
}

func (s *Service) GetHotelImages(ctx context.Context, req *hotelsAPI.GetHotelImagesReq) (*hotelsAPI.GetHotelImagesRsp, error) {
	request, err := convertToHotelImagesRequest(req, s.config)
	if err != nil {
		return nil, status.Error(codes.InvalidArgument, "bad request")
	}
	response, err := s.client.GetHotelImages(ctx, request)
	if err != nil {
		return nil, clientscommon.ConvertHTTPErrorToGRPCError(ctx, s.logger, err, serviceTitle)
	}
	return convertHotelImagesRsp(response)
}

func (s *Service) GetSimilarHotels(ctx context.Context, req *hotelsAPI.GetSimilarHotelsReq) (*hotelsAPI.GetSimilarHotelsRsp, error) {
	request, err := convertToSimilarHotelsRequest(req)
	if err != nil {
		return nil, status.Error(codes.InvalidArgument, "bad request")
	}
	response, err := s.client.GetSimilarHotels(ctx, request)
	if err != nil {
		return nil, clientscommon.ConvertHTTPErrorToGRPCError(ctx, s.logger, err, serviceTitle)
	}
	return s.convertSimilarHotelsRsp(ctx, response, req.ImageParams)
}

func (s *Service) GetHotelInfo(ctx context.Context, req *hotelsAPI.GetHotelInfoReq) (*hotelsAPI.GetHotelInfoRsp, error) {
	request, err := convertToHotelInfoRequest(req, s.config.HotelInfo)
	if err != nil {
		return nil, status.Error(codes.InvalidArgument, "bad request")
	}
	response, err := s.client.GetHotelInfo(ctx, request)
	if err != nil {
		return nil, clientscommon.ConvertHTTPErrorToGRPCError(ctx, s.logger, err, serviceTitle)
	}
	return s.convertToHotelInfoRsp(ctx, response)
}

func (s *Service) GetHotelCounters(ctx context.Context, req *hotelsAPI.GetHotelCountersReq) (*hotelsAPI.GetHotelCountersRsp, error) {
	request, err := convertToGetCountersReq(req)
	if err != nil {
		return nil, err
	}

	response, err := s.client.GetHotelsCounters(ctx, request)
	if err != nil {
		return nil, clientscommon.ConvertHTTPErrorToGRPCError(ctx, s.logger, err, serviceTitle)
	}
	return convertHotelsCountersRsp(*response), nil
}

func (s *Service) convertSimilarHotelsRsp(ctx context.Context, response *models.GetSimilarHotelsResponse, imageParams *hotelsAPI.ImageParams) (*hotelsAPI.GetSimilarHotelsRsp, error) {
	if response == nil {
		return nil, nil
	}
	rsp := response.SimilarHotelsInfo

	hotels, err := s.ConvertHotels(ctx, rsp.Hotels, imageParams)
	if err != nil {
		return nil, err
	}

	return &hotelsAPI.GetSimilarHotelsRsp{
		SimilarHotelsInfo: &hotelsAPI.SimilarHotelsInfo{
			NextPollingRequestDelayMs: &wrapperspb.Int32Value{Value: int32(rsp.NextPollingRequestDelayMs)},
			OfferSearchProgress:       convertOfferSearchProgress(&rsp.OfferSearchProgress),
			PollingFinished:           rsp.OfferSearchProgress.Finished,
			Hotels:                    hotels,
		},
	}, nil
}

func (s *Service) convertToSearchHotelsRequest(req *hotelsAPI.SearchHotelsReq) (request models.SearchHotelsRequest, err error) {
	converted := models.SearchHotelsRequest{}

	converted.AnalyticsParams = ConvertAnalyticsData(req.AnalyticsData)

	main := req.QueryData
	converted.QueryData.TotalHotelsLimit = int(s.config.HotelsLimit)
	converted.QueryData.ImageOnlyTop = true
	limit := int(s.config.ImagesLimit)
	if req.ImageParams != nil && req.ImageParams.Limit > 0 {
		limit = int(req.ImageParams.Limit)
	}
	converted.QueryData.ImageLimit = limit
	if main.SearchContext != nil {
		converted.QueryData.Context = &main.SearchContext.Value
	}
	converted.QueryData.Adults = int(main.Adults)
	if main.GeoId != nil {
		geoID := int(main.GeoId.Value)
		converted.QueryData.GeoID = &geoID
	}
	converted.QueryData.HotelSearchStartReason = convert(&main.SearchReason)
	if len(main.ChildrenAges) > 0 {
		for _, val := range main.ChildrenAges {
			converted.QueryData.ChildrenAges = append(converted.QueryData.ChildrenAges, int(val))
		}
	}
	converted.QueryData.CheckinDate = common.FormatDateProto(main.CheckinDate)
	converted.QueryData.CheckoutDate = common.FormatDateProto(main.CheckoutDate)
	converted.QueryData.PageHotelCount = int(main.PageHotelCount)

	if main.PollingData != nil {
		converted.QueryData.PollingData = &models.PollingData{}
		converted.QueryData.PollingData.PollIteration = int(main.PollingData.PollIteration)
		converted.QueryData.PollingData.PollEpoch = int(main.PollingData.PollEpoch)
		if main.PollingData.NavigationToken != nil {
			converted.QueryData.PollingData.NavigationToken = &main.PollingData.NavigationToken.Value
		}
	}

	converted.QueryData.Bbox = convertBbox(main.Bbox)
	converted.QueryData.SortOrigin = convertCoordinates(main.NearestHotelsSortOrigin)

	filters := req.FilterData

	atoms := make([]string, 0)
	priceFilter := models.HotelPriceFilter{}
	filterSort := hotelsAPI.Sort_SORT_UNKNOWN
	for _, f := range filters {
		if f.GetPrice() != nil {
			if f.GetPrice().GetFrom() != nil {
				priceFilter.PriceMin = f.GetPrice().GetFrom().GetValue()
			}
			if f.GetPrice().GetTo() != nil {
				priceFilter.PriceMax = f.GetPrice().GetTo().GetValue()
			}
		} else if f.GetAtoms() != nil {
			atoms = append(atoms, f.GetAtoms().GetValues()...)
		} else if f.GetSort() != nil {
			filterSort = f.GetSort().Sorter
		}
	}

	converted.PriceFilter = priceFilter
	converted.Filters = models.HotelsSearchFilter{
		Atoms: atoms,
	}

	sort := filterSort
	sorter := req.QuickSort
	if sorter != nil {
		sort = sorter.Sorter
	}
	converted.Sorter.SorterID = convertSorter(sort)

	return converted, nil
}

func convertCoordinates(coordinates *hotelsAPI.Coordinates) *models.Coordinates {
	var res *models.Coordinates
	if coordinates != nil {
		res = &models.Coordinates{
			Latitude:  coordinates.Latitude,
			Longitude: coordinates.Longitude,
		}
	}
	return res
}

func convertBbox(bbox *hotelsAPI.BoundingBox) *models.BoundingBox {
	var res *models.BoundingBox
	if bbox != nil && bbox.LeftDown != nil && bbox.UpRight != nil {
		res = &models.BoundingBox{
			Coordinates: []models.Coordinates{
				*convertCoordinates(bbox.LeftDown),
				*convertCoordinates(bbox.UpRight),
			},
		}
	}
	return res
}

func ConvertPagingData(paging *hotelsAPI.PagingParams) *models.PagingParams {
	res := models.PagingParams{}
	if paging == nil {
		return &res
	}

	res.Offset = int(paging.Offset)
	res.Limit = int(paging.Limit)

	return &res
}

func ConvertHotelID(hotelID *hotelsAPI.HotelID) models.GetHotelInfoQueryData {
	return models.GetHotelInfoQueryData{
		Permalink: hotelID.GetPermalink(),
		HotelSlug: hotelID.GetHotelSlug(),
	}
}

func (s *Service) convertToSearchHotelRsp(ctx context.Context, response *models.SearchHotelsResponse, imageParams *hotelsAPI.ImageParams) (rsp *hotelsAPI.SearchHotelsRsp, err error) {
	translate := s.translateService.GetFunc(ctx)
	r := *response
	converted := &hotelsAPI.SearchHotelsRsp{}
	converted.SearchContext = r.SearchContext

	navigationTokens := hotelsAPI.NavigationTokens{}
	token := &wrappers.StringValue{}
	token.Value = r.NavigationTokens.NextPage
	navigationTokens.NextPage = token
	converted.NavigationTokens = &navigationTokens

	bbox := hotelsAPI.BoundingBox{
		LeftDown: &hotelsAPI.Coordinates{
			Latitude:  r.Bbox[0].Latitude,
			Longitude: r.Bbox[0].Longitude,
		},
		UpRight: &hotelsAPI.Coordinates{
			Latitude:  r.Bbox[1].Latitude,
			Longitude: r.Bbox[1].Longitude,
		},
	}
	converted.Bbox = &bbox

	searchProgress := hotelsAPI.OfferSearchProgress{}
	searchProgress.Finished = r.OfferSearchProgress.Finished
	searchProgress.PartnersTotal = r.OfferSearchProgress.PartnersTotal
	searchProgress.PartnersComplete = r.OfferSearchProgress.PartnersComplete
	converted.OfferSearchProgress = &searchProgress

	converted.PollingFinished = r.OfferSearchProgress.Finished
	nextPollingDelayMs := wrapperspb.Int32Value{Value: int32(r.NextPollingRequestDelayMs)}
	converted.NextPollingRequestDelayMs = &nextPollingDelayMs

	hotels, err := s.ConvertHotels(ctx, r.Hotels, imageParams)
	if err != nil {
		return nil, err
	}
	converted.Hotels = hotels
	converted.FoundHotelCount = int32(r.FoundHotelsCount)

	quickFilters := make([]*hotelsAPI.QuickFilters, 0)

	var selectedSort *hotelsAPI.QuickSorter
	sorters := make([]*hotelsAPI.QuickSorter, 0, len(r.SortInfo.QuickSorterGroup))
	for _, quickSorterGroup := range r.SortInfo.QuickSorterGroup {
		sorter := getSorter(quickSorterGroup.SortTypes[0])
		if sorter == hotelsAPI.Sort_SORT_UNKNOWN {
			continue
		}
		sort := &hotelsAPI.QuickSorter{
			Sorter:    sorter,
			Name:      quickSorterGroup.SortTypes[0].Name,
			ShortName: s.getShortNameSort(sorter, translate),
		}
		sorters = append(sorters, sort)
		if quickSorterGroup.SortTypes[0].ID == r.SortInfo.SelectedSortID {
			selectedSort = sort
		}
	}
	converted.Sorters = sorters

	quickFilterSort := hotelsAPI.QuickFilters{
		Value: &hotelsAPI.QuickFilters_Sort{
			Sort: &hotelsAPI.QuickFilterSort{
				Name:     translate(HotelSortNameKey),
				Selected: selectedSort,
				Available: &hotelsAPI.QuickFilterSort_SortList{
					Values: sorters,
				},
			},
		},
	}
	quickFilters = append(quickFilters, &quickFilterSort)

	histogramBounds := make([]uint32, len(r.FilterInfo.PriceFilter.HistogramBounds))
	for i, bound := range r.FilterInfo.PriceFilter.HistogramBounds {
		histogramBounds[i] = uint32(bound)
	}
	histogramCounts := make([]uint32, len(r.FilterInfo.PriceFilter.HistogramCounts))
	for i, count := range r.FilterInfo.PriceFilter.HistogramCounts {
		histogramCounts[i] = uint32(count)
	}
	converted.PriceFilter = &hotelsAPI.PriceFilter{
		Name:             translate(PriceNameKey),
		Currency:         r.FilterInfo.PriceFilter.Currency,
		MaxPriceEstimate: uint32(r.FilterInfo.PriceFilter.MaxPriceEstimate),
		MinPriceEstimate: uint32(r.FilterInfo.PriceFilter.MinPriceEstimate),
		HistogramBounds:  histogramBounds,
		HistogramCounts:  histogramCounts,
	}
	createPrice := func() *hotelsAPI.PriceInterval {
		return &hotelsAPI.PriceInterval{
			Currency: converted.PriceFilter.Currency,
			From: &wrapperspb.UInt32Value{
				Value: converted.PriceFilter.MinPriceEstimate,
			},
			To: &wrapperspb.UInt32Value{
				Value: converted.PriceFilter.MaxPriceEstimate,
			},
		}
	}
	price := createPrice()
	selectedPrice := createPrice()
	if r.FilterInfo.Params.FilterPriceFrom != nil {
		selectedPrice.From.Value = uint32(*r.FilterInfo.Params.FilterPriceFrom)
	}
	if r.FilterInfo.Params.FilterPriceTo != nil {
		selectedPrice.To.Value = uint32(*r.FilterInfo.Params.FilterPriceTo)
	}
	quickFilterPrice := hotelsAPI.QuickFilters{
		Value: &hotelsAPI.QuickFilters_Price{
			Price: &hotelsAPI.QuickFilterPrice{
				ShortName:     &wrappers.StringValue{Value: translate(PriceShortNameKey)},
				Name:          &wrappers.StringValue{Value: translate(PriceNameKey)},
				Price:         price,
				SelectedPrice: selectedPrice,
			},
		},
	}
	quickFilters = append(quickFilters, &quickFilterPrice)

	//Похоже что тут приходит только кешбек плюса (для desktop), если передать X-Ya-User-Device=touch придут и другие быстрые фильтры
	for _, f := range r.FilterInfo.QuickFilters {
		//https://st.yandex-team.ru/TRAVELAPP-1919
		if f.ID == "all-inclusive-quick" || f.ID == "5-stars" {
			continue
		}
		atomsOn := make([]*wrappers.StringValue, len(f.AtomsOn))
		for k := range f.AtomsOn {
			atomsOn[k] = &wrappers.StringValue{Value: f.AtomsOn[k]}
		}
		atomsOff := make([]*wrappers.StringValue, len(f.AtomsOff))
		for k := range f.AtomsOff {
			atomsOff[k] = &wrappers.StringValue{Value: f.AtomsOff[k]}
		}
		filter := hotelsAPI.QuickFilters{
			Value: &hotelsAPI.QuickFilters_Filter{
				Filter: &hotelsAPI.QuickFilter{
					Id:       &wrappers.StringValue{Value: f.ID},
					Name:     &wrappers.StringValue{Value: f.Name},
					Hint:     &wrappers.StringValue{Value: f.Hint},
					Enabled:  filterEnabled(f.Enabled, f.Hint),
					AtomsOn:  atomsOn,
					AtomsOff: atomsOff,
				},
			},
		}

		quickFilters = append(quickFilters, &filter)
	}

	detailedFilters := r.FilterInfo.DetailedFilter
	detailedFiltersRsp := hotelsAPI.Filters{}
	detailed := make([]*hotelsAPI.Filter, len(detailedFilters))
	for i, f := range detailedFilters {
		filter := convertDetailedFilter(f)
		detailed[i] = filter
	}

	converted.QuickFilters = quickFilters
	detailedFiltersRsp.DetailedFilters = detailed

	filtersBatches := r.FilterInfo.DetailedFilterBatches
	batch := make([]*hotelsAPI.FilterBatch, len(filtersBatches))
	for i, f := range filtersBatches {
		res := make([]*hotelsAPI.Filter, len(f.Filters))
		for j, ff := range f.Filters {
			res[j] = convertDetailedFilter(ff)
		}
		batch[i] = &hotelsAPI.FilterBatch{
			DetailedFilter: res,
		}
	}
	detailedFiltersRsp.DetailedFiltersBatches = batch

	converted.DetailedFilters = &detailedFiltersRsp

	converted.Params = &hotelsAPI.FilterParams{
		Atoms: r.FilterInfo.Params.FilterAtoms,
	}
	if r.FilterInfo.Params.FilterPriceFrom != nil {
		converted.Params.PriceFrom = &v1.Price{
			Value:    float64(*r.FilterInfo.Params.FilterPriceFrom),
			Currency: r.FilterInfo.PriceFilter.Currency,
		}
	}
	if r.FilterInfo.Params.FilterPriceTo != nil {
		converted.Params.PriceTo = &v1.Price{
			Value:    float64(*r.FilterInfo.Params.FilterPriceTo),
			Currency: r.FilterInfo.PriceFilter.Currency,
		}
	}

	converted.SearchPagePollingId = response.PollingSearchID

	return converted, nil
}

func filterEnabled(enabled bool, hint string) bool {
	return enabled && hint != "0" && hint != "+0"
}

func (s *Service) ConvertHotels(ctx context.Context, hotels []models.HotelWithOffers, imageParams *hotelsAPI.ImageParams) ([]*hotelsAPI.Hotel, error) {
	result := make([]*hotelsAPI.Hotel, len(hotels))

	imageSizeStr := s.config.ImageSizeDefault
	if imageParams != nil && imageParams.Size != hotelsAPI.ImageSize_IMAGE_SIZE_UNKNOWN {
		imageSizeStr = convertImageSize(imageParams.Size)
	}
	for i, h := range hotels {
		result[i] = ConvertHotelWithOffers(&h, &imageSizeStr, s.translateService.GetFunc(ctx))
	}
	return result, nil
}
func convertImageSize(imageSize hotelsAPI.ImageSize) string {
	s := int32(imageSize)
	if str, ok := hotelsAPI.ImageSize_name[int32(imageSize)]; ok && s != 0 {
		arr := strings.Split(str, "_")
		if len(arr) == 3 {
			return arr[2]
		}
	}
	return "XS"
}

func (s *Service) convertToHotelInfoRsp(ctx context.Context, response *models.GetHotelInfoResponse) (*hotelsAPI.GetHotelInfoRsp, error) {
	getTranslation := s.translateService.GetFunc(ctx)

	hotel := ConvertHotel(&response.Hotel, &s.config.ImageSizeDefault)
	// у отеля тут бейджи не заполнены, но есть у оффера
	hotel.IsPlusAvailable = isPlusAvailable(response.OffersInfo.DefaultOffer.Badges)

	converted := hotelsAPI.GetHotelInfoRsp{
		OffersInfo:       convertOffersInfo(&response.OffersInfo, getTranslation),
		Hotel:            hotel,
		SearchParams:     buildSearchParams(response.SearchParams),
		HotelDescription: convertHotelDescription(response.HotelDescription),
		ReviewsInfo:      ConvertReviewsRsp(response.ReviewsInfo, getTranslation, &s.config.Reviews),
		RatingsInfo:      convertRatingInfo(response.RatingsInfo),
	}

	return &converted, nil
}

func getSorter(sorter models.QuickSorter) hotelsAPI.Sort {
	switch sorter.ID {
	case "high-rating-first":
		return hotelsAPI.Sort_SORT_HIGH_RATING_FIRST
	case "cheap-first":
		return hotelsAPI.Sort_SORT_PRICE_ASC
	case "expensive-first":
		return hotelsAPI.Sort_SORT_PRICE_DESC
	case "relevant-first":
		return hotelsAPI.Sort_SORT_RELEVANT_FIRST
	}
	return hotelsAPI.Sort_SORT_UNKNOWN
}

func (s *Service) getShortNameSort(sorter hotelsAPI.Sort, getTranslation func(key TranslationKey) string) string {
	switch sorter {
	case hotelsAPI.Sort_SORT_HIGH_RATING_FIRST:
		return getTranslation(SortHighRatingFirstShortNameKey)
	case hotelsAPI.Sort_SORT_PRICE_ASC:
		return getTranslation(SortPriceAscShortNameKey)
	case hotelsAPI.Sort_SORT_PRICE_DESC:
		return getTranslation(SortPriceDescShortNameKey)
	case hotelsAPI.Sort_SORT_RELEVANT_FIRST:
		return getTranslation(SortRelevantFirstShortNameKey)
	}
	return getTranslation(SortHighRatingFirstShortNameKey)
}

func mapClientResponseToProto(response *models.SuggestResponse) *hotelsAPI.HotelsSuggestRsp {
	var result hotelsAPI.HotelsSuggestRsp
	for _, g := range response.Groups {
		group := hotelsAPI.SuggestGroup{
			Title: g.Name,
		}
		for _, item := range g.Items {
			var iType hotelsAPI.SuggestItemType
			var exists bool
			if iType, exists = typeMap[item.RedirectParams.Type]; !exists {
				continue
			}
			apiItem := &hotelsAPI.SuggestItem{
				Id:       item.ID,
				Type:     iType,
				Title:    item.Name,
				Subtitle: item.Description,
			}
			if item.RedirectParams != nil {
				var data hotelsAPI.SuggestItemData
				if item.RedirectParams.GeoID != nil {
					data.Region = &hotelsAPI.RegionSuggestData{GeoId: *item.RedirectParams.GeoID}
				}
				if item.RedirectParams.HotelSlug != "" || item.RedirectParams.Permalink != nil {
					data.Hotel = &hotelsAPI.HotelSuggestData{
						Slug: item.RedirectParams.HotelSlug,
					}
					if item.RedirectParams.Permalink != nil {
						data.Hotel.Permalink = *item.RedirectParams.Permalink
					}
				}
				data.SearchParams = buildSearchParams(item.RedirectParams.OfferSearchParams)

				apiItem.Data = &data
				group.Items = append(group.Items, apiItem)
			}
		}
		if len(group.Items) > 0 {
			result.Groups = append(result.Groups, &group)
		}
	}
	return &result
}

func buildSearchParams(params *models.OfferSearchParams) *hotelsAPI.OfferSearchParams {
	if params == nil {
		return nil
	}
	result := &hotelsAPI.OfferSearchParams{
		CheckinDate:  mapDate(params.CheckinDate),
		CheckoutDate: mapDate(params.CheckoutDate),
		Adults:       uint32(params.Adults),
		ChildrenAges: make([]uint32, len(params.ChildrenAges)),
	}
	for k, age := range params.ChildrenAges {
		result.ChildrenAges[k] = uint32(age)
	}
	return result
}

type TravelAPIClient interface {
	Suggest(ctx context.Context, query string, limit int, lang string, domain string, sessionID string, requestIndex int) (*models.SuggestResponse, error)
	LogSuggest(ctx context.Context, selectedID string, sessionID string, requestIndex int, isManualClick bool, isTrustedUser bool) error
	SearchHotels(ctx context.Context, request models.SearchHotelsRequest) (*models.SearchHotelsResponse, error)
	GetHotelInfo(ctx context.Context, request *models.GetHotelInfoRequest) (*models.GetHotelInfoResponse, error)
	GetHotelsCounters(ctx context.Context, request *models.GetCountersRequest) (*models.GetCountersResponse, error)
	GetHotelOffers(ctx context.Context, request *models.GetHotelOffersRequest) (*models.GetHotelOffersResponse, error)
	GetSimilarHotels(ctx context.Context, request *models.GetSimilarHotelsRequest) (*models.GetSimilarHotelsResponse, error)
	GetHotelImages(ctx context.Context, request *models.GetHotelImagesRequest) (*models.GetHotelImagesResponse, error)
}
