package hotels

// TODO(adurnev) вынести в общий пакет и подумать над использованием https://st.yandex-team.ru/TRAVELAPP-1921
import (
	"context"
	"fmt"

	"google.golang.org/grpc/codes"
	"google.golang.org/grpc/status"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/ctxlog"
	"a.yandex-team.ru/travel/app/backend/internal/common"
	"a.yandex-team.ru/travel/app/backend/internal/l10n"
)

type TranslationService struct {
	logger      log.Logger
	l10nService L10nService
}

type L10nService interface {
	Get(keysetName string, language string) (*l10n.Keyset, error)
}

type Translation interface {
	Get(ctx context.Context, key TranslationKey) string
	GetFunc(ctx context.Context) func(key TranslationKey) string
	GetKeysets() map[string][]string
}

func NewTranslationService(logger log.Logger, l10nService L10nService) *TranslationService {
	return &TranslationService{
		logger:      logger,
		l10nService: l10nService,
	}
}

func (t *TranslationService) Get(ctx context.Context, key TranslationKey) string {
	keysetName := key.GetKeyset()
	keyset, _ := t.getKeyset(ctx, keysetName)
	return keyset.Keys[key.GetKey()]
}

func (t *TranslationService) GetFunc(ctx context.Context) func(key TranslationKey) string {
	return func(key TranslationKey) string {
		return t.Get(ctx, key)
	}
}

//TODO(adurnev) подумать над reflection
func (t *TranslationService) GetKeysets() map[string][]string {
	return map[string][]string{
		HotelReviewKey.GetKeyset(""): {
			string(SortRelevantFirstTitleKey),
			string(SortRelevantFirstShortTitleKey),
			string(SortTimeDescTitleKey),
			string(SortTimeDescShortTitleKey),
			string(SortRatingAscTitleKey),
			string(SortRatingAscShortTitleKey),
			string(SortRatingDescTitleKey),
			string(SortRatingDescShortTitleKey),
			string(HotelReviewSortNameKey),
		},
		HotelKey.GetKeyset(""): {
			string(SortHighRatingFirstShortNameKey),
			string(SortPriceAscShortNameKey),
			string(SortPriceDescShortNameKey),
			string(SortRelevantFirstShortNameKey),
			string(RefundFullyRefundableNameKey),
			string(RefundRefundableWithPenaltyNameKey),
			string(RefundNonRefundableNameKey),
			string(RefundUnknownNameKey),
			string(PriceNameKey),
			string(PriceShortNameKey),
			string(HotelSortNameKey),
		},
	}
}

func (t *TranslationService) getKeyset(ctx context.Context, keySet string) (*l10n.Keyset, error) {
	lang := common.GetLanguage(ctx)
	keyset, err := t.l10nService.Get(keySet, lang)
	if err != nil {
		msg := fmt.Sprintf("error keySet for %s, %s", keySet, lang)
		ctxlog.Error(ctx, t.logger, msg, log.Error(err))
		return nil, status.Error(codes.Unknown, msg)
	}
	return keyset, nil
}

type TranslationKey interface {
	GetKeyset() string
	GetKey() string
}

// HotelReview

type HotelReviewKey string

func (HotelReviewKey) GetKeyset() string {
	return "backend_hotel_reviews"
}

func (h HotelReviewKey) GetKey() string {
	return string(h)
}

const (
	HotelReviewSortNameKey         HotelReviewKey = "sort_name"
	SortRelevantFirstTitleKey      HotelReviewKey = "sort_relevant_first_title"
	SortRelevantFirstShortTitleKey HotelReviewKey = "sort_relevant_first_short_title"
	SortTimeDescTitleKey           HotelReviewKey = "sort_time_desc_title"
	SortTimeDescShortTitleKey      HotelReviewKey = "sort_time_desc_short_title"
	SortRatingAscTitleKey          HotelReviewKey = "sort_rating_asc_title"
	SortRatingAscShortTitleKey     HotelReviewKey = "sort_rating_asc_short_title"
	SortRatingDescTitleKey         HotelReviewKey = "sort_rating_desc_title"
	SortRatingDescShortTitleKey    HotelReviewKey = "sort_rating_desc_short_title"
)

// Hotel

type HotelKey string

func (HotelKey) GetKeyset() string {
	return "backend_hotel"
}

func (h HotelKey) GetKey() string {
	return string(h)
}

const (
	HotelSortNameKey                HotelKey = "sort_name"
	SortHighRatingFirstShortNameKey HotelKey = "sort_high_rating_first_short_name"
	SortPriceAscShortNameKey        HotelKey = "sort_price_asc_short_name"
	SortPriceDescShortNameKey       HotelKey = "sort_price_desc_short_name"
	SortRelevantFirstShortNameKey   HotelKey = "sort_relevant_first_short_name"

	RefundFullyRefundableNameKey       HotelKey = "refund_fully_refundable_name"
	RefundRefundableWithPenaltyNameKey HotelKey = "refund_refundable_with_penalty_name"
	RefundNonRefundableNameKey         HotelKey = "refund_non_refundable_name"
	RefundUnknownNameKey               HotelKey = "refund_unknown_name"

	PriceNameKey      HotelKey = "price_name"
	PriceShortNameKey HotelKey = "price_short_name"
)
