package l10n

import (
	"encoding/json"
	"io/ioutil"
	"strconv"
	"sync"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/app/backend/internal/lib/tankerclient"
)

type localizedKeyset struct {
	name      string
	languages map[string]*Keyset
}

type Keyset struct {
	Name     string
	Tag      string
	Language string
	Keys     map[string]string
}

type Localizable interface {
	GetL10NKeysetName() string
	GetL10NKeys() []string
}

var UnsupportedLanguage = xerrors.NewSentinel("l10n: unsupported language")

type Service struct {
	logger             log.Logger
	localizedKeysets   map[string]*localizedKeyset
	mutex              sync.RWMutex
	cfg                Config
	supportedLanguages map[string]struct{}
	localizables       []Localizable
	keysets            map[string][]string
}

func CreateAndLoad(cfg Config, logger log.Logger) (*Service, error) {
	supportedLanguages := make(map[string]struct{}, len(cfg.SupportedLanguages))
	for _, l := range cfg.SupportedLanguages {
		supportedLanguages[l] = struct{}{}
	}
	s := &Service{
		logger:             logger,
		cfg:                cfg,
		localizables:       make([]Localizable, 0),
		keysets:            map[string][]string{},
		supportedLanguages: supportedLanguages,
	}
	if err := s.Reload(); err != nil {
		return nil, xerrors.Errorf("unable to load initial l10n data: %w", err)
	}
	return s, nil
}

func (s *Service) Reload() error {
	s.logger.Info("Will load keysets")
	data, err := ioutil.ReadFile(s.cfg.StoragePath)
	if err != nil {
		return xerrors.Errorf("unable to load l10n data: %w", err)
	}
	var exported tankerclient.ExportResponse
	if err := json.Unmarshal(data, &exported); err != nil {
		return xerrors.Errorf("unable to unmarshal l10n data: %w", err)
	}

	result := make(map[string]*localizedKeyset, len(exported.Keysets))
	for name, ks := range exported.Keysets {
		lk := localizedKeyset{
			languages: make(map[string]*Keyset, len(ks.Meta.Languages)),
			name:      name,
		}
		for _, lang := range ks.Meta.Languages {
			keys := make(map[string]string, len(ks.Keys))
			for _, key := range ks.Keys {
				keyName := key.Name
				if translation, found := key.Translations[lang]; found {
					if translation.Status == tankerclient.TranslationStatusApproved && translation.Payload.SingularForm != nil {
						keys[keyName] = *translation.Payload.SingularForm
					}
				}
			}
			if len(keys) > 0 {
				lk.languages[lang] = &Keyset{
					Name:     name,
					Tag:      strconv.FormatInt(ks.CommitID, 10),
					Language: lang,
					Keys:     keys,
				}
			}
		}
		result[name] = &lk
	}
	for _, l := range s.localizables {
		if ok := s.hasKeysInNewData(l.GetL10NKeysetName(), l.GetL10NKeys(), result); !ok {
			return xerrors.Errorf("some required keys are missing in keyset %s", l.GetL10NKeysetName())
		}
	}
	for keyset, keys := range s.keysets {
		if ok := s.hasKeysInNewData(keyset, keys, result); !ok {
			return xerrors.Errorf("some required keys are missing in keyset %s", keyset)
		}
	}
	s.mutex.Lock()
	defer s.mutex.Unlock()
	s.localizedKeysets = result
	s.logger.Info("Keysets updated")
	return nil
}

func (s *Service) hasKeys(keysetName string, keys []string) bool {
	for lang := range s.supportedLanguages {
		keyset, err := s.Get(keysetName, lang)
		if err != nil {
			return false
		}
		for _, k := range keys {
			_, ok := keyset.Keys[k]
			if !ok {
				return false
			}
		}
	}
	return true
}

func (s *Service) hasKeysInNewData(keysetName string, keys []string, newData map[string]*localizedKeyset) bool {
	keysets, ok := newData[keysetName]
	if !ok {
		return false
	}
	for lang := range s.supportedLanguages {
		keyset, ok := keysets.languages[lang]
		if !ok {
			return false
		}
		for _, k := range keys {
			_, ok := keyset.Keys[k]
			if !ok {
				return false
			}
		}
	}
	return true
}

func (s *Service) ID() string {
	return "l10n"
}

func (s *Service) Get(keyset string, language string) (*Keyset, error) {
	if _, ok := s.supportedLanguages[language]; !ok {
		return nil, UnsupportedLanguage
	}
	s.mutex.RLock()
	defer s.mutex.RUnlock()
	if ks, found := s.localizedKeysets[keyset]; found {
		if lks, langFound := ks.languages[language]; langFound {
			return lks, nil
		} else {
			return nil, xerrors.Errorf("localization '%s' is not defined for keyset '%s'", language, keyset)
		}
	} else {
		return nil, xerrors.Errorf("keyset '%s' not found", keyset)
	}
}

func (s *Service) Register(l Localizable) error {
	if ok := s.hasKeys(l.GetL10NKeysetName(), l.GetL10NKeys()); !ok {
		return xerrors.Errorf("l10n error registering localizable")
	}
	s.localizables = append(s.localizables, l)
	return nil
}

func (s *Service) RegisterKeyset(keyset string, keys []string) error {
	if ok := s.hasKeys(keyset, keys); !ok {
		return xerrors.Errorf("l10n error registering keyset")
	}
	s.keysets[keyset] = keys
	return nil
}
