package aviabackendclient

import (
	"context"
	"io"
	"net/url"

	"github.com/go-resty/resty/v2"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/ctxlog"
	"a.yandex-team.ru/library/go/core/metrics"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/app/backend/internal/lib/clientscommon"
)

type HTTPClient struct {
	logger     log.Logger
	config     Config
	httpClient *resty.Client
	metrics    *clientscommon.HTTPClientMetrics
}

func NewHTTPClient(
	logger log.Logger,
	config Config,
	logRequestAndResponse bool,
	metricsRegistry metrics.Registry,
) *HTTPClient {
	client := resty.New().SetTimeout(config.Timeout).SetLogger(logger).OnRequestLog(clientscommon.DoNotLogTVMHeaders)
	if logRequestAndResponse {
		client.Debug = true // Влияет только на логирование запроса и ответа
	}
	return &HTTPClient{
		logger:     logger,
		config:     config,
		httpClient: client,
		metrics:    clientscommon.NewHTTPClientMetrics(metricsRegistry, "avia-backend"),
	}
}

func (c *HTTPClient) execute(ctx context.Context, method, path string, body, result interface{}, queryParams url.Values) error {
	endpoint := c.config.BaseURL + path

	var errResponse map[string]interface{}
	r := c.httpClient.R().
		SetContext(ctx).
		SetBody(body).
		SetError(&errResponse)
	if result != nil {
		r = r.SetResult(result)
	}
	if queryParams != nil {
		r = r.SetQueryParamsFromValues(queryParams)
	}
	response, err := r.Execute(method, endpoint)
	c.metrics.StoreCallResult(method, path, response)
	if err != nil {
		return ResponseError.Wrap(err)
	}
	if !response.IsSuccess() {
		raw, _ := io.ReadAll(response.RawResponse.Body)
		return xerrors.Errorf("unexpected response from avia-backend service: %w", StatusError{
			Status:      response.StatusCode(),
			Response:    errResponse,
			ResponseRaw: string(raw),
		})
	}
	return nil
}

func (c *HTTPClient) GeoLookup(ctx context.Context, nationalVersion, lang string, geoID int32) (*GeoLookupRsp, error) {
	// Пример запроса:
	// curl "http://backend.production.avia.yandex.net/?national_version=ru&lang=ru&name=geoLookup" \
	//     -X POST -d '[{"name":"geoLookup","params":{"geoId":"10725"},"fields":[{"name":"searchCity","fields":["id","key","title","geoId","code","iataCode","sirenaCode"]}]}]' \
	//     -H "Accept: application/json" \
	//     -H "Content-Type: application/json"
	// Ответ:
	// {"status":"success","data":[{"searchCity":{"title":"Москва","geoId":213,"key":"c213","id":213,"code":"MOW","iataCode":"MOW","sirenaCode":"МОВ"}}]}
	queryParams := url.Values{
		"name":             {"geoLookup"},
		"national_version": {nationalVersion},
		"lang":             {lang},
	}
	path := "/"
	var rsp GeoLookupRsp
	if err := c.execute(ctx, resty.MethodPost, path, BuildGeoLookupReq(geoID), &rsp, queryParams); err != nil {
		ctxlog.Debug(ctx, c.logger, "GeoLookup error", log.Error(err))
		return nil, err
	}
	ctxlog.Debug(ctx, c.logger, "GeoLookup success")
	return &rsp, nil
}

func (c *HTTPClient) TopFlights(ctx context.Context, nationalVersion, lang string, fromKey, toKey, date string, limit int) (*TopFlightsRsp, error) {
	path := "/"
	var rsp TopFlightsRsp

	queryParams := url.Values{
		"name":             {"topFlights"},
		"national_version": {nationalVersion},
		"lang":             {lang},
	}

	if err := c.execute(ctx, resty.MethodPost, path, BuildTopFlightsReq(fromKey, toKey, date, limit), &rsp, queryParams); err != nil {
		ctxlog.Debug(ctx, c.logger, "TopFlights error", log.Error(err))
		return nil, err
	}
	ctxlog.Debug(ctx, c.logger, "TopFlights success")
	return &rsp, nil
}
