package aviasuggestclient

import (
	"context"
	"net/url"

	"github.com/go-resty/resty/v2"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/ctxlog"
	"a.yandex-team.ru/library/go/core/metrics"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/app/backend/internal/lib/clientscommon"
)

type FieldType string

const (
	FieldTo   = "to"
	FieldFrom = "from"
)

type HTTPClient struct {
	logger     log.Logger
	config     Config
	httpClient *resty.Client
	metrics    *clientscommon.HTTPClientMetrics
}

func NewHTTPClient(
	logger log.Logger,
	config Config,
	logRequestAndResponse bool,
	metricsRegistry metrics.Registry,
) *HTTPClient {
	client := resty.New().SetTimeout(config.Timeout).SetLogger(logger).OnRequestLog(clientscommon.DoNotLogTVMHeaders)
	if logRequestAndResponse {
		client.Debug = true // Влияет только на логирование запроса и ответа
	}
	return &HTTPClient{
		logger:     logger,
		config:     config,
		httpClient: client,
		metrics:    clientscommon.NewHTTPClientMetrics(metricsRegistry, "avia-suggest"),
	}
}

func (c *HTTPClient) execute(ctx context.Context, path string, result interface{}, queryParams url.Values) error {
	endpoint := c.config.BaseURL + path

	var errResponse map[string]interface{}
	r := c.httpClient.R().
		SetContext(ctx).
		SetError(&errResponse)
	if result != nil {
		r = r.SetResult(result)
	}
	if queryParams != nil {
		r = r.SetQueryParamsFromValues(queryParams)
	}
	response, err := r.Execute(resty.MethodGet, endpoint)
	c.metrics.StoreCallResult(resty.MethodGet, path, response)
	if err != nil {
		return clientscommon.ResponseError.Wrap(err)
	}
	if !response.IsSuccess() {
		raw := response.Body()
		return xerrors.Errorf("unexpected response from avia-ticket-daemon service: %w", clientscommon.StatusError{
			Status:      response.StatusCode(),
			Response:    errResponse,
			ResponseRaw: string(raw),
		})
	}
	return nil
}

func (c *HTTPClient) Suggest(
	ctx context.Context,
	nationalVersion string,
	lang string,
	field FieldType,
	query string,
	otherQuery string,
	otherPoint string,
) (*SuggestResponse, error) {
	queryParams := url.Values{
		"national_version":   {nationalVersion},
		"lang":               {lang},
		"field":              {string(field)},
		"query":              {query},
		"other_query":        {otherQuery},
		"other_point":        {otherPoint},
		"have_airport_field": {"true"},
		"hidden_field":       {"true"},
		"need_country":       {"true"},
		// Этот параметр запроса именно camelCase
		"includePrices": {"true"},
	}
	path := "/v2/avia"
	var rawRsp SuggestResponseRaw
	if err := c.execute(ctx, path, &rawRsp, queryParams); err != nil {
		ctxlog.Debug(ctx, c.logger, "avia suggests error", log.Error(err))
		return nil, err
	}
	ctxlog.Debug(ctx, c.logger, "avia suggests ok")
	rsp, err := toSuggestResponse(rawRsp)
	if err != nil {
		return nil, xerrors.Errorf("unknown suggest response format: %w", err)
	}
	return rsp, nil
}
