package aviasuggestclient

import (
	"encoding/json"
	"reflect"
	"strings"

	"a.yandex-team.ru/library/go/core/xerrors"
)

type SuggestResponseRaw []interface{}

type SuggestResponse struct {
	Query    string
	Suggests []Suggest
}

type Bool struct {
	bool
}

func NewBool(value bool) Bool {
	return Bool{value}
}

func (b *Bool) GetValue() bool {
	return b.bool
}

type Suggest struct {
	Level                int       `json:"-"`
	Title                string    `json:"-"`
	PointKey             string    `json:"point_key"`
	PointCode            string    `json:"point_code"`
	RegionTitle          string    `json:"region_title"`
	CityTitle            string    `json:"city_title"`
	CountryTitle         string    `json:"country_title"`
	Missprint            int       `json:"missprint"`
	Hidden               Bool      `json:"hidden"`
	HaveAirport          Bool      `json:"have_airport"`
	HaveNotHiddenAirport Bool      `json:"have_not_hidden_airport"`
	Nested               []Suggest `json:"-"`
}

func (b *Bool) UnmarshalJSON(bt []byte) (err error) {
	s := strings.Trim(string(bt), "\"")
	if s == "1" {
		b.bool = true
		return
	} else if s == "0" {
		b.bool = false
		return
	}
	return xerrors.Errorf("unknown Bool value %s", s)
}

func toSuggestResponse(raw SuggestResponseRaw) (*SuggestResponse, error) {
	if len(raw) < 2 {
		return nil, xerrors.Errorf("invalid suggest response structure: not enough elements in top array")
	}

	result := SuggestResponse{}
	switch v := raw[0].(type) {
	case string:
		result.Query = v
	default:
		return nil, xerrors.Errorf("invalid suggest response structure: 0 element of top array invalid type '%s'", getType(v))
	}

	var rawSuggests []interface{} = nil
	switch v := raw[1].(type) {
	case []interface{}:
		rawSuggests = v
	default:
		return nil, xerrors.Errorf("invalid suggest response structure: 1 element of top array invalid type '%s'", getType(v))
	}

	suggests, err := toSuggests(rawSuggests)
	if err != nil {
		return nil, xerrors.Errorf("toSuggests error: %w", err)
	}
	result.Suggests = suggests

	return &result, nil
}

func toSuggests(raw []interface{}) ([]Suggest, error) {
	suggests := make([]Suggest, len(raw))
	for i, rawSuggest := range raw {
		suggest, err := toSuggest(rawSuggest)
		if err != nil {
			return nil, xerrors.Errorf("toSuggests error: %w", err)
		}
		suggests[i] = *suggest
	}
	return suggests, nil
}

func toSuggest(rawSuggest interface{}) (*Suggest, error) {
	var raw []interface{} = nil
	switch v := rawSuggest.(type) {
	case []interface{}:
		raw = v
	default:
		return nil, xerrors.Errorf("invalid suggest response structure: suggest must be array, got '%s'", getType(v))
	}

	if len(raw) < 4 {
		return nil, xerrors.Errorf("invalid suggest response structure: suggest array length must be greater than 4, got %d", len(raw))
	}

	result := Suggest{}

	switch v := raw[0].(type) {
	case float64:
		result.Level = int(v)
	default:
		return nil, xerrors.Errorf("invalid suggest response structure: 0 element of suggest array must be int, got '%s'", getType(v))
	}

	switch v := raw[1].(type) {
	case string:
		result.Title = v
	default:
		return nil, xerrors.Errorf("invalid suggest response structure: 1 element of suggest array must be string, got '%s'", getType(v))
	}

	switch v := raw[2].(type) {
	case map[string]interface{}:
		b, err := json.Marshal(v)
		if err != nil {
			return nil, xerrors.Errorf("invalid suggest response structure: %w", err)
		}
		err = json.Unmarshal(b, &result)
		if err != nil {
			return nil, xerrors.Errorf("invalid suggest response structure: %w", err)
		}
	default:
		return nil, xerrors.Errorf("invalid suggest response structure: 2 element of suggest array must be map, got %s", getType(v))
	}

	switch v := raw[3].(type) {
	case []interface{}:
		nested, err := toSuggests(v)
		if err != nil {
			return nil, xerrors.Errorf("invalid suggest response structure: %w", err)
		}
		result.Nested = nested
	default:
		return nil, xerrors.Errorf("invalid suggest response structure: 3 element of suggest array must be array, got '%s'", getType(v))
	}

	return &result, nil
}

func getType(v interface{}) string {
	return reflect.TypeOf(v).Name()
}
