package aviatdapiclient

import (
	"context"
	"fmt"
	"net/url"
	"strconv"

	"github.com/go-resty/resty/v2"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/ctxlog"
	"a.yandex-team.ru/library/go/core/metrics"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/app/backend/internal/common"
	"a.yandex-team.ru/travel/app/backend/internal/lib/clientscommon"
	exp3pb "a.yandex-team.ru/travel/app/backend/internal/lib/exp3matcher/proto/v1"
)

type RestyClientBuilder func(client *HTTPClient) *resty.Client

type RestyClientModifier func(*resty.Client) *resty.Client

type HTTPClient struct {
	logger                log.Logger
	config                Config
	logRequestAndResponse bool
	restyClientBuilder    RestyClientBuilder // Нужен для тестов, чтобы заменить для возможности мокать через httpmock
	metrics               *clientscommon.HTTPClientMetrics
}

func NewHTTPClient(
	logger log.Logger,
	config Config,
	logRequestAndResponse bool,
	metricsRegistry metrics.Registry,
) *HTTPClient {
	client := &HTTPClient{
		logger:                logger,
		config:                config,
		logRequestAndResponse: logRequestAndResponse,
		restyClientBuilder:    defaultRestyClientBuilder,
		metrics:               clientscommon.NewHTTPClientMetrics(metricsRegistry, "avia-tdapi"),
	}
	return client
}

func (c *HTTPClient) execute(
	ctx context.Context,
	method string,
	path string,
	result interface{},
	queryParams url.Values,
	clientModifier RestyClientModifier,
) error {
	endpoint := c.config.BaseURL + path

	var errResponse map[string]interface{}
	client := c.restyClientBuilder(c)
	if clientModifier != nil {
		client = clientModifier(client)
	}
	r := client.R().
		SetContext(ctx).
		SetError(&errResponse)
	if result != nil {
		r = r.SetResult(result)
	}
	if queryParams != nil {
		r = r.SetQueryParamsFromValues(queryParams)
	}
	response, err := r.Execute(method, endpoint)
	c.metrics.StoreCallResult(method, path, response)
	if err != nil {
		return clientscommon.ResponseError.Wrap(err)
	}
	if !response.IsSuccess() {
		raw := response.Body()
		return xerrors.Errorf("unexpected response from avia-ticket-daemon service: %w", clientscommon.StatusError{
			Status:      response.StatusCode(),
			Response:    errResponse,
			ResponseRaw: string(raw),
		})
	}
	return nil
}

func (c *HTTPClient) InitSearch(
	ctx context.Context,
	nationalVersion string,
	lang string,
	adults uint32,
	children uint32,
	infants uint32,
	dateForward Date,
	dateBackward *Date,
	serviceClass ServiceClass,
	pointFrom string,
	pointTo string,
) (*InitSearchRsp, error) {
	queryParams := url.Values{
		"national":     {nationalVersion},
		"lang":         {lang},
		"service":      {common.ServiceName},
		"adults":       {strconv.Itoa(int(adults))},
		"children":     {strconv.Itoa(int(children))},
		"infants":      {strconv.Itoa(int(infants))},
		"date_forward": {common.FormatDate(dateForward.Time)},
		"klass":        {string(serviceClass)},
		"point_from":   {pointFrom},
		"point_to":     {pointTo},
	}
	if dateBackward != nil {
		queryParams.Add("date_backward", common.FormatDate(dateBackward.Time))
	}
	path := "/jsendapi/init_search/"
	var rsp InitSearchRsp
	retryCondition := func(rsp *resty.Response, err error) bool {
		if rsp != nil && rsp.StatusCode() >= 499 {
			return true
		}
		return false
	}
	clientModifier := func(rc *resty.Client) *resty.Client {
		return rc.SetRetryCount(c.config.InitRetryCount).SetRetryWaitTime(c.config.InitRetryWait).AddRetryCondition(retryCondition)
	}
	if err := c.execute(ctx, resty.MethodGet, path, &rsp, queryParams, clientModifier); err != nil {
		ctxlog.Debug(ctx, c.logger, "InitSearch: error", log.Error(err))
		return nil, err
	}
	ctxlog.Debug(ctx, c.logger, "InitSearch: ok")
	return &rsp, nil
}

func (c *HTTPClient) SearchResult(ctx context.Context, qid string, exp3tdAPIConfig *exp3pb.TdApiConfig) (*SearchResultRsp, error) {
	ignoreOutdated := "true"
	if exp3tdAPIConfig.InstantSearchEnabled {
		ignoreOutdated = "false"
	}

	queryParams := url.Values{
		"service":        {common.ServiceName},
		"ignoreOutdated": {ignoreOutdated},
	}

	path := fmt.Sprintf("/jsendapi/v3/results/%s/0/0/", url.PathEscape(qid))
	var rsp SearchResultRsp
	if err := c.execute(ctx, resty.MethodGet, path, &rsp, queryParams, nil); err != nil {
		ctxlog.Debug(ctx, c.logger, "SearchResult: error", log.Error(err))
		return nil, err
	}
	ctxlog.Debug(ctx, c.logger, "SearchResult: ok")
	return &rsp, nil
}

func defaultRestyClientBuilder(c *HTTPClient) *resty.Client {
	client := resty.New().SetTimeout(c.config.Timeout).SetLogger(c.logger).OnRequestLog(clientscommon.DoNotLogTVMHeaders)
	if c.logRequestAndResponse {
		client.Debug = true // Влияет только на логирование запроса и ответа
	}
	return client
}
