package aviatdapiclient

import (
	"fmt"
	"time"

	"a.yandex-team.ru/travel/app/backend/internal/lib/clientscommon"
)

type Date struct {
	time.Time
}

type ServiceClass string

const (
	EconomyServiceClass  ServiceClass = "economy"
	BusinessServiceClass ServiceClass = "business"
)

type ResponseStatus string

const (
	ErrorResponseStatus   ResponseStatus = "error"
	SuccessResponseStatus ResponseStatus = "success"
)

type TransportType string

const (
	TrainTransportType TransportType = "train"
	BusTransportType   TransportType = "bus"
	PlaneTransportType TransportType = "plane"
)

type InitSearchRsp struct {
	Status ResponseStatus `json:"status"`
	Data   InitSearchData `json:"data,omitempty"`
}

type InitSearchData struct {
	QID string `json:"qid"`
}

type SearchResultRsp struct {
	Status ResponseStatus   `json:"status"`
	Data   SearchResultData `json:"data,omitempty"`
}

type SearchResultData struct {
	Variants  SearchResultVariants  `json:"variants"`
	Reference SearchResultReference `json:"reference"`
	Progress  SearchResultProgress  `json:"progress"`
}

type SearchResultVariants struct {
	Fares []SearchResultFare `json:"fares"`
}

type SearchResultFare struct {
	Prices []SearchResultPrice `json:"prices"`
	Route  [][]string          `json:"route"`
}

type SearchResultPrice struct {
	FareFamiliesHash string       `json:"fare_families_hash"`
	Charter          bool         `json:"charter"`
	FareCodes        [][][]string `json:"fare_codes"`
	Baggage          [][]string   `json:"baggage"`
	Selfconnect      bool         `json:"selfconnect"`
	PartnerCode      string       `json:"partnerCode"`
	Tariff           Price        `json:"tariff"`
	TariffNational   Price        `json:"tariffNational"`
	FareFamilies     [][]string   `json:"fare_families"`
	FromCompany      bool         `json:"fromCompany"`
	Boy              bool         `json:"boy"`
}

type Price struct {
	Currency string  `json:"currency"`
	Value    float64 `json:"value"`
}

func (p *Price) GetCorrectedCurrency() string {
	return *clientscommon.GetCorrectedCurrency(&p.Currency)
}

type SearchResultReference struct {
	Partners       []Partner                `json:"partners"`
	FareFamilies   map[string]FareFamily    `json:"fareFamilies"`
	Settlements    []Settlement             `json:"settlements"`
	Alliances      []Alliance               `json:"alliances"`
	CompanyTariffs []CompanyTariff          `json:"companyTariffs"`
	AviaCompanies  []AviaCompany            `json:"aviaCompanies"`
	Stations       []Station                `json:"stations"`
	Companies      []Company                `json:"companies"`
	Flights        []Flight                 `json:"flights"`
	BaggageTariffs map[string]BaggageTariff `json:"baggageTariffs"`
}

type Partner struct {
	ID      uint64 `json:"id"`
	Code    string `json:"code"`
	Title   string `json:"title"`
	LogoSVG string `json:"logoSvg"`
	LogoPNG string `json:"logoPng"`
}

// Структуру fare-families можно посмотреть здесь https://nda.ya.ru/t/K3TsyySo4oNjXa
// Отличие в том, что здесь парсинг ответа TD-API, а по ссылке исходные данные.
type FareFamily struct {
	BaseClass       string            `json:"base_class"`
	Terms           []FFTerm          `json:"terms"`
	Brand           string            `json:"brand"`
	TariffGroupName map[string]string `json:"tariff_group_name"` // Переводы - ru, en
	Key             string            `json:"key"`
}

type FFTerm struct {
	Code string                 `json:"code"`
	Rule map[string]interface{} `json:"rule"`
}

const (
	FFTermCodeRefundable string = "refundable"
	FFTermCodeCarryOn    string = "carry_on"
	FFTermCodeBaggage    string = "baggage"
)

type FFTermRuleBaggage struct {
	Places *int `json:"places"`
	Weight *int `json:"weight"`
}
type FFTermRuleRefundable struct {
	Charge       *FFTermRuleRefundableCharge `json:"charge"`
	Availability AvailabilityType            `json:"availability"`
}
type FFTermRuleRefundableCharge struct {
	Currency string `json:"currency"`
	Value    string `json:"value"`
}
type FFTermRuleCarryOn struct {
	Places *int `json:"places"`
	Weight *int `json:"weight"`
}

type AvailabilityType string

const (
	AvailabilityTypeUnknown      AvailabilityType = ""
	AvailabilityTypeNotAvailable AvailabilityType = "NOT_AVAILABLE"
	AvailabilityTypeCharge       AvailabilityType = "CHARGE"
	AvailabilityTypeFree         AvailabilityType = "FREE"
)

type Settlement struct {
	ID              uint64 `json:"id"`
	AviaCode        string `json:"code"`
	Title           string `json:"title"`
	TitleGenitive   string `json:"phraseFrom"`
	TitleLocative   string `json:"phraseIn"`
	TitleAccusative string `json:"phraseTo"`
	Preposition     string `json:"preposition"`
}

type Station struct {
	ID              uint64        `json:"id"`
	SettlementID    *uint64       `json:"settlement,omitempty"`
	AviaCode        string        `json:"code"`
	Title           string        `json:"title"`
	TransportType   TransportType `json:"tType"`
	Preposition     string        `json:"preposition"`
	TitleGenitive   string        `json:"phraseFrom"`
	TitleLocative   string        `json:"phraseIn"`
	TitleAccusative string        `json:"phraseTo"`
}

type Alliance struct {
	ID    uint64 `json:"id"`
	Title string `json:"title"`
}

type CompanyTariff struct {
	ID             uint64  `json:"id"`
	Published      bool    `json:"published"`
	CarryOn        bool    `json:"carryon"`
	CarryOnNorm    float64 `json:"carryonNorm"`
	BaggageAllowed bool    `json:"baggageAllowed"`
	BaggageNorm    float64 `json:"baggageNorm"`
}

type AviaCompany struct {
	ID                   uint64 `json:"id"`
	CarryOnLength        int    `json:"carryonLength,omitempty"`
	CarryOnWidth         int    `json:"carryonWidth,omitempty"`
	CarryOnHeight        int    `json:"carryonHeight,omitempty"`
	CostType             string `json:"costType"`
	CarryOnSizeBucket    string `json:"carryonSizeBucket"`
	BaggageRules         string `json:"baggageRules"`
	BaggageDimensionsSum int    `json:"baggageDimensionsSum,omitempty"`
}

type Company struct {
	ID         uint64  `json:"id"`
	AllianceID *uint64 `json:"alliance,omitempty"`
	Title      string  `json:"title"`
	LogoSVG    string  `json:"logoSvg"`
	LogoPNG    string  `json:"logoPng"`
	Color      string  `json:"color"`
}

type Flight struct {
	Key                  string               `json:"key"`
	Number               string               `json:"number"`
	Departure            DateTime             `json:"departure"`
	Arrival              DateTime             `json:"arrival"`
	CompanyID            uint64               `json:"company"`
	OperatingAviaCompony OperatingAviaCompany `json:"operating,omitempty"`
	CompanyTariffID      uint64               `json:"companyTariff"`
	AviaCompanyID        *uint64              `json:"aviaCompany"`
	StationFromID        uint64               `json:"from"`
	StationToID          uint64               `json:"to"`
}

type OperatingAviaCompany struct {
	Company uint64 `json:"company"`
	Number  string `json:"number"`
}

type DateTime struct {
	Local    string `json:"local"`
	Timezone string `json:"tzname"`
	Offset   int64  `json:"offset"`
}

func (d *DateTime) ISO8601() string {
	sign := "+"
	offset := d.Offset
	if d.Offset < 0 {
		sign = "-"
		offset = -offset
	}
	hours := offset / 60
	minutes := offset % 60
	return fmt.Sprintf("%s%s%.2d:%.2d", d.Local, sign, hours, minutes)
}

type BaggageTariff struct {
	Included BaggageTariffDetails `json:"included"`
	Pieces   BaggageTariffDetails `json:"pc"`
	Weight   BaggageTariffDetails `json:"wt"`
}

type BaggageTariffDetails struct {
	Count  int    `json:"count"`
	Source string `json:"source"`
}

type SearchResultProgress struct {
	Current uint32 `json:"current"`
	Total   uint32 `json:"all"`
}
