package calendarclient

import (
	"context"
	"net/url"

	"github.com/go-resty/resty/v2"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/ctxlog"
	"a.yandex-team.ru/library/go/core/metrics"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/app/backend/internal/common"
	"a.yandex-team.ru/travel/app/backend/internal/lib/clientscommon"
)

const ServiceName = "calendar"

type HTTPClient struct {
	logger          log.Logger
	config          Config
	httpClient      *resty.Client
	headerProviders []common.HeaderProvider
	metrics         *clientscommon.HTTPClientMetrics
}

func NewHTTPClient(logger log.Logger, config Config, metricsRegistry metrics.Registry, headerProviders ...common.HeaderProvider) *HTTPClient {
	return &HTTPClient{
		logger:          logger,
		config:          config,
		httpClient:      resty.New().SetTimeout(config.Timeout),
		headerProviders: headerProviders,
		metrics:         clientscommon.NewHTTPClientMetrics(metricsRegistry, "calendar-api"),
	}
}

func (c *HTTPClient) execute(ctx context.Context, method, path string, result interface{}, queryParams url.Values) error {
	endpoint := c.config.BaseURL + path

	var errResponse map[string]interface{}
	r := c.httpClient.R().
		SetContext(ctx).
		SetError(&errResponse)

	for _, provider := range c.headerProviders {
		headers, err := provider(ctx)
		if err != nil {
			return err
		}
		for k, v := range headers {
			if v != "" {
				r.SetHeader(k, v)
			}
		}
	}

	if result != nil {
		r = r.SetResult(result)
	}
	if queryParams != nil {
		r = r.SetQueryParamsFromValues(queryParams)
	}
	response, err := r.Execute(method, endpoint)
	c.metrics.StoreCallResult(method, path, response)
	if err != nil {
		return clientscommon.ResponseError.Wrap(err)
	}
	if !response.IsSuccess() {
		raw := response.Body()
		return xerrors.Errorf("unexpected response from calendar service: %w", clientscommon.StatusError{
			Status:      response.StatusCode(),
			Response:    errResponse,
			ResponseRaw: string(raw),
		})
	}
	return nil
}

func (c *HTTPClient) GetHolidays(ctx context.Context, dateFrom Date, dateTo Date) (*HolidaysRsp, error) {
	queryParams := url.Values{
		"from":    {common.FormatDate(dateFrom.Time)},
		"to":      {common.FormatDate(dateTo.Time)},
		"for":     {c.config.GeoBaseID},
		"outMode": {"all"},
	}
	path := "/internal/get-holidays"
	var rsp HolidaysRsp
	if err := c.execute(ctx, resty.MethodGet, path, &rsp, queryParams); err != nil {
		ctxlog.Debug(ctx, c.logger, "GetDays: error", log.Error(err))
		return nil, err
	}
	ctxlog.Debug(ctx, c.logger, "GetDays: ok")
	return &rsp, nil
}
