package clientscommon

import (
	"context"
	"fmt"
	"net"
	"net/http"

	"google.golang.org/grpc/codes"
	"google.golang.org/grpc/status"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/ctxlog"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/app/backend/internal/common"
)

var (
	ResponseError      = xerrors.NewSentinel("response error")
	PreconditionError  = xerrors.NewSentinel("precondition error")
	ParseResponseError = xerrors.NewSentinel("parse response error")
)

type StatusError struct {
	Status        int
	Response      map[string]interface{}
	ResponseRaw   string
	InternalError error
}

func (e StatusError) Error() string {
	return fmt.Sprintf("unexpected http status: %d, response: %s", e.Status, e.ResponseRaw)
}

func ConvertHTTPErrorToGRPCError(ctx context.Context, logger log.Logger, err error, serviceTitle string) error {
	var httpErr StatusError

	if xerrors.As(err, &httpErr) {
		// Не все ошибки должны попадать в ErrorBooster/Sentry
		if httpErr.Status >= 500 {
			ctxlog.Error(ctx, logger, serviceTitle+" got http error", log.Int("status", httpErr.Status), log.Any("response", httpErr.Response), log.Error(httpErr.InternalError))
		} else {
			ctxlog.Warn(ctx, logger, serviceTitle+" got http error", log.Int("status", httpErr.Status), log.Any("response", httpErr.Response), log.Error(httpErr.InternalError))
		}
		switch {
		case httpErr.Status == http.StatusBadRequest:
			return status.Error(codes.InvalidArgument, serviceTitle+" bad request: "+httpErr.ResponseRaw)
		case httpErr.Status == http.StatusUnauthorized || httpErr.Status == http.StatusForbidden:
			return status.Error(codes.Unauthenticated, serviceTitle+" unauthenticated")
		case httpErr.Status == http.StatusNotFound:
			return status.Error(codes.NotFound, serviceTitle+" not found")
		case httpErr.Status == http.StatusGatewayTimeout || httpErr.Status == http.StatusRequestTimeout:
			return status.Error(codes.DeadlineExceeded, serviceTitle+" timeout error")
		case httpErr.Status == http.StatusServiceUnavailable:
			return status.Error(codes.Unavailable, serviceTitle+" unavailable")
		case httpErr.Status >= 500:
			return status.Error(codes.Unknown, serviceTitle+" invalid response")
		default:
			return status.Error(codes.Unknown, serviceTitle+" unexpected error")
		}
	} else {
		ctxlog.Error(ctx, logger, "error:", log.Error(err))
		switch {
		case xerrors.Is(err, common.NoTvmServiceTicketError):
			return status.Error(codes.Unauthenticated, serviceTitle+" no service ticket error")
		case xerrors.Is(err, PreconditionError):
			return status.Error(codes.FailedPrecondition, serviceTitle+" precondition error")
		case xerrors.Is(err, ResponseError):
			unwrapped := xerrors.Unwrap(err)
			switch typed := unwrapped.(type) {
			case net.Error:
				if typed.Timeout() {
					return status.Error(codes.DeadlineExceeded, serviceTitle+" timeout error")
				} else {
					return status.Error(codes.Unknown, serviceTitle+" can not get response")
				}
			default:
				return status.Error(codes.Unknown, serviceTitle+" can not get response")
			}

		case xerrors.Is(err, ParseResponseError):
			return status.Error(codes.Unknown, serviceTitle+" parse response error")
		default:
			return status.Error(codes.Unknown, serviceTitle+" invalid response")
		}
	}
}
