package clientscommon

import (
	"context"
	"sync"

	"github.com/go-resty/resty/v2"
	"google.golang.org/grpc"
	"google.golang.org/grpc/codes"
	"google.golang.org/grpc/status"

	"a.yandex-team.ru/library/go/core/metrics"
	"a.yandex-team.ru/library/go/core/metrics/solomon"
)

var (
	// GRPCUserErrorCodes can covert to http 4xx
	// See: https://github.com/googleapis/googleapis/blob/master/google/rpc/code.proto
	GRPCUserErrorCodes = map[codes.Code]bool{
		codes.OutOfRange:         true,
		codes.Aborted:            true,
		codes.FailedPrecondition: true,
		codes.ResourceExhausted:  true,
		codes.Unauthenticated:    true,
		codes.PermissionDenied:   true,
		codes.AlreadyExists:      true,
		codes.NotFound:           true,
		codes.InvalidArgument:    true,
		codes.Canceled:           true,
	}
)

type HTTPClientMetrics struct {
	r       metrics.Registry
	methods sync.Map
}

type httpMethodMetrics struct {
	registerOnce    sync.Once
	Count2xx        metrics.Counter
	Count4xx        metrics.Counter
	Count5xx        metrics.Counter
	CountOtherError metrics.Counter
}

func (mMetrics *httpMethodMetrics) register(r metrics.Registry, method, path string) {
	mMetrics.registerOnce.Do(func() {
		r = r.WithTags(map[string]string{"method": method, "path": path})
		mMetrics.Count2xx = r.WithTags(map[string]string{"status": "2XX"}).Counter("count.rate")
		solomon.Rated(mMetrics.Count2xx)
		mMetrics.Count4xx = r.WithTags(map[string]string{"status": "4XX"}).Counter("count.rate")
		solomon.Rated(mMetrics.Count4xx)
		mMetrics.Count5xx = r.WithTags(map[string]string{"status": "5XX"}).Counter("count.rate")
		solomon.Rated(mMetrics.Count5xx)
		mMetrics.CountOtherError = r.WithTags(map[string]string{"status": "OTHER_ERROR"}).Counter("count.rate")
		solomon.Rated(mMetrics.CountOtherError)
	})
}

func NewHTTPClientMetrics(r metrics.Registry, clientName string) *HTTPClientMetrics {
	return &HTTPClientMetrics{
		r: r.WithTags(map[string]string{"destination": clientName}).WithPrefix("http.client.calls"),
	}
}

func (m *HTTPClientMetrics) StoreCallResult(method, path string, rsp *resty.Response) {
	key := method + "::" + path
	value, _ := m.methods.LoadOrStore(key, &httpMethodMetrics{})
	mMetrics := value.(*httpMethodMetrics)
	mMetrics.register(m.r, method, path)
	if rsp == nil {
		mMetrics.CountOtherError.Inc()
	} else {
		code := rsp.StatusCode()
		if code >= 200 && code < 300 {
			mMetrics.Count2xx.Inc()
		} else if code >= 400 && code < 500 {
			mMetrics.Count4xx.Inc()
		} else if code >= 500 {
			mMetrics.Count5xx.Inc()
		} else {
			mMetrics.CountOtherError.Inc()
		}
	}
}

type grpcMethodMetrics struct {
	registerOnce     sync.Once
	CountOK          metrics.Counter
	CountUserError   metrics.Counter
	CountServerError metrics.Counter
}

func (mMetrics *grpcMethodMetrics) register(r metrics.Registry, method string) {
	mMetrics.registerOnce.Do(func() {
		r = r.WithTags(map[string]string{"endpoint": method})
		mMetrics.CountOK = r.Counter("count_ok.rate")
		solomon.Rated(mMetrics.CountOK)
		mMetrics.CountUserError = r.Counter("count_user_error.rate")
		solomon.Rated(mMetrics.CountUserError)
		mMetrics.CountServerError = r.Counter("count_server_error.rate")
		solomon.Rated(mMetrics.CountServerError)
	})
}

func NewGrpcClientMetricsInterceptor(r metrics.Registry, clientName string) grpc.UnaryClientInterceptor {
	r = r.WithTags(map[string]string{"destination": clientName}).WithPrefix("grpc.client.calls")
	var methods sync.Map
	return func(ctx context.Context, method string, req, reply interface{}, cc *grpc.ClientConn, invoker grpc.UnaryInvoker, opts ...grpc.CallOption) error {
		err := invoker(ctx, method, req, reply, cc, opts...)
		value, _ := methods.LoadOrStore(method, &grpcMethodMetrics{})
		mMetrics := value.(*grpcMethodMetrics)
		mMetrics.register(r, method)
		if err == nil {
			mMetrics.CountOK.Inc()
		} else {
			code := status.Code(err)
			if _, ok := GRPCUserErrorCodes[code]; ok {
				mMetrics.CountUserError.Inc()
			} else {
				mMetrics.CountServerError.Inc()
			}
		}
		return err
	}
}
