package exp3matcher

import (
	"context"
	"encoding/json"
	"net"
	"net/http"
	"strings"

	"github.com/go-resty/resty/v2"
	"google.golang.org/protobuf/encoding/protojson"
	"google.golang.org/protobuf/proto"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/ctxlog"
	"a.yandex-team.ru/library/go/core/metrics"
	"a.yandex-team.ru/library/go/core/xerrors"
	geobaseLib "a.yandex-team.ru/library/go/yandex/geobase"
	v1 "a.yandex-team.ru/travel/app/backend/api/serverconfig/v1"
	"a.yandex-team.ru/travel/app/backend/internal/common"
	"a.yandex-team.ru/travel/app/backend/internal/l10n"
	"a.yandex-team.ru/travel/app/backend/internal/lib/clientscommon"
	exp3pb "a.yandex-team.ru/travel/app/backend/internal/lib/exp3matcher/proto/v1"
	"a.yandex-team.ru/travel/library/go/geobase"
)

const (
	serviceTitle = "Taxi exp3-matcher"
)

type HTTPClient struct {
	config      Config
	httpClient  *resty.Client
	logger      log.Logger
	metrics     *clientscommon.HTTPClientMetrics
	l10nService L10nService
	geoBase     geobase.Geobase
}

type L10nService interface {
	Get(keysetName string, language string) (*l10n.Keyset, error)
}

func NewHTTPClient(
	config Config,
	logger log.Logger,
	metricsRegistry metrics.Registry,
	l10nService L10nService,
	geoBase geobase.Geobase,
) *HTTPClient {
	client := resty.New()
	transport := http.Transport{
		Dial: func(_, _ string) (net.Conn, error) {
			return net.DialTimeout("unix", config.SocketFilePath, config.Timeout)
		},
	}
	client.SetTransport(&transport).SetScheme("http").SetBaseURL("localhost").SetTimeout(config.Timeout)
	return &HTTPClient{
		config:      config,
		httpClient:  client,
		logger:      logger,
		metrics:     clientscommon.NewHTTPClientMetrics(metricsRegistry, "exp3-matcher"),
		l10nService: l10nService,
		geoBase:     geoBase,
	}
}

func (c *HTTPClient) GetUpdateInfoConfig(ctx context.Context) *v1.UpdateInfo {
	req := &GetConfigsRequest{
		Consumer: "travel-app/server-config/update-info",
	}
	c.AddArgsFromContext(ctx, req)

	rsp := &exp3pb.GetUpdateInfoConfigRsp{}
	err := c.executeProto(ctx, resty.MethodPost, "/v1/configs", req, rsp)
	if err != nil {
		ctxlog.Warn(ctx, c.logger, "not found update config", log.Error(err))
		return nil
	}
	if len(rsp.Items) != 1 {
		rspStr, _ := json.Marshal(rsp)
		ctxlog.Warn(ctx, c.logger, "unexpected update config", log.String("response", string(rspStr)))
		return nil
	}
	updateInfo := rsp.Items[0].Value
	c.localizeUpdateInfo(ctx, updateInfo)
	return updateInfo
}

func (c *HTTPClient) GetUserProfileConfig(ctx context.Context) *v1.UserProfileConfig {
	req := &GetConfigsRequest{
		Consumer: "travel-app/server-config/user-profile",
	}
	c.AddArgsFromContext(ctx, req)

	rsp := &exp3pb.GetUserProfileConfigRsp{}
	err := c.executeProto(ctx, resty.MethodPost, "/v1/configs", req, rsp)
	if err != nil {
		ctxlog.Error(ctx, c.logger, "not found user profile config", log.Error(err))
		return nil
	}
	if len(rsp.Items) != 1 {
		rspStr, _ := json.Marshal(rsp)
		ctxlog.Error(ctx, c.logger, "unexpected user profile config", log.String("response", string(rspStr)))
		return nil
	}

	result := rsp.Items[0].Value
	return result
}

func (c *HTTPClient) GetRetryPolicyConfig(ctx context.Context) *v1.RetryPolicyCfg {
	req := &GetConfigsRequest{
		Consumer: "travel-app/server-config/retry-policy",
	}
	c.AddArgsFromContext(ctx, req)

	rsp := &exp3pb.GetRetryPolicyConfigRsp{}
	err := c.executeProto(ctx, resty.MethodPost, "/v1/configs", req, rsp)
	if err != nil {
		ctxlog.Error(ctx, c.logger, "not found retry policy", log.Error(err))
		return nil
	}
	if len(rsp.Items) != 1 {
		rspStr, _ := json.Marshal(rsp)
		ctxlog.Error(ctx, c.logger, "unexpected retry policy", log.String("response", string(rspStr)))
		return nil
	}

	result := rsp.Items[0].Value
	result = c.validateRetryPolicy(ctx, result)
	return result
}

func (c *HTTPClient) GetAviaConfig(ctx context.Context) *exp3pb.GetAviaConfigRspData {
	req := &GetConfigsRequest{
		Consumer: "travel-app/backend-config/avia-config",
	}
	c.AddArgsFromContext(ctx, req)

	rsp := &exp3pb.GetAviaConfigRsp{}
	err := c.executeProto(ctx, resty.MethodPost, "/v1/configs", req, rsp)
	if err != nil {
		ctxlog.Error(ctx, c.logger, "not found avia instant-search config", log.Error(err))
		return nil
	}
	if len(rsp.Items) != 1 {
		rspStr, _ := json.Marshal(rsp)
		ctxlog.Error(ctx, c.logger, "unexpected avia instant-search", log.String("response", string(rspStr)))
		return nil
	}

	result := rsp.Items[0].Value
	return result
}

func (c *HTTPClient) AddArgsFromContext(ctx context.Context, r *GetConfigsRequest) {
	userAgent := common.GetUserAgent(ctx)
	user := common.GetUser(ctx)
	deviceID := common.GetDeviceID(ctx)
	locale := common.GetLocale(ctx)
	ip := common.GetRealIP(ctx)

	r.AddStringArg("application.platform", strings.ToLower(userAgent.OSName))
	r.AddApplicationVersionArg("application.version", userAgent.AppVersion)
	r.AddIntArg("application.version_code", int(userAgent.AppVersionCode))
	r.AddStringArg("appmetrica_device_id", deviceID)
	if user == nil {
		r.AddIntArg("passport_uid", 0)
	} else {
		r.AddIntArg("passport_uid", int(user.ID))
	}
	r.AddStringArg("locale.language", locale.Language)
	r.AddStringArg("locale.country_code", locale.CountryCodeAlpha2)

	var geoID, regionID, countryID int
	if ip != nil {
		userGeoRegion, err := c.geoBase.GetRegionByIP(*ip)
		if err != nil {
			ctxlog.Warn(ctx, c.logger, "error get user geo by ip", log.Error(err))
		} else {
			geoID = int(userGeoRegion.ID)
			userRegion, err := c.geoBase.RoundToRegion(geoID, geobaseLib.RegionTypeRegion)
			if err != nil {
				ctxlog.Warn(ctx, c.logger, "error round user geo to region", log.Error(err))
			} else {
				regionID = int(userRegion.ID)
			}
			userCountry, err := c.geoBase.RoundToRegion(geoID, geobaseLib.RegionTypeCountry)
			if err != nil {
				ctxlog.Warn(ctx, c.logger, "error round user geo to country", log.Error(err))
			} else {
				countryID = int(userCountry.ID)
			}
		}
	}
	r.AddIntArg("geo_id", geoID)
	r.AddIntArg("region_geo_id", regionID)
	r.AddIntArg("country_geo_id", countryID)
}

func (c *HTTPClient) execute(ctx context.Context, method, path string, body, result interface{}) error {
	var errResponse map[string]interface{}
	r := c.httpClient.R().SetContext(ctx)
	r.SetBody(body)
	if result != nil {
		r = r.SetResult(result)
	}
	r.SetError(&errResponse)

	response, err := r.Execute(method, path)
	c.metrics.StoreCallResult(method, path, response)
	if err != nil {
		return clientscommon.ResponseError.Wrap(err)
	}
	if !response.IsSuccess() {
		raw := response.Body()
		return xerrors.Errorf("unexpected response from %v service: %w", serviceTitle, clientscommon.StatusError{
			Status:      response.StatusCode(),
			Response:    errResponse,
			ResponseRaw: string(raw),
		})
	}
	return nil
}

func (c *HTTPClient) executeProto(ctx context.Context, method, path string, body interface{}, result proto.Message) error {
	var mapResponse map[string]interface{}
	r := c.httpClient.R().SetContext(ctx)
	r.SetBody(body)
	r.SetError(&mapResponse)
	response, err := r.Execute(method, path)
	c.metrics.StoreCallResult(method, path, response)
	if err != nil {
		return clientscommon.ResponseError.Wrap(err)
	}
	raw := response.Body()
	if response.IsSuccess() {
		err = protojson.Unmarshal(raw, result)
		if err != nil {
			err = xerrors.Errorf("error parse response: %w\nraw response: %s", err, string(raw))
			return clientscommon.ParseResponseError.Wrap(err)
		}
	} else {
		return xerrors.Errorf("unexpected response from %v service: %w", serviceTitle, clientscommon.StatusError{
			Status:      response.StatusCode(),
			Response:    mapResponse,
			ResponseRaw: string(raw),
		})
	}
	return nil
}

func (c *HTTPClient) localizeUpdateInfo(ctx context.Context, info *v1.UpdateInfo) {
	lang := common.GetLanguage(ctx)
	if info != nil && info.Alert != nil {
		{
			ok, newString, err := c.tryLocalizeString(lang, info.Alert.Title)
			if err != nil {
				info.Alert.Title = ""
				ctxlog.Error(ctx, c.logger, "localization error on update_info.alert.title", log.Error(err))
			} else if ok {
				info.Alert.Title = newString
			}
		}
		{
			ok, newString, err := c.tryLocalizeString(lang, info.Alert.Subtitle)
			if err != nil {
				info.Alert.Subtitle = ""
				ctxlog.Error(ctx, c.logger, "localization error on update_info.alert.subtitle", log.Error(err))
			} else if ok {
				info.Alert.Subtitle = newString
			}
		}
	}
}

func (c *HTTPClient) tryLocalizeString(lang, source string) (ok bool, result string, err error) {
	items := strings.Split(source, ":")
	if len(items) > 1 && items[0] == "tanker" {
		if len(items) == 3 {
			keySetName := items[1]
			keyName := items[2]
			keySet, err := c.l10nService.Get(keySetName, lang)
			if err != nil {
				return false, "", err
			}
			value, ok := keySet.Keys[keyName]
			if !ok {
				return false, "", xerrors.Errorf("key '%s' not found in keyset '%s'", keyName, keySetName)
			}
			return true, value, nil
		} else {
			return false, "", xerrors.Errorf("tanker string format should be 'tanker:<keyset>:<key>', but found '%s'", source)
		}
	}
	return false, "", nil
}

func (c *HTTPClient) validateRetryPolicy(ctx context.Context, source *v1.RetryPolicyCfg) *v1.RetryPolicyCfg {
	if source.DefaultConfig == nil || source.DefaultConfig.PolicyId == "" {
		ctxlog.Error(ctx, c.logger, "empty default retry policy config")
		return nil
	}
	result := &v1.RetryPolicyCfg{
		RetryPolicies:          make([]*v1.RetryPolicyCfg_Policy, 0, len(source.RetryPolicies)),
		RequestsRetryPolicyCfg: make([]*v1.RetryPolicyCfg_RequestInfo, 0, len(source.RequestsRetryPolicyCfg)),
	}
	policyIds := map[string]int{}
	for _, rp := range source.RetryPolicies {
		if count, ok := policyIds[rp.Id]; ok {
			policyIds[rp.Id] = count + 1
		} else {
			policyIds[rp.Id] = 1
			result.RetryPolicies = append(result.RetryPolicies, rp)
		}
	}
	for policyID, count := range policyIds {
		if count > 1 {
			ctxlog.Error(ctx, c.logger, "there are several policies with same id",
				log.String("policyID", policyID), log.Int("count", count))
		}
	}
	for _, rrpc := range source.RequestsRetryPolicyCfg {
		if _, ok := policyIds[rrpc.PolicyId]; ok {
			result.RequestsRetryPolicyCfg = append(result.RequestsRetryPolicyCfg, rrpc)
		} else {
			ctxlog.Error(ctx, c.logger, "there is no policy with id",
				log.String("policyID", rrpc.PolicyId))
		}
	}
	if _, ok := policyIds[source.DefaultConfig.PolicyId]; ok {
		result.DefaultConfig = source.DefaultConfig
	} else {
		ctxlog.Error(ctx, c.logger, "there is no policy for default config",
			log.String("policyID", source.DefaultConfig.PolicyId))
		return nil
	}
	return result
}
