package priceindexclient

import (
	"context"
	"net/url"
	"time"

	"github.com/go-resty/resty/v2"
	"github.com/opentracing/opentracing-go"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/ctxlog"
	"a.yandex-team.ru/library/go/core/metrics"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/app/backend/internal/common"
	"a.yandex-team.ru/travel/app/backend/internal/lib/clientscommon"
)

type Client interface {
	GetPriceBatch(ctx context.Context, data *PriceBatchReq) (rsp *PriceBatchRsp, err error)
}

type HTTPClient struct {
	logger     log.Logger
	config     *Config
	httpClient *resty.Client
	metrics    *clientscommon.HTTPClientMetrics
}

func NewHTTPClient(
	logger log.Logger,
	config *Config,
	logRequestAndResponse bool,
	metricsRegistry metrics.Registry,
) *HTTPClient {
	client := resty.New().SetTimeout(config.Timeout).SetLogger(logger)
	if logRequestAndResponse {
		client.Debug = true // Влияет только на логирование запроса и ответа
	}
	return &HTTPClient{
		logger:     logger,
		config:     config,
		httpClient: client,
		metrics:    clientscommon.NewHTTPClientMetrics(metricsRegistry, "price-index"),
	}
}

func (c *HTTPClient) GetPriceBatch(ctx context.Context, data *PriceBatchReq) (*PriceBatchRsp, error) {
	path := "/search_methods/v1/min_price_batch_search/" + c.config.NationalVersion
	reqs := make([]BatchBodyReq, len(data.Requests))
	for i, r := range data.Requests {
		reqs[i] = BatchBodyReq{
			FromID: r.FromID,
			ToID:   r.ToID,
		}
		df := common.FormatDateProto(r.DateForward)
		if df == "" {
			df = time.Now().Format("2006-01-02")
		}
		reqs[i].ForwardDate = df
		db := common.FormatDateProto(r.DateBackward)
		if db != "" {
			reqs[i].BackwardDate = &db
		}
		if r.Adults != nil {
			reqs[i].Adults = r.Adults
		}
		if r.Children != nil {
			reqs[i].Children = r.Children
		}
		if r.Infants != nil {
			reqs[i].Infants = r.Infants
		}
	}
	body := BatchBodyReqs{
		MinRequests: reqs,
	}

	var result PriceBatchRsp
	if err := c.execute(ctx, resty.MethodPost, path, body, &result, nil); err != nil {
		ctxlog.Error(ctx, c.logger, "unable to price-index request", log.Error(err))
		return nil, xerrors.Errorf("unable to price-index request: %w", err)
	}
	if result.Status != "ok" {
		ctxlog.Error(ctx, c.logger, "bad status in price-index response", log.Any("result", result))
		return nil, xerrors.Errorf("bad status in price-index response: %v", result.Status)
	}

	return &result, nil
}

func (c *HTTPClient) doExecute(ctx context.Context, method, path string, body, result interface{}, queryParams url.Values) error {
	endpoint := c.config.BaseURL + path

	var errResponse map[string]interface{}
	r := c.httpClient.R().
		SetContext(ctx).
		SetBody(body).
		SetError(&errResponse)
	if result != nil {
		r = r.SetResult(result)
	}
	if queryParams != nil {
		r = r.SetQueryParamsFromValues(queryParams)
	}

	if span := opentracing.SpanFromContext(ctx); span != nil {
		err := opentracing.GlobalTracer().Inject(
			span.Context(),
			opentracing.HTTPHeaders,
			opentracing.HTTPHeadersCarrier(r.Header))
		if err != nil {
			err := xerrors.Errorf("err is %s", err.Error())
			ctxlog.Error(ctx, c.logger, "unable attach tracing info to request.", log.Error(err))
		}
	}

	response, err := r.Execute(method, endpoint)
	c.metrics.StoreCallResult(method, path, response)
	if err != nil || !response.IsSuccess() {
		rawResponse := ""
		if response != nil {
			rawResponse = string(response.Body())
		}

		return xerrors.Errorf("bad response from price-index service: %w", clientscommon.StatusError{
			Status:        response.StatusCode(),
			Response:      errResponse,
			ResponseRaw:   rawResponse,
			InternalError: err,
		})
	}
	return nil
}

func (c *HTTPClient) execute(ctx context.Context, method, path string, body, result interface{}, queryParams url.Values) error {
	span, ctx := opentracing.StartSpanFromContext(ctx, path)
	defer span.Finish()
	return c.doExecute(ctx, method, path, body, result, queryParams)
}
