package tankerclient

import (
	"context"
	"fmt"
	"io"
	"net/url"

	"github.com/go-resty/resty/v2"

	"a.yandex-team.ru/library/go/core/xerrors"
)

type ServiceTicketGetter func(context.Context) (string, error)

type HTTPClient struct {
	config     *Config
	httpClient *resty.Client
}

const (
	authorizationHeader = "Authorization"
	oauth               = "OAuth"
)

func NewHTTPClient(
	config *Config,
) *HTTPClient {
	return &HTTPClient{
		config:     config,
		httpClient: resty.New().SetTimeout(config.Timeout),
	}
}

func (c *HTTPClient) Export(ctx context.Context) (*ExportResponse, error) {
	targetURL := fmt.Sprintf("%s/v1/project/%s/export", c.config.BaseURL, url.PathEscape(c.config.Project))
	if c.config.Token == "" {
		return nil, NoOAuthTokenError
	}
	var result ExportResponse
	var errResponse map[string]interface{}
	response, err := c.httpClient.R().
		SetContext(ctx).
		SetResult(&result).
		SetHeader(authorizationHeader, fmt.Sprintf("%s %s", oauth, c.config.Token)).
		SetError(&errResponse).
		Execute(resty.MethodGet, targetURL)

	if err != nil {
		return nil, ResponseError.Wrap(err)
	}
	if !response.IsSuccess() {
		raw, _ := io.ReadAll(response.RawResponse.Body)
		return nil, xerrors.Errorf("unexpected response from tanker service: %w", StatusError{
			Status:      response.StatusCode(),
			Response:    errResponse,
			ResponseRaw: string(raw),
		})
	}
	return &result, nil
}
