package travelapiclient

import (
	"context"
	"net/url"

	"github.com/go-resty/resty/v2"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/ctxlog"
	"a.yandex-team.ru/library/go/core/metrics"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/app/backend/internal/common"
	"a.yandex-team.ru/travel/app/backend/internal/lib/clientscommon"
)

const serviceTitle = "Travel API"

const (
	sessionKeyHeader     = "X-Ya-Session-Key"
	yandexUIDHeader      = "X-Ya-YandexUid"
	deviceIDHeader       = "X-Ya-Device-Id"
	userAgentHeader      = "X-Ya-User-Agent"
	userIPHeader         = "X-Ya-User-Ip"
	serviceHeader        = "X-Ya-Service"
	appPlatformHeader    = "X-Ya-App-Platform"
	iosPlatform          = "ios"
	androidPlatform      = "android"
	userDeviceHeader     = "X-Ya-User-Device"
	userDeviceTouchValue = "touch"
	isStaffHeader        = "X-Ya-User-IsStaff"
	isPlusHeader         = "X-Ya-User-IsPlus"
	trueHeaderValue      = "true"
	falseHeaderValue     = "false"
)

type HTTPClient struct {
	logger          log.Logger
	env             common.EnvType
	httpClient      *resty.Client
	headerProviders []common.HeaderProvider
	config          TravelAPIConfig
	metrics         *clientscommon.HTTPClientMetrics
}

func NewHTTPClient(
	logger log.Logger,
	env common.EnvType,
	config TravelAPIConfig,
	logRequestAndResponse bool,
	metricsRegistry metrics.Registry,
	headerProviders ...common.HeaderProvider,
) *HTTPClient {
	client := resty.New().SetTimeout(config.Timeout).SetLogger(logger).OnRequestLog(clientscommon.DoNotLogTVMHeaders)
	if logRequestAndResponse {
		client.Debug = true // Влияет только на логирование запроса и ответа
	}
	return &HTTPClient{
		logger:          logger,
		env:             env,
		httpClient:      client,
		headerProviders: headerProviders,
		config:          config,
		metrics:         clientscommon.NewHTTPClientMetrics(metricsRegistry, "travel-api"),
	}
}

func (c *HTTPClient) execute(ctx context.Context, method, path string, body, result interface{}, queryParams url.Values) error {
	endpoint := c.config.BaseURL + path
	var errResponse map[string]interface{}

	r, err := c.prepareRequest(ctx)
	if err != nil {
		return clientscommon.PreconditionError.Wrap(err)
	}

	r.SetBody(body)
	if queryParams != nil {
		r.SetQueryParamsFromValues(queryParams)
	}
	if result != nil {
		r = r.SetResult(result)
	}
	r.SetError(&errResponse)

	response, err := r.Execute(method, endpoint)
	c.metrics.StoreCallResult(method, path, response)
	if err != nil {
		return clientscommon.ResponseError.Wrap(err)
	}
	if !response.IsSuccess() {
		raw := response.Body()
		return xerrors.Errorf("unexpected response from %v service: %w", serviceTitle, clientscommon.StatusError{
			Status:      response.StatusCode(),
			Response:    errResponse,
			ResponseRaw: string(raw),
		})
	}
	return nil
}

//todo в execute не проставляется result для ручки статуса авиа (предположительно не понимает что json) - сделал временно вторую реализацию
func (c *HTTPClient) execute2(ctx context.Context, method, path string, body, queryParams url.Values) ([]byte, error) {
	endpoint := c.config.BaseURL + path
	var errResponse map[string]interface{}

	r, err := c.prepareRequest(ctx)
	if err != nil {
		return nil, clientscommon.PreconditionError.Wrap(err)
	}

	r.SetBody(body)
	if queryParams != nil {
		r.SetQueryParamsFromValues(queryParams)
	}
	r.SetError(&errResponse)

	response, err := r.Execute(method, endpoint)
	c.metrics.StoreCallResult(method, path, response)
	if err != nil {
		return nil, clientscommon.ResponseError.Wrap(err)
	}
	if !response.IsSuccess() {
		raw := response.Body()
		return nil, xerrors.Errorf("unexpected response from %v service: %w", serviceTitle, clientscommon.StatusError{
			Status:      response.StatusCode(),
			Response:    errResponse,
			ResponseRaw: string(raw),
		})
	}
	return response.Body(), nil
}

func (c *HTTPClient) prepareRequest(ctx context.Context) (*resty.Request, error) {
	r := c.httpClient.R().
		SetContext(ctx).
		SetHeader(serviceHeader, common.ServiceName).
		SetHeader(userDeviceHeader, userDeviceTouchValue)
	deviceID := common.GetDeviceID(ctx)
	if deviceID != "" {
		r.SetHeader(deviceIDHeader, deviceID)
	}
	uuid := common.GetUUID(ctx)
	if uuid != "" {
		r.SetHeader(yandexUIDHeader, uuid)
		r.SetHeader(sessionKeyHeader, uuid)
	}
	rawUserAgent := common.GetRawUserAgent(ctx)
	if rawUserAgent != "" {
		r.SetHeader(userAgentHeader, rawUserAgent)
	}
	ip := common.GetRealIP(ctx)
	if ip != nil {
		r.SetHeader(userIPHeader, *ip)
	}
	switch common.GetUserAgent(ctx).KnownOS {
	case common.OSTypeIOS:
		r.SetHeader(appPlatformHeader, iosPlatform)
	case common.OSTypeAndroid:
		r.SetHeader(appPlatformHeader, androidPlatform)
	default:
		ctxlog.Error(ctx, c.logger, "unknown os in user agent", log.String("user agent", rawUserAgent))
	}

	auth := common.GetAuth(ctx)
	isStaffValue := falseHeaderValue
	isPlusValue := falseHeaderValue
	if auth != nil {
		if auth.IsStaff() {
			isStaffValue = trueHeaderValue
		}
		if auth.IsPlus() {
			isPlusValue = trueHeaderValue
		}
	}
	r.SetHeader(isStaffHeader, isStaffValue).
		SetHeader(isPlusHeader, isPlusValue)

	for _, provider := range c.headerProviders {
		headers, err := provider(ctx)
		if err != nil {
			return nil, err
		}
		for k, v := range headers {
			if v != "" {
				r.SetHeader(k, v)
			}
		}
	}
	return r, nil
}
