package travelapiclient

import (
	"context"
	"fmt"
	"net/url"
	"strconv"
	"strings"

	"github.com/go-resty/resty/v2"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/ctxlog"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/app/backend/internal/common"
	"a.yandex-team.ru/travel/app/backend/internal/lib/travelapiclient/models"
)

const (
	searchPrefix = "/hotels_portal/v1"
)

func (c *HTTPClient) Suggest(ctx context.Context, query string, limit int, lang string, domain string, sessionID string, requestIndex int) (*models.SuggestResponse, error) {
	var result models.SuggestResponse
	queryParams := url.Values{}
	queryParams.Add("query", query)
	queryParams.Add("limit", strconv.Itoa(limit))
	queryParams.Add("sessionId", sessionID)
	queryParams.Add("requestIndex", strconv.Itoa(requestIndex))
	queryParams.Add("language", lang)
	queryParams.Add("domain", domain)
	if err := c.execute(ctx, resty.MethodGet, searchPrefix+"/suggest", nil, &result, queryParams); err != nil {
		ctxlog.Error(ctx, c.logger, "unable to execute suggest request", log.Error(err))
		return nil, xerrors.Errorf("unable to execute suggest request: %w", err)
	}
	return &result, nil
}

func (c *HTTPClient) LogSuggest(ctx context.Context, selectedID string, sessionID string, requestIndex int, isManualClick bool, isTrustedUser bool) error {
	queryParams := url.Values{}
	queryParams.Add("selectedId", selectedID)
	queryParams.Add("sessionId", sessionID)
	queryParams.Add("requestIndex", strconv.Itoa(requestIndex))
	queryParams.Add("isManualClick", strconv.FormatBool(isManualClick))
	queryParams.Add("isTrustedUser", strconv.FormatBool(isTrustedUser))
	if err := c.execute(ctx, resty.MethodPost, searchPrefix+"/log_suggest", nil, nil, queryParams); err != nil {
		ctxlog.Error(ctx, c.logger, "unable to execute log suggest request", log.Error(err))
		return xerrors.Errorf("unable to execute log suggest request: %w", err)
	}
	return nil
}

func (c *HTTPClient) SearchHotels(ctx context.Context, request models.SearchHotelsRequest) (*models.SearchHotelsResponse, error) {
	var result models.SearchHotelsResponse
	queryParams := request.BuildURLParams()
	if err := c.execute(ctx, resty.MethodGet, searchPrefix+"/search_hotels", nil, &result, queryParams); err != nil {
		ctxlog.Error(ctx, c.logger, "unable to hotels search request", log.Error(err))
		return nil, xerrors.Errorf("unable to hotels search request: %w", err)
	}
	params := request.QueryData
	for i := range result.Hotels {
		result.Hotels[i].Hotel.HotelURL = buildHotelURL(result.Hotels[i].Hotel.Slug, result.PollingSearchID, &params.SearchParams)
	}
	return &result, nil
}

func (c *HTTPClient) GetHotelInfo(ctx context.Context, request *models.GetHotelInfoRequest) (*models.GetHotelInfoResponse, error) {
	var result models.GetHotelInfoResponse
	queryParams := request.BuildURLParams()
	if err := c.execute(ctx, resty.MethodGet, searchPrefix+"/get_hotel_info", nil, &result, queryParams); err != nil {
		ctxlog.Error(ctx, c.logger, "unable to hotel info request", log.Error(err))
		return nil, xerrors.Errorf("unable to hotel info request: %w", err)
	}
	params := request.Params
	searchParams := models.SearchParams{}
	if params != nil {
		searchParams.CheckinDate = params.CheckinDate
		searchParams.CheckoutDate = params.CheckoutDate
		searchParams.Adults = params.Adults
		searchParams.ChildrenAges = params.ChildrenAges
	}
	hotelURL := buildHotelURL(result.Hotel.Slug, "", &searchParams)
	result.Hotel.HotelURL = hotelURL

	return &result, nil
}

func (c *HTTPClient) GetHotelImages(ctx context.Context, request *models.GetHotelImagesRequest) (*models.GetHotelImagesResponse, error) {
	var result models.GetHotelImagesResponse
	queryParams := request.BuildURLParams()
	if err := c.execute(ctx, resty.MethodGet, searchPrefix+"/get_hotel_images", nil, &result, queryParams); err != nil {
		ctxlog.Error(ctx, c.logger, "unable to hotel images request", log.Error(err))
		return nil, xerrors.Errorf("unable to hotel images request: %w", err)
	}
	return &result, nil
}

func (c *HTTPClient) GetHotelOffers(ctx context.Context, request *models.GetHotelOffersRequest) (*models.GetHotelOffersResponse, error) {
	var result models.GetHotelOffersResponse
	queryParams := request.BuildURLParams()
	if err := c.execute(ctx, resty.MethodGet, searchPrefix+"/get_hotel_offers", nil, &result, queryParams); err != nil {
		ctxlog.Error(ctx, c.logger, "unable to hotel offers request", log.Error(err))
		return nil, xerrors.Errorf("unable to hotel offers request: %w", err)
	}
	return &result, nil
}

func (c *HTTPClient) GetSimilarHotels(ctx context.Context, request *models.GetSimilarHotelsRequest) (*models.GetSimilarHotelsResponse, error) {
	var result models.GetSimilarHotelsResponse
	queryParams := request.BuildURLParams()
	if err := c.execute(ctx, resty.MethodGet, searchPrefix+"/get_similar_hotels", nil, &result, queryParams); err != nil {
		ctxlog.Error(ctx, c.logger, "unable to similar hotels request", log.Error(err))
		return nil, xerrors.Errorf("unable to similar hotels request: %w", err)
	}
	return &result, nil
}

func (c *HTTPClient) GetHotelsCounters(ctx context.Context, request *models.GetCountersRequest) (*models.GetCountersResponse, error) {
	var result models.GetCountersResponse
	queryParams := request.BuildURLParams()
	if err := c.execute(ctx, resty.MethodGet, searchPrefix+"/count_hotels", nil, &result, queryParams); err != nil {
		ctxlog.Error(ctx, c.logger, "unable to hotels counters request", log.Error(err))
		return nil, xerrors.Errorf("unable to hotels counters request: %w", err)
	}
	return &result, nil
}

// Hotel order

func (c *HTTPClient) GetOrderStatus(ctx context.Context, orderID string) (*models.GetOrderStatusRsp, error) {
	res := models.GetOrderStatusRsp{}
	err := c.execute(ctx, resty.MethodGet, "/booking_flow/v1/get_order_status", nil, &res, url.Values{
		"id": []string{orderID},
	})
	if err != nil {
		return nil, xerrors.Errorf("error while get hotel order status :%w", err)
	}
	return &res, nil
}

func (c *HTTPClient) GetOrder(ctx context.Context, orderID string) (*models.GetOrderRsp, error) {
	res := models.GetOrderRsp{}
	err := c.execute(ctx, resty.MethodGet, "/booking_flow/v1/get_order", nil, &res, url.Values{
		"id": []string{orderID},
	})
	if err != nil {
		return nil, xerrors.Errorf("error while get hotel order status :%w", err)
	}
	return &res, nil
}

func (c *HTTPClient) GetOrderByToken(ctx context.Context, request *models.GetOrderByTokenRequest) (*models.GetOrderByTokenResponse, error) {
	res := models.GetOrderByTokenResponse{}
	queryParams := request.BuildURLParams()
	err := c.execute(ctx, resty.MethodGet, "/booking_flow/v1/get_order_info_by_token", nil, &res, queryParams)
	if err != nil {
		return nil, xerrors.Errorf("error while get order by token :%w", err)
	}

	return &res, nil
}

func (c *HTTPClient) StartPayment(ctx context.Context, orderID string, paymentTestContextToken *string) error {
	body := models.StartPaymentReq{
		ReturnURL:               c.config.PaymentRedirectURL,
		CustomerSource:          common.ServiceName,
		OrderID:                 orderID,
		PaymentTestContextToken: paymentTestContextToken,
	}
	err := c.execute(ctx, resty.MethodPost, "/booking_flow/v1/start_payment", body, nil, nil)
	if err != nil {
		return xerrors.Errorf("error while start payment :%w", err)
	}
	return nil
}

func (c *HTTPClient) GetHotelHappyPage(ctx context.Context, orderID string) (*models.GetHotelHappyPageRsp, error) {
	res := models.GetHotelHappyPageRsp{}
	err := c.execute(ctx, resty.MethodGet, "/orders/v1/get_order_happy_page", nil, &res, url.Values{
		"orderId": []string{orderID},
		"geoId":   []string{"0"}, //todo(adurnev)
	})
	if err != nil {
		return nil, xerrors.Errorf("error while get happy page :%w", err)
	}

	return &res, nil
}

func buildHotelURL(slug, searchID string, searchParams *models.SearchParams) string {
	slugParts := strings.Split(slug, "/")
	for i := range slugParts {
		slugParts[i] = url.PathEscape(slugParts[i])
	}
	slug = strings.Join(slugParts, "/")

	path := fmt.Sprintf("/hotels/%s/", slug)
	queryValues := url.Values{}
	if searchParams != nil {
		childrenStrings := make([]string, len(searchParams.ChildrenAges))
		for i, c := range searchParams.ChildrenAges {
			childrenStrings[i] = strconv.Itoa(c)
		}
		childrenAges := strings.Join(childrenStrings, ",")

		queryValues.Set("adults", strconv.Itoa(searchParams.Adults))
		queryValues.Set("checkinDate", searchParams.CheckinDate)
		queryValues.Set("checkoutDate", searchParams.CheckoutDate)
		queryValues.Set("childrenAges", childrenAges)
		queryValues.Set("searchPagePollingId", searchID)
		queryValues.Set("seed", "app-search")
	}
	return fmt.Sprintf("%s?%s", path, queryValues.Encode())
}

func (c *HTTPClient) CreateOrder(ctx context.Context, req *models.CreateOrderRequest) (*models.CreateOrderResponse, error) {
	var result models.CreateOrderResponse
	err := c.execute(ctx, resty.MethodPost, "/booking_flow/v1/create_order", req, &result, nil)
	if err != nil {
		return nil, xerrors.Errorf("error while create order :%w", err)
	}
	return &result, nil
}

func (c *HTTPClient) EstimateDiscount(ctx context.Context, req *models.EstimateDiscountRequest) (*models.EstimateDiscountResponse, error) {
	var result models.EstimateDiscountResponse
	err := c.execute(ctx, resty.MethodPost, "/booking_flow/v1/estimate_discount", req, &result, nil)
	if err != nil {
		return nil, xerrors.Errorf("error while estimate discount :%w", err)
	}
	return &result, nil
}
