package models

import (
	"net/url"
	"time"
)

type PaymentErrorCode string

const (
	PaymentErrorCodeAuthorizationReject     PaymentErrorCode = "AUTHORIZATION_REJECT"
	PaymentErrorCodeExpiredCard             PaymentErrorCode = "EXPIRED_CARD"
	PaymentErrorCodeFail3DS                 PaymentErrorCode = "FAIL_3DS"
	PaymentErrorCodeLimitExceeded           PaymentErrorCode = "LIMIT_EXCEEDED"
	PaymentErrorCodeNotEnoughFunds          PaymentErrorCode = "NOT_ENOUGH_FUNDS"
	PaymentErrorCodeTransactionNotPermitted PaymentErrorCode = "TRANSACTION_NOT_PERMITTED"
	PaymentErrorCodeUserCancelled           PaymentErrorCode = "USER_CANCELLED"
	PaymentErrorCodeRestrictedCard          PaymentErrorCode = "RESTRICTED_CARD"
	PaymentErrorCodeBlacklisted             PaymentErrorCode = "BLACKLISTED"
	PaymentErrorCodePaymentTimeout          PaymentErrorCode = "PAYMENT_TIMEOUT"
	PaymentErrorCodeOther                   PaymentErrorCode = "OTHER"
)

type OrderStatus string

const (
	OrderStatusInProgress               OrderStatus = "IN_PROGRESS"
	OrderStatusReserved                 OrderStatus = "RESERVED"
	OrderStatusReservedWithRestrictions OrderStatus = "RESERVED_WITH_RESTRICTIONS"
	OrderStatusAwaitsPayment            OrderStatus = "AWAITS_PAYMENT"
	OrderStatusPaymentFailed            OrderStatus = "PAYMENT_FAILED"
	OrderStatusConfirmed                OrderStatus = "CONFIRMED"
	OrderStatusCancelled                OrderStatus = "CANCELLED"
	OrderStatusCancelledWithRefund      OrderStatus = "CANCELLED_WITH_REFUND"
	OrderStatusRefunded                 OrderStatus = "REFUNDED"
	OrderStatusRefundFailed             OrderStatus = "REFUND_FAILED"
	OrderStatusFailed                   OrderStatus = "FAILED"
)

type GetOrderStatusRsp struct {
	Status       OrderStatus `json:"status"`
	PaymentURL   *string     `json:"payment_url"`
	PaymentError *string     `json:"payment_error"`
}

type GetOrderRsp struct {
	Status  OrderStatus `json:"status"`
	Payment *Payment    `json:"payment"`
}

type Payment struct {
	Current    *CurrentPayment `json:"current"`
	AmountPaid Price           `json:"amount_paid"`
	Error      *PaymentError   `json:"error"`
}

type CurrentPayment struct {
	PaymentURL *string `json:"payment_url"`
}

type PaymentError struct {
	Code PaymentErrorCode `json:"code"`
}

type HappyPageOrderType string

const (
	HappyPageOrderTypeAvia  HappyPageOrderType = "AVIA"
	HappyPageOrderTypeHotel HappyPageOrderType = "HOTEL"
)

type GetHappyPageRsp struct {
	CrossSale CrossSaleBlocks    `json:"cross_sale"`
	OrderType HappyPageOrderType `json:"order_type"`
}
type CrossSaleBlocks struct {
	Blocks []CrossSaleBlock `json:"blocks"`
}
type CrossSaleBlock struct {
	Order     int       `json:"order"`
	BlockType string    `json:"block_type"`
	UIPayload UIPayload `json:"ui_payload"`
}
type UIPayload struct {
	AdFoxID string `json:"ad_fox_id"`
}
type HotelGuestInfo struct {
	CustomerEmail      string       `json:"customer_email"`
	CustomerPhone      string       `json:"customer_phone"`
	AllowsSubscription bool         `json:"allows_subscription"`
	Guests             []HotelGuest `json:"guests"`
}
type HotelGuest struct {
	FirstName string `json:"first_name"`
	LastName  string `json:"last_name"`
	Empty     bool   `json:"empty"`
}
type CancellationInfoPartner struct {
	Refundable  bool      `json:"refundable"`
	Penalties   []Penalty `json:"penalties"`
	Highlighted bool      `json:"highlighted"`
}

type Penalty struct {
	StartsAt *string `json:"starts_at"`
	EndsAt   *string `json:"ends_at"`
	Type     string  `json:"type"`
	Amount   *string `json:"amount"`
	Currency *string `json:"currency"`
}

type HotelOrderInfo struct {
	TravelToken                string                  `json:"travel_token"`
	Label                      string                  `json:"label"`
	Checksum                   string                  `json:"checksum"`
	SessionKey                 string                  `json:"session_key"`
	BasicHotelInfo             HotelCheckout           `json:"basic_hotel_info"`
	PartnerHotelInfo           HotelPartner            `json:"partner_hotel_info"`
	PartnerRoomInfo            RoomPartner             `json:"partner_room_info"`
	BedGroups                  []BedGroupsCheckout     `json:"bed_groups"`
	RequestInfo                RequestInfo             `json:"request_info"`
	RateInfo                   RateInfo                `json:"rate_info"`
	PansionInfo                PansionInfo             `json:"pansion_info"`
	CancellationInfo           CancellationInfoPartner `json:"cancellation_info"`
	CancellationInfoUnfiltered CancellationInfoPartner `json:"cancellation_info_unfiltered"`
	LegalInfoGroup             LegalInfoGroup          `json:"legal_info"`
	PartnerID                  string                  `json:"partner_id"`
	DirectPartner              bool                    `json:"direct_partner"`
	PromoCampaigns             PromoCampaigns          `json:"promo_campaigns"`
}

type GetHotelHappyPageRsp struct {
	GetHappyPageRsp
	Order HotelOrder `json:"order"`
}

type HotelOrder struct {
	ID            string         `json:"id"`
	YandexOrderID string         `json:"yandex_order_id"`
	OrderInfo     HotelOrderInfo `json:"order_info"`
	GuestInfo     HotelGuestInfo `json:"guest_info"`
	Status        OrderStatus    `json:"status"`
	//OrderCancellationDetails interface{}    `json:"order_cancellation_details"`
	Payment               PaymentReceipts       `json:"payment"`
	ConfirmationInfo      ConfirmationInfo      `json:"confirmation_info"`
	RefundInfo            RefundInfo            `json:"refund_info"`
	PromoCampaignsApplied PromoCampaignsApplied `json:"applied_promo_campaigns"`
}

type PaymentReceipts struct {
	Receipts  []Receipt         `json:"receipts"`
	ErrorInfo *PaymentErrorCode `json:"error_info"`
	//Next                 interface{}       `json:"next"`
	UsesDeferredPayments bool `json:"uses_deferred_payments"`
	//AmountPaid           Price `json:"amount_paid"`
}

type Receipt struct {
	URL  string `json:"url"`
	Type string `json:"type"`
}

type ConfirmationInfo struct {
	ConfirmationID string `json:"confirmation_id"`
}

type RefundReason string

const (
	RefundReasonUser     RefundReason = "USER"
	RefundReasonOperator RefundReason = "OPERATOR"
	RefundReasonSchedule RefundReason = "SCHEDULE"
)

type RefundInfo struct {
	RefundDateTime       string       `json:"refund_date_time"`
	PenaltyIntervalIndex int          `json:"penalty_interval_index"`
	Refund               Amount       `json:"refund"`
	Penalty              Amount       `json:"penalty"`
	Reason               RefundReason `json:"reason"`
}

type StartPaymentReq struct {
	CustomerSource          string  `json:"customer_source"`
	OrderID                 string  `json:"order_id"`
	PaymentTestContextToken *string `json:"payment_test_context_token,omitempty"`
	ReturnURL               string  `json:"return_url"`
}

type GetOrderByTokenRequest struct {
	Token     string
	Label     string
	IP        string
	UserAgent string
}

func (req *GetOrderByTokenRequest) BuildURLParams() url.Values {
	queryParams := url.Values{}

	queryParams.Add("token", req.Token)
	queryParams.Add("label", req.Label)
	queryParams.Add("customerIp", req.IP)
	queryParams.Add("customerUserAgent", req.UserAgent)

	return queryParams
}

type GetOrderByTokenResponse struct {
	HotelOrderInfo
	AllGuestsRequired       bool                    `json:"all_guests_required"`
	DeferredPaymentSchedule DeferredPaymentSchedule `json:"deferred_payment_schedule"`
	ExtraVisitAndUserParams ExtraVisitAndUserParams `json:"extra_visit_and_user_params"`
	RefundInfo              CancellationInfo        `json:"refund_info"`
}

type LegalInfoPartner struct {
	ActualAddress  string `json:"actual_address"`
	LegalAddress   string `json:"legal_address"`
	Name           string `json:"name"`
	Ogrn           string `json:"ogrn"`
	RegistryNumber string `json:"registry_number"`
	WorkingHours   string `json:"working_hours"`
}

type LegalInfoGroup struct {
	Hotel   LegalInfoPartner `json:"hotel"`
	Partner LegalInfoPartner `json:"partner"`
	Yandex  LegalInfoPartner `json:"yandex"`
}

type AmountValue struct {
	Currency string `json:"currency"`
	Value    int    `json:"value"`
}
type Amount struct {
	Amount   string `json:"amount"`
	Currency string `json:"currency"`
}
type AmountWithType struct {
	Amount
	Type string `json:"type"`
}

type HotelPayment struct {
	Amount          AmountValue `json:"amount"`
	PaymentEndsAt   time.Time   `json:"payment_ends_at"`
	PenaltyIfUnpaid AmountValue `json:"penalty_if_unpaid"`
	Percentage      int         `json:"percentage"`
}
type DeferredPaymentSchedule struct {
	DeferredPayments []HotelPayment `json:"deferred_payments"`
	InitialPayment   HotelPayment   `json:"initial_payment"`
	ZeroFirstPayment bool           `json:"zero_first_payment"`
}

type Address struct {
	City        string `json:"city"`
	CountryCode string `json:"country_code"`
	Line1       string `json:"line_1"`
	Line2       string `json:"line_2"`
}

type Checkin struct {
	BeginTime           string `json:"begin_time"`
	EndTime             string `json:"end_time"`
	Instructions        string `json:"instructions"`
	SpecialInstructions string `json:"special_instructions"`
}
type Checkout struct {
	Time string `json:"time"`
}
type Descriptions struct {
	Overview string `json:"overview"`
}
type Fees struct {
	Mandatory string `json:"mandatory"`
	Optional  string `json:"optional"`
}
type ImagesLinks struct {
	Links map[string]ImageLink `json:"links"`
}
type ImageLink struct {
	Href   string `json:"href"`
	Method string `json:"method"`
}
type Location struct {
	Coordinates Coordinates `json:"coordinates"`
}
type Policies struct {
	KnowBeforeYouGo string `json:"know_before_you_go"`
}
type Property struct {
	Rating string `json:"rating"`
	Type   string `json:"type"`
}
type Ratings struct {
	Property Property `json:"property"`
}
type Points struct {
	Amount     int    `json:"amount"`
	PointsName string `json:"points_name"`
	PointsType string `json:"points_type"`
}
type WhiteLabel struct {
	Eligible bool   `json:"eligible"`
	Points   Points `json:"points"`
}
type Data struct {
	FirstSendDate      string      `json:"first_send_date"`
	LastSendDate       string      `json:"last_send_sate"`
	NumberOfPromocodes int         `json:"number_of_promocodes"`
	PromocodeCost      AmountValue `json:"promocode_cost"`
}

type ExtraCharges struct {
	Face    Amount `json:"face"`
	Payable Amount `json:"payable"`
	Type    string `json:"type"`
}

type HotelChargesBase struct {
	Base         Amount           `json:"base"`
	TaxesAndFees []AmountWithType `json:"taxes_and_fees"`
}

type Totals struct {
	Base                   Amount           `json:"base"`
	Discount               Amount           `json:"discount"`
	Grand                  Amount           `json:"grand"`
	PriceAfterPlusWithdraw AmountValue      `json:"price_after_plus_withdraw"`
	StrikeThrough          Amount           `json:"strike_through"`
	TaxesAndFees           []AmountWithType `json:"taxes_and_fees"`
	TaxesAndFeesSum        Amount           `json:"taxes_and_fees_sum"`
}
type HotelCharges struct {
	Daily   []HotelChargesBase `json:"daily"`
	Nightly []HotelChargesBase `json:"nightly"`
	Totals  Totals             `json:"totals"`
}
type RateInfo struct {
	ExtraCharges []ExtraCharges `json:"extra_charges"`
	HotelCharges HotelCharges   `json:"hotel_charges"`
}
