package models

import (
	"fmt"
	"net/url"
	"strconv"
	"strings"
	"time"

	"a.yandex-team.ru/travel/app/backend/internal/common"
)

type SuggestType string

const (
	SuggestTypeRegion       SuggestType = "region"
	SuggestTypeHotel        SuggestType = "hotel"
	SuggestTypeHotelsNearby SuggestType = "hotels_nearby"
	SuggestTypeHistory      SuggestType = "history"
	SuggestTypeCrossSale    SuggestType = "cross_sale"
)

type Date struct {
	time.Time
}

type RedirectParam struct {
	BBox              string             `json:"bbox"`
	GeoID             *int64             `json:"geo_id"`
	HotelSlug         string             `json:"hotel_slug"`
	OfferSearchParams *OfferSearchParams `json:"offer_search_params"`
	Permalink         *uint64            `json:"permalink,string"`
	SelectedSortID    string             `json:"selected_sort_id"`
	SortOrigin        string             `json:"sort_origin"`
	Type              SuggestType        `json:"type"`
}

type SuggestItem struct {
	ID             string         `json:"id"`
	Name           string         `json:"name"`
	Description    string         `json:"description"`
	RedirectParams *RedirectParam `json:"redirect_params"`
}

type SuggestGroup struct {
	Name  string        `json:"name"`
	Items []SuggestItem `json:"items"`
}

type SuggestResponse struct {
	Groups []SuggestGroup `json:"groups"`
}

type Coordinates struct {
	Latitude  float64 `json:"lat"`
	Longitude float64 `json:"lon"`
}

func (c *Coordinates) String() string {
	return fmt.Sprintf("%g,%g", c.Longitude, c.Latitude)
}

type BoundingBoxRsp []Coordinates

type BoundingBox struct {
	Coordinates []Coordinates
}

func (bb *BoundingBox) String() string {
	return fmt.Sprintf("%s~%s", bb.Coordinates[0].String(), bb.Coordinates[1].String())
}

type GeoLocationStatus int

const (
	GeoLocationStatusUnknown GeoLocationStatus = iota
	GeoLocationStatusAvailable
	GeoLocationStatusForbidden
)

var geoStatusMap = map[GeoLocationStatus]string{
	GeoLocationStatusUnknown:   "unknown",
	GeoLocationStatusAvailable: "available",
	GeoLocationStatusForbidden: "forbidden",
}

func (s GeoLocationStatus) String() string {
	if val, ok := geoStatusMap[s]; ok {
		return val
	}
	return "unknown"
}

type HotelSearchStartReason int

const (
	StartSearchReasonUnknown = iota
	StartSearchReasonQueryByLocation
	StartSearchReasonMapBounds
	StartSearchReasonNavigationToken
	StartSearchReasonFilters
	StartSearchReasonFiltersTypeSearchText
	StartSearchReasonSort
	StartSearchReasonMount
)

var startSearchReasonMap = map[HotelSearchStartReason]string{
	StartSearchReasonQueryByLocation:       "queryByLocation",
	StartSearchReasonMapBounds:             "mapBounds",
	StartSearchReasonNavigationToken:       "navigationToken",
	StartSearchReasonFilters:               "filters",
	StartSearchReasonFiltersTypeSearchText: "filtersTypeSearchText",
	StartSearchReasonSort:                  "sort",
	StartSearchReasonMount:                 "mount",
}

func (s HotelSearchStartReason) String() string {
	if val, ok := startSearchReasonMap[s]; ok {
		return val
	}
	return "mount"
}

type PollingData struct {
	PollEpoch       int
	PollIteration   int
	NavigationToken *string
}

type HotelsSearchMainParams struct {
	SearchParams
	PollingData            *PollingData
	Context                *string
	GeoID                  *int
	SortOrigin             *Coordinates
	GeoLocationStatus      GeoLocationStatus
	UserCoordinates        *Coordinates
	TotalHotelsLimit       int
	PageHotelCount         int
	Bbox                   *BoundingBox
	HotelSearchStartReason HotelSearchStartReason
	// TODO(adurnev) use HotelImageParams
	ImageLimit   int
	ImageOnlyTop bool
}

type HotelsSearchFilter struct {
	Atoms []string
}

type HotelPriceFilter struct {
	PriceMin uint32
	PriceMax uint32
}

type HotelsSearchSorter struct {
	SorterID string
}

type AnalyticsParams struct {
	UtmSource   *string
	UtmMedium   *string
	UtmCampaign *string
	UtmTerm     *string
	UtmContent  *string
}

type SearchHotelsRequest struct {
	QueryData       HotelsSearchMainParams
	Filters         HotelsSearchFilter
	PriceFilter     HotelPriceFilter
	Sorter          HotelsSearchSorter
	AnalyticsParams *AnalyticsParams
}

func (req *SearchHotelsRequest) BuildURLParams() url.Values {
	queryParams := url.Values{}

	addHotelSearchParams(&req.QueryData.SearchParams, queryParams)

	mainParams := req.QueryData
	if mainParams.Context != nil {
		queryParams.Add("context", *mainParams.Context)
	}

	if mainParams.GeoLocationStatus != GeoLocationStatusUnknown && mainParams.UserCoordinates != nil {
		queryParams.Add("geo_location_status", mainParams.GeoLocationStatus.String())
		queryParams.Add("user_coordinates", mainParams.UserCoordinates.String())
	}
	if mainParams.SortOrigin != nil {
		queryParams.Add("sort_origin", mainParams.SortOrigin.String())
	}
	if mainParams.Bbox != nil {
		queryParams.Add("bbox", mainParams.Bbox.String())
	}
	queryParams.Add("start_search_reason", mainParams.HotelSearchStartReason.String())
	if mainParams.GeoID != nil {
		queryParams.Add("geo_id", strconv.Itoa(*mainParams.GeoID))
	}
	queryParams.Add("total_hotel_limits", strconv.Itoa(mainParams.TotalHotelsLimit))
	queryParams.Add("page_hotel_count", strconv.Itoa(mainParams.PageHotelCount))

	if mainParams.PollingData != nil {
		queryParams.Add("poll_epoch", strconv.Itoa(mainParams.PollingData.PollEpoch))
		queryParams.Add("poll_iteration", strconv.Itoa(mainParams.PollingData.PollIteration))
		if mainParams.PollingData.NavigationToken != nil {
			queryParams.Add("navigation_token", *mainParams.PollingData.NavigationToken)
		}
	}
	if mainParams.ImageLimit > 0 {
		queryParams.Add("image_limit", strconv.Itoa(mainParams.ImageLimit))
	}
	//TODO(adurnev) запрашивать только sizes: orig, после https://st.yandex-team.ru/TRAVELAPP-1583
	addHotelOnlyTopImages(mainParams.ImageOnlyTop, queryParams)

	if len(req.Filters.Atoms) > 0 {
		queryParams.Add("filter_atoms", strings.Join(req.Filters.Atoms, ","))
	}
	if req.PriceFilter.PriceMin > 0 {
		queryParams.Add("filter_price_from", strconv.Itoa(int(req.PriceFilter.PriceMin)))
	}
	if req.PriceFilter.PriceMax > 0 {
		queryParams.Add("filter_price_to", strconv.Itoa(int(req.PriceFilter.PriceMax)))
	}

	if req.Sorter.SorterID != "" {
		queryParams.Add("selected_sort_id", req.Sorter.SorterID)
	}

	addAnalyticsParams(req.AnalyticsParams, queryParams)

	return queryParams
}

type GetHotelInfoRequest struct {
	ReviewPagingParams *PagingParams
	ReviewSort         string
	ReviewPhrase       *PhraseReq
	SimilarLimit       int
	ImageParams        *HotelImageParams
	Params             *SearchParams
	QueryData          *GetHotelInfoQueryData
	AnalyticsParams    *AnalyticsParams
}

func (req *GetHotelInfoRequest) BuildURLParams() url.Values {
	queryParams := url.Values{}

	addHotelReviewSort(req.ReviewSort, queryParams)
	addHotelReviewKeyPhrase(req.ReviewPhrase, queryParams)
	addHotelReviewPaging(req.ReviewPagingParams, queryParams)
	addHotelImageParams(req.ImageParams, queryParams)
	//addTestUGC(env, queryParams)
	addHotelSearchParams(req.Params, queryParams)
	addHotelInfoParams(req.QueryData, queryParams)
	addAnalyticsParams(req.AnalyticsParams, queryParams)

	return queryParams
}

type HotelImageParams struct {
	PagingParams *PagingParams
	OnlyTop      bool
	Sizes        []string
}

type GetHotelInfoQueryData struct {
	HotelSlug string
	Permalink uint64
}

type GetHotelsCountersMainParams struct {
	CheckinDate         string
	CheckoutDate        string
	Adults              int
	ChildrenAge         []int
	SearchPagePollingID string
	Atoms               []string
	Bbox                *BoundingBox
	PriceFrom           uint32
	PriceTo             uint32
}

type GetCountersRequest struct {
	QueryData       GetHotelsCountersMainParams
	AnalyticsParams *AnalyticsParams
}

func (r GetCountersRequest) BuildURLParams() url.Values {
	queryParams := url.Values{}

	mainParams := r.QueryData
	queryParams.Add("checkin_date", mainParams.CheckinDate)
	queryParams.Add("checkout_date", mainParams.CheckoutDate)
	queryParams.Add("adults", strconv.Itoa(mainParams.Adults))
	queryParams.Add("bbox", mainParams.Bbox.String())
	if len(mainParams.ChildrenAge) > 0 {
		for i := 0; i < len(mainParams.ChildrenAge); i++ {
			queryParams.Add("children_ages", strconv.Itoa(mainParams.ChildrenAge[i]))
		}
	}
	if len(mainParams.Atoms) > 0 {
		atoms := strings.Join(mainParams.Atoms, ",")
		queryParams.Add("filter_atoms", atoms)
	}
	if mainParams.PriceFrom > 0 {
		queryParams.Add("filter_price_from", strconv.Itoa(int(mainParams.PriceFrom)))
	}
	if mainParams.PriceTo > 0 {
		queryParams.Add("filter_price_to", strconv.Itoa(int(mainParams.PriceTo)))
	}

	addAnalyticsParams(r.AnalyticsParams, queryParams)
	return queryParams
}

type GetCountersResponse struct {
	TotalHotels int        `json:"found_hotel_count"`
	FilterInfo  FilterInfo `json:"filter_info"`
}

type NavigationTokens struct {
	NextPage string `json:"next_page"`
}

type OfferSearchProgress struct {
	Finished         bool     `json:"finished"`
	FinishedPartners []string `json:"finished_partners"`
	PartnersTotal    uint32   `json:"partners_total"`
	PartnersComplete uint32   `json:"partners_complete"`
	PendingPartners  []string `json:"pending_partners"`
}

type HotelWithOffers struct {
	Hotel Hotel `json:"hotel"`
	// Завершён ли поллинг в этом отеле
	PollingFinished bool `json:"search_is_finished"`
	// Информация о предложении в отеле
	OffersInfo []OfferInfo `json:"offers"`
	// Доп инфармация (кешбек)
	Badges []HotelBadge `json:"badges"`
}

type Size struct {
	Identifier string `json:"size"`
	Width      uint32 `json:"width"`
	Height     uint32 `json:"height"`
}

type Image struct {
	URLTemplate string      `json:"url_template"`
	Sizes       []Size      `json:"sizes,omitempty"`
	ID          string      `json:"id"`
	Moderation  *Moderation `json:"moderation"`
	Tags        []string    `json:"tags,omitempty"`
}

type Price struct {
	Value    float64 `json:"value"`
	Currency string  `json:"currency"`
}

type PansionInfo struct {
	ID   string `json:"id"`
	Name string `json:"name"`
}

type PromoCampaigns struct {
	Mir2020    Mir2020    `json:"mir2020"`
	Taxi2020   Taxi2020   `json:"taxi2020"`
	WhiteLabel WhiteLabel `json:"white_label"`
	YandexEda  YandexEda  `json:"yandex_eda"`
	YandexPlus YandexPlus `json:"yandex_plus"`
}
type Mir2020 struct {
	CashbackAmount       int    `json:"cashback_amount"`
	CashbackAmountString string `json:"cashback_amount_string"`
	Eligible             bool   `json:"eligible"`
}
type Taxi2020 struct {
	Eligible bool `json:"eligible"`
}
type YandexEda struct {
	Data     Data `json:"data"`
	Eligible bool `json:"eligible"`
}
type YandexPlus struct {
	Eligible       bool   `json:"eligible"`
	Points         uint32 `json:"points"`
	WithdrawPoints int    `json:"withdraw_points"`
}
type YandexPlusMode string

const (
	YandexPlusModeTopup    YandexPlusMode = "TOPUP"
	YandexPlusModeWithdraw YandexPlusMode = "WITHDRAW"
)

type YandexPlusApplied struct {
	Mode   YandexPlusMode `json:"mode"`
	Points uint32         `json:"points"`
}

type OfferInfo struct {
	ID               string            `json:"id"`
	Name             string            `json:"name"`
	Price            Price             `json:"price"`
	PansionInfo      *PansionInfo      `json:"meal_type"`
	CancellationInfo *CancellationInfo `json:"cancellation_info"`
	YandexPlusInfo   *YandexPlus       `json:"offer_yandex_plus_info"`
	Badges           []HotelBadge      `json:"badges"`
	DiscountInfo     *DiscountInfo     `json:"discount_info"`
	RoomID           string            `json:"room_id"`
	OperatorID       string            `json:"operator_id"`
	LandingURL       string            `json:"landing_url"`
	Token            string            `json:"token"`
	YandexOffer      bool              `json:"yandex_offer"`
}

type HotelBadge struct {
	ID                  string              `json:"id"`
	Text                string              `json:"text"`
	AdditionalPromoInfo AdditionalPromoInfo `json:"additional_promo_info"`
}

type Rubric struct {
	ID   string `json:"id"`
	Name string `json:"name"`
}

type GeoInfo struct {
	// Строковое представление геофичи (например, "1км от центра" или "Рядом аэропорт")
	Name string `json:"name"`
	// Идентификатор иконки для отрисовки
	Icon string `json:"icon"`
}

type FilterParams struct {
	FilterAtoms     []string `json:"filter_atoms"`
	FilterPriceFrom *int     `json:"filter_price_from"`
	FilterPriceTo   *int     `json:"filter_price_to"`
}

type PriceFilter struct {
	Currency         string `json:"currency"`
	HistogramBounds  []int  `json:"histogram_bounds"`
	HistogramCounts  []int  `json:"histogram_counts"`
	MaxPriceEstimate int    `json:"max_price_estimate"`
	MinPriceEstimate int    `json:"min_price_estimate"`
}

type QuickFilters struct {
	ID string
	// Название.
	Name string
	// Подсказка
	Hint string
	// Эффект, как отображать этот фильтр.
	// Примеры: подкраска рейтинга, спец-значок для "только спецпредложения".
	// Примеры: для быстрого фильтра -- какой-нибудь спец-цвет или другая штука.
	Effect string
	// Если enabled == true, то можно на контрол нажимать.
	// Если enabled == false, то нажимать нельзя (будет пустая выдача).
	Enabled  bool
	AtomsOn  []string `json:"atoms_on"`
	AtomsOff []string `json:"atoms_off"`
}

type SortInfo struct {
	SelectedSortID   string             `json:"selected_sort_id"`
	QuickSorterGroup []QuickSorterGroup `json:"available_sort_type_groups"`
}

type QuickSorterGroup struct {
	SortTypes []QuickSorter `json:"sort_types"`
}

type QuickSorter struct {
	ID   string `json:"id"`
	Name string `json:"name"`
}

type DetailedFilter struct {
	ID      string
	Name    string
	Hint    string
	Effect  string
	Enabled bool
	Atoms   []string
}

type Filter struct {
	ID    string           `json:"id,omitempty"`
	Name  string           `json:"name,omitempty"`
	Type  string           `json:"type,omitempty"`
	Items []DetailedFilter `json:"items,omitempty"`
}

type CustomQuickFilter string

const (
	CustomQuickFilterGroupIDPansion             CustomQuickFilter = "pansion"
	CustomQuickFilterGroupIDPansionAllInclusive CustomQuickFilter = "all_inclusive"
	CustomQuickFilterGroupIDPansionBreakfast    CustomQuickFilter = "breakfast_included"

	CustomQuickFilterGroupIDStars  CustomQuickFilter = "stars"
	CustomQuickFilterGroupIDStars5 CustomQuickFilter = "stars-5"

	CustomQuickFilterGroupIDOptions             CustomQuickFilter = "options-not-mir-no-black-friday"
	CustomQuickFilterGroupIDOptionsCancellation CustomQuickFilter = "free_cancellation"
)

type FilterAggregation struct {
	Type           string  `json:"type"`
	DetailedFilter *Filter `json:"detailed_filters,omitempty"`
}

type FilterAggregationBatches struct {
	Filters []FilterAggregation `json:"items"`
}

type FilterInfo struct {
	PriceFilter           PriceFilter                `json:"price_filter"`
	Params                FilterParams               `json:"params"`
	QuickFilters          []QuickFilters             `json:"quick_filters"`
	DetailedFilter        []FilterAggregation        `json:"detailed_filters,omitempty"`
	DetailedFilterBatches []FilterAggregationBatches `json:"detailed_filters_batches,omitempty"`
}

type Permalink interface{}

type HotelBase struct {
	// идентификатор отеля, может быть числом или строкой
	Permalink Permalink `json:"permalink"`
	//Нужен для формирорования ссылки)
	Slug string `json:"hotel_slug"`
	// Название отеля
	Name string `json:"name"`
	//координаты отеля
	Coordinates Coordinates `json:"coordinates"`
	// Адрес отеля
	Address string `json:"address"`
	// Количество звезд. От null до 5
	Stars *int `json:"stars"`
	// Рейтинг. От 1 до 5. Дробное число, 1 знак после запятой.
	Rating        float64 `json:"rating"`
	IsYandexHotel bool    `json:"is_yandex_hotel"`
}
type Hotel struct {
	HotelBase
	// Рубрика отеля в Справочнике
	Rubric Rubric `json:"category"`
	// Количество фотографий
	TotalImageCount uint32 `json:"total_image_count"`
	// Фичи отеля из справочника
	Features []Feature `json:"main_amenities"`
	// Фичи отеля
	FeaturesGroups []FeaturesGroup `json:"amenity_groups,omitempty"`
	//Информация о гео (близость моря, центра)
	GeoInfo *GeoInfo `json:"geo_feature"`
	// Информация о ближайших остановках транспорта. Отсутствует для похожих отелей.
	NearestStations []TransportStation `json:"nearest_stations"`
	// Приблизительное расстояние от пользователя до отеля (в метрах). Передаётся только при сортировке по расстоянию.
	DistanceMeters float64 `json:"distance_meters"`
	// Приблизительное расстояние от пользователя до отеля (в виде текста с единицей измерения).
	// Передаётся только при сортировке по расстоянию.
	DistanceText string `json:"distance_text"`
	// Количество отзывов
	TotalTextReviewCount int32 `json:"total_text_review_count"`
	// Добавлен в избранное
	IsFavorite bool `json:"is_favorite"`
	// Информация об изображениях
	Images []Image
	//Ссылка на отель
	HotelURL string
}
type HotelCheckout struct {
	HotelBase
	Breadcrumbs      Breadcrumbs `json:"breadcrumbs"`
	ImageURLTemplate string      `json:"image_url_template"`
	LocationType     string      `json:"location_type"`
	Phone            string      `json:"phone"`
	WorkingHours     string      `json:"working_hours"`
	LegalInfo        LegalInfo   `json:"legal_info"`
}
type HotelPartner struct {
	Address      Address            `json:"address"`
	Amenities    map[string]Feature `json:"amenities"`
	Checkin      Checkin            `json:"checkin"`
	Checkout     Checkout           `json:"checkout"`
	Descriptions Descriptions       `json:"descriptions"`
	Fees         Fees               `json:"fees"`
	Images       []ImagesLinks      `json:"images"`
	Location     Location           `json:"location"`
	Name         string             `json:"name"`
	Phone        string             `json:"phone"`
	Policies     Policies           `json:"policies"`
	PropertyID   string             `json:"property_id"`
	Ratings      Ratings            `json:"ratings"`
}
type LegalInfo struct {
	Inn     string `json:"inn"`
	Ogrn    string `json:"ogrn"`
	Address string `json:"address"`
}

type SearchHotelsResponse struct {
	//Необходим для следующих запросов на поллинг
	SearchContext string `json:"context"`
	//Ограничивающий прямоугольник для карты.
	//Если пришел в запросе, то отдаем такой же; иначе вычисляем сами и отдаем фронту.
	Bbox BoundingBoxRsp `json:"bbox_as_struct"`
	//Навигационные токены.
	NavigationTokens NavigationTokens `json:"navigation_tokens"`
	//Информация о прогрессе поиска
	OfferSearchProgress OfferSearchProgress `json:"offer_search_progress"`
	//Рекомендуемая пауза перед следующей итерацией поллинга
	NextPollingRequestDelayMs int `json:"next_polling_request_delay_ms"`
	//Количество отелей для текста в фильтрах "Искать цены в X отелях"
	FoundHotelsCount int `json:"found_hotel_count"`
	//Найденные отели
	Hotels []HotelWithOffers `json:"hotels"`
	//Фильтры
	FilterInfo FilterInfo `json:"filter_info"`
	//Сортировки
	SortInfo SortInfo `json:"sort_info"`
	//Ответ необходимо отбросить, если
	//Если ответ относится к более старой эпохе.
	//Если ответ относится к более старой итерации в текущей эпохе.
	PollEpoch int
	//Идентификатор поллинга
	PollingSearchID string `json:"search_page_polling_id"`
}

type GetHotelInfoResponse struct {
	Breadcrumbs             Breadcrumbs             `json:"breadcrumbs"`
	ExtraVisitAndUserParams ExtraVisitAndUserParams `json:"extra_visit_and_user_params"`
	Hotel                   Hotel                   `json:"hotel"`
	HotelDescription        *HotelDescription       `json:"hotel_description,omitempty"`
	OffersInfo              OffersInfo              `json:"offers_info"`
	ParentRequestID         string                  `json:"parent_request_id"`
	RatingsInfo             *RatingsInfo            `json:"ratings_info,omitempty"`
	ReviewsInfo             *ReviewsInfo            `json:"reviews_info,omitempty"`
	SearchParams            *OfferSearchParams      `json:"search_params,omitempty"`
	SeoBreadcrumbs          SeoBreadcrumbs          `json:"seo_breadcrumbs"`
	SeoInfo                 SeoInfo                 `json:"seo_info"`
	SimilarHotelsInfo       SimilarHotelsInfo       `json:"similar_hotels_info"`
}

type Linguistics struct {
	AblativeCase      string `json:"ablative_case"`
	AccusativeCase    string `json:"accusative_case"`
	DativeCase        string `json:"dative_case"`
	DirectionalCase   string `json:"directional_case"`
	GenitiveCase      string `json:"genitive_case"`
	InstrumentalCase  string `json:"instrumental_case"`
	LocativeCase      string `json:"locative_case"`
	NominativeCase    string `json:"nominative_case"`
	Preposition       string `json:"preposition"`
	PrepositionalCase string `json:"prepositional_case"`
}
type GeoRegions struct {
	GeoID       int         `json:"geo_id"`
	Linguistics Linguistics `json:"linguistics"`
	Slug        string      `json:"slug"`
	Type        int         `json:"type"`
}
type Items struct {
	BreadcrumbType string `json:"breadcrum_type"`
	GeoRegions
}
type Breadcrumbs struct {
	GeoRegions []GeoRegions `json:"geo_regions"`
	Items      []Items      `json:"items"`
}
type AdditionalProps struct {
	GeoID string `json:"geo_id"`
}
type VisitHotels struct {
	BookingPage   AdditionalProps `json:"booking_page"`
	HotelPage     AdditionalProps `json:"hotel_page"`
	SearchPage    AdditionalProps `json:"search_page"`
	SeoRegionPage AdditionalProps `json:"seoRegion_page"`
}
type VisitParams struct {
	Hotels VisitHotels `json:"hotels"`
}
type ExtraVisitAndUserParams struct {
	VisitParams VisitParams `json:"visit_params"`
}
type Category struct {
	ID   string `json:"id"`
	Name string `json:"name"`
}
type GeoFeature struct {
	Icon string `json:"icon"`
	ID   string `json:"id"`
	Name string `json:"name"`
}
type Moderation struct {
	Status string `json:"status"`
}
type MetroLine struct {
	Color string `json:"color"`
	ID    string `json:"id"`
	Name  string `json:"name"`
}
type TransportStation struct {
	ID string `json:"id"`
	// METRO|OTHER
	Type         string `json:"type"`
	Name         string `json:"name"`
	DistanceText string `json:"distance_text"`
	//Coordinates    string    `json:"coordinates"`
	//DistanceMeters int       `json:"distance_meters"`
	//MetroLine      MetroLine `json:"metro_line"`
}
type SafetyMeasures struct {
	ID   string `json:"id"`
	Name string `json:"name"`
}
type HotelDescription struct {
	Text string `json:"text"`
}
type MaxPrice struct {
	Price
}
type MinPrice struct {
	Price
}
type AggregatedOfferInfo struct {
	CancellationInfoAggregate *string  `json:"cancellation_info_aggregate,omitempty"`
	MaxPrice                  MaxPrice `json:"max_price"`
	MinPrice                  MinPrice `json:"min_price"`
	PansionAggregate          *string  `json:"pansion_aggregate,omitempty"`
}
type Link struct {
	Text string `json:"text"`
	URL  string `json:"url"`
}
type AdditionalPromoInfo struct {
	Link  Link   `json:"link"`
	Text  string `json:"text"`
	Title string `json:"title"`
}
type RefundRules struct {
	EndsAt   *time.Time `json:"ends_at"`
	Penalty  *Price     `json:"penalty"`
	StartsAt *time.Time `json:"starts_at"`
	Type     string     `json:"type"`
}
type CancellationInfo struct {
	HasFreeCancellation bool          `json:"has_free_cancellation"`
	RefundRules         []RefundRules `json:"refund_rules"`
	RefundType          string        `json:"refund_type"`
}
type DiscountInfo struct {
	Percent            int    `json:"percent"`
	Reason             string `json:"reason"`
	ShowDiscountInfo   bool   `json:"show_discount_info"`
	StrikethroughPrice Price  `json:"strikethrough_price"`
}
type MealType struct {
	ID   string `json:"id"`
	Name string `json:"name"`
}

type OperatorInfo struct {
	BookOnYandex bool   `json:"book_on_yandex"`
	GreenURL     string `json:"green_url"`
	IconURL      string `json:"icon_url"`
	ID           string `json:"id"`
	Name         string `json:"name"`
}
type OperatorByID map[string]OperatorInfo

type PartnerOffers struct {
	CancellationInfoAggregate    *string           `json:"cancellation_info_aggregate"`
	DefaultOffer                 OfferInfo         `json:"default_offer"`
	DefaultOfferCancellationInfo *CancellationInfo `json:"default_offer_cancellation_info"`
	DefaultOfferPansion          *PansionInfo      `json:"default_offer_pansion"`
	OperatorID                   string            `json:"operator_id"`
	PansionAggregate             *string           `json:"pansion_aggregate"`
}
type FeaturePartner struct {
	//может быть и числом и с плавающей точкой
	//ID   interface{} `json:"id"`
	Name string `json:"name"`
}
type Feature struct {
	Icon *string `json:"icon,omitempty"`
	ID   string  `json:"id"`
	Name string  `json:"name"`
}
type FeaturesGroup struct {
	Feature
	Features []Feature `json:"amenities"`
}

type Area struct {
	Unit  string `json:"unit"`
	Value int    `json:"value"`
}
type Configuration struct {
	Icon              string `json:"icon"`
	ID                string `json:"id"`
	NameInflectedForm string `json:"name_inflected_form"`
	NameInitialForm   string `json:"name_initial_form"`
	Quantity          int    `json:"quantity"`
}
type BedGroups struct {
	Configuration []Configuration `json:"configuration"`
	ID            string          `json:"id"`
}
type BedGroupsCheckout struct {
	Description string `json:"description"`
	ID          string `json:"id"`
}

type Room struct {
	AmenityGroups             []FeaturesGroup `json:"amenity_groups"`
	Area                      Area            `json:"area"`
	Badges                    []HotelBadge    `json:"badges"`
	BedGroups                 []BedGroups     `json:"bed_groups"`
	CancellationInfoAggregate string          `json:"cancellation_info_aggregate"`
	Description               string          `json:"description"`
	ID                        string          `json:"id"`
	Images                    []Image         `json:"images"`
	MainAmenities             []Feature       `json:"main_amenities"`
	Name                      string          `json:"name"`
	PansionAggregate          string          `json:"pansion_aggregate"`
}
type RoomPartner struct {
	Amenities    map[string]FeaturePartner `json:"amenities"`
	Descriptions Descriptions              `json:"descriptions"`
	Images       []ImagesLinks             `json:"images"`
	Name         string                    `json:"name"`
}
type OffersInfo struct {
	AggregatedOfferInfo       AggregatedOfferInfo `json:"aggregated_offer_info"`
	BannerType                string              `json:"banner_type"`
	DefaultOffer              OfferInfo           `json:"default_offer"`
	GroupBy                   string              `json:"group_by"`
	MainOffers                []OfferInfo         `json:"main_offers"`
	NextPollingRequestDelayMs int                 `json:"next_polling_request_delay_ms"`
	OfferCount                int                 `json:"offer_count"`
	OfferSearchProgress       OfferSearchProgress `json:"offer_search_progress"`
	OperatorByID              OperatorByID        `json:"operator_by_id"`
	OperatorCount             int                 `json:"operator_count"`
	PartnerOffers             []PartnerOffers     `json:"partner_offers"`
	Rooms                     []Room              `json:"rooms"`
}
type FeatureRatings struct {
	ID              string `json:"id"`
	Name            string `json:"name"`
	PositivePercent int    `json:"positive_percent"`
}
type RatingsInfo struct {
	FeatureRatings []FeatureRatings `json:"feature_ratings"`
	Teaser         string           `json:"teaser"`
}
type KeyPhrases struct {
	Name        string `json:"name"`
	ReviewCount int    `json:"review_count"`
}
type Author struct {
	AvatarURLTemplate string `json:"avatar_url"`
	Level             string `json:"level"`
	Name              string `json:"name"`
	ProfileURL        string `json:"profile_url"`
}
type Fragments struct {
	Position int `json:"position"`
	Size     int `json:"size"`
}
type KeyPhraseMatch struct {
	Fragments []Fragments `json:"fragments"`
}
type TextReview struct {
	Author            Author          `json:"author"`
	BusinessComment   string          `json:"business_comment"`
	CommentCount      int             `json:"comment_count"`
	ID                string          `json:"id"`
	Images            []Image         `json:"images,omitempty"`
	KeyPhraseMatch    *KeyPhraseMatch `json:"key_phrase_match"`
	Moderation        *Moderation     `json:"moderation"`
	Rating            int             `json:"rating"`
	Snippet           string          `json:"snippet"`
	Text              string          `json:"text"`
	TotalDislikeCount int             `json:"total_dislike_count"`
	TotalLikeCount    int             `json:"total_like_count"`
	UpdatedAt         string          `json:"updated_at"`
	UserReaction      string          `json:"user_reaction"`
}

type ReviewsInfo struct {
	HasMore              bool         `json:"has_more"`
	KeyPhrases           []KeyPhrases `json:"key_phrases,omitempty"`
	TextReviews          []TextReview `json:"text_reviews,omitempty"`
	TotalKeyPhraseCount  int          `json:"total_key_phrase_count"`
	TotalTextReviewCount int          `json:"total_text_review_count"`
	UserTextReview       *TextReview  `json:"user_text_review"`
}

// TODO(adurnev) обьединить?
type OfferSearchParams struct {
	Adults       int   `json:"adults"`
	CheckinDate  Date  `json:"checkin_date"`
	CheckoutDate Date  `json:"checkout_date"`
	ChildrenAges []int `json:"children_ages"`
}
type SearchParams struct {
	Adults       int    `json:"adults"`
	CheckinDate  string `json:"checkin_date"`
	CheckoutDate string `json:"checkout_date"`
	ChildrenAges []int  `json:"children_ages"`
}
type RequestInfo struct {
	SearchParams
	NumAdults             int `json:"num_adults"`
	SelectedBedGroupIndex int `json:"selected_bed_group_index"`
}
type SeoBreadcrumbs struct {
	Breadcrumbs
}
type ImageSize struct {
	Height int `json:"height"`
	Width  int `json:"width"`
}
type OpenGraph struct {
	Description string    `json:"description"`
	Image       string    `json:"image"`
	ImageSize   ImageSize `json:"image_size"`
	Title       string    `json:"title"`
}
type SchemaOrg struct {
	Address     string  `json:"address"`
	Image       string  `json:"image"`
	Name        string  `json:"name"`
	PriceRange  string  `json:"price_range"`
	RatingValue float32 `json:"rating_value"`
	ReviewCount int     `json:"review_count"`
}
type SeoInfo struct {
	Description string    `json:"description"`
	OpenGraph   OpenGraph `json:"open_graph"`
	SchemaOrg   SchemaOrg `json:"schema_org"`
	Title       string    `json:"title"`
}

type SimilarHotelsInfo struct {
	Hotels                    []HotelWithOffers   `json:"hotels"`
	NextPollingRequestDelayMs int                 `json:"next_polling_request_delay_ms"`
	OfferSearchProgress       OfferSearchProgress `json:"offer_search_progress"`
	OperatorByID              OperatorByID        `json:"operator_by_id"`
}

type PagingParams struct {
	Offset int
	Limit  int
}

type GetHotelImagesRequest struct {
	QueryData       *GetHotelInfoQueryData
	ParentRequestID string
	ImageParams     *HotelImageParams
}

func (req *GetHotelImagesRequest) BuildURLParams() url.Values {
	queryParams := url.Values{}

	addHotelInfoParams(req.QueryData, queryParams)
	addParentRequestID(req.ParentRequestID, queryParams)
	addHotelImageParams(req.ImageParams, queryParams)

	return queryParams
}

type GetHotelImagesResponse struct {
	TotalImageCount int32   `json:"total_image_count"`
	Images          []Image `json:"images"`
}

type GetHotelOffersRequest struct {
	Params              *SearchParams
	QueryData           *GetHotelInfoQueryData
	AnalyticsParams     *AnalyticsParams
	SearchPagePollingID string
	//TODO(adurnev) не заполняется
	ParentRequestID string
}

func (req *GetHotelOffersRequest) BuildURLParams() url.Values {
	queryParams := url.Values{}

	addHotelSearchParams(req.Params, queryParams)
	addHotelInfoParams(req.QueryData, queryParams)
	if req.SearchPagePollingID != "" {
		queryParams.Add("search_page_polling_id", req.SearchPagePollingID)
	}
	addParentRequestID(req.ParentRequestID, queryParams)
	addAnalyticsParams(req.AnalyticsParams, queryParams)

	return queryParams
}

type GetHotelOffersResponse struct {
	//ExtraVisitAndUserParams ExtraVisitAndUserParams `json:"extra_visit_and_user_params"`
	OffersInfo OffersInfo `json:"offers_info"`
}

type GetSimilarHotelsRequest struct {
	Params              *SearchParams
	QueryData           *GetHotelInfoQueryData
	AnalyticsParams     *AnalyticsParams
	SearchPagePollingID string
	ParentRequestID     string
	SimilarHotelLimit   int32
}

func (req *GetSimilarHotelsRequest) BuildURLParams() url.Values {
	queryParams := url.Values{}

	addHotelSearchParams(req.Params, queryParams)
	addHotelInfoParams(req.QueryData, queryParams)
	if req.SearchPagePollingID != "" {
		queryParams.Add("search_page_polling_id", req.SearchPagePollingID)
	}
	addParentRequestID(req.ParentRequestID, queryParams)
	if req.SimilarHotelLimit > 0 {
		queryParams.Add("similar_hotels_limit", strconv.Itoa(int(req.SimilarHotelLimit)))
	}
	addAnalyticsParams(req.AnalyticsParams, queryParams)

	return queryParams
}

type GetSimilarHotelsResponse struct {
	SimilarHotelsInfo SimilarHotelsInfo `json:"similar_hotels_info"`
}

type CreateOrderBase struct {
	AppliedPromoCampaigns PromoCampaignsApplied `json:"applied_promo_campaigns"`
	Checksum              string                `json:"checksum"`
	CustomerIP            string                `json:"customer_ip"`
	CustomerLogin         string                `json:"customer_login"`
	CustomerPassportID    string                `json:"customer_passport_id"`
	CustomerUserAgent     string                `json:"customer_user_agent"`
	CustomerYandexUID     string                `json:"customer_yandex_uid"`
	Label                 string                `json:"label"`
	PromoCodes            []string              `json:"promo_codes"`
	SelectedBedGroupIndex int                   `json:"selected_bed_group_index"`
	SessionKey            string                `json:"session_key"`
	Token                 string                `json:"token"`
}

type CreateOrderRequest struct {
	CreateOrderBase
	AllowsSubscription      bool               `json:"allows_subscription"`
	CustomerEmail           string             `json:"customer_email"`
	CustomerPhone           string             `json:"customer_phone"`
	Guests                  []Guest            `json:"guests"`
	PaymentTestContextToken string             `json:"payment_test_context_token"`
	SubscriptionParams      SubscriptionParams `json:"subscription_params,omitempty"`
	UseDeferredPayments     bool               `json:"use_deferred_payments"`
}
type PromoCampaignsApplied struct {
	WhiteLabel *WhiteLabelApplied `json:"white_label,omitempty"`
	YandexPlus *YandexPlusApplied `json:"yandex_plus,omitempty"`
}
type WhiteLabelApplied struct {
	CustomerNumber string `json:"customer_number"`
}
type Guest struct {
	Empty     bool   `json:"empty"`
	FirstName string `json:"first_name"`
	LastName  string `json:"last_name"`
}
type SubscriptionParams struct {
	Language        string `json:"language"`
	NationalVersion string `json:"national_version"`
	Subscribe       bool   `json:"subscribe"`
	Timezone        string `json:"timezone"`
}
type CreateOrderResponse struct {
	//TODO(adurnev) контаркт из другого ПР (https://a.yandex-team.ru/review/2709636/files/3#file-0-193505581) если id мало
	ID string `json:"id"`
}

type EstimateDiscountRequest struct {
	CreateOrderBase
}

type EstimateDiscountResponse struct {
	CodeApplicationResults []CodeApplicationResult `json:"code_application_results"`
	OriginalAmount         DiscountAmount          `json:"original_amount"`
	DiscountedAmount       DiscountAmount          `json:"discounted_amount"`
	DiscountAmount         DiscountAmount          `json:"discount_amount"`
	//TODO(adurnev) нужны контракты из другого ПР
	//promoCampaigns.yandexPlus
	//deferredPaymentSchedule
}

type CodeApplicationResult struct {
	Code           string         `json:"code"`
	DiscountAmount DiscountAmount `json:"discount_amount"`
	//https://a.yandex-team.ru/arcadia/travel/frontend/spec/hotels/booking-flow/estimate_discount/models.ts
	Type string `json:"type"`
}

//TODO(adurnev) использовать Amount из другого ПР
type DiscountAmount struct {
	Currency string `json:"currency"`
	Value    int    `json:"value"`
}

func (d *Date) UnmarshalJSON(b []byte) (err error) {
	s := strings.Trim(string(b), "\"")
	if s == "null" {
		d.Time = time.Time{}
		return
	}
	t, err := common.ParseDate(s)
	if err != nil {
		return err
	}
	d.Time = t
	return nil
}

func (d *Date) MarshalJSON() ([]byte, error) {
	if d == nil {
		return nil, nil
	}
	return []byte(fmt.Sprintf("%q", common.FormatDate(d.Time))), nil
}

func addAnalyticsParams(analyticsParams *AnalyticsParams, queryParams url.Values) {
	if analyticsParams == nil {
		return
	}
	if analyticsParams.UtmSource != nil {
		queryParams.Add("utm_source", *analyticsParams.UtmSource)
	}
	if analyticsParams.UtmMedium != nil {
		queryParams.Add("utm_medium", *analyticsParams.UtmMedium)
	}
	if analyticsParams.UtmCampaign != nil {
		queryParams.Add("utm_campaign", *analyticsParams.UtmCampaign)
	}
	if analyticsParams.UtmTerm != nil {
		queryParams.Add("utm_term", *analyticsParams.UtmTerm)
	}
	if analyticsParams.UtmContent != nil {
		queryParams.Add("utm_content", *analyticsParams.UtmContent)
	}
}

func addHotelSearchParams(searchParams *SearchParams, queryParams url.Values) {
	if searchParams == nil {
		return
	}
	if searchParams.CheckinDate != "" {
		queryParams.Add("checkin_date", searchParams.CheckinDate)
	}
	if searchParams.CheckoutDate != "" {
		queryParams.Add("checkout_date", searchParams.CheckoutDate)
	}
	if searchParams.Adults > 0 {
		queryParams.Add("adults", strconv.Itoa(searchParams.Adults))
	}
	if len(searchParams.ChildrenAges) > 0 {
		for i := 0; i < len(searchParams.ChildrenAges); i++ {
			queryParams.Add("children_ages", strconv.Itoa(searchParams.ChildrenAges[i]))
		}
	}
}

func addHotelInfoParams(req *GetHotelInfoQueryData, queryParams url.Values) {
	if req == nil {
		return
	}
	if req.Permalink > 0 {
		queryParams.Add("permalink", strconv.Itoa(int(req.Permalink)))
	} else if req.HotelSlug != "" {
		queryParams.Add("hotel_slug", req.HotelSlug)
	}
}

func addHotelOnlyTopImages(onlyTop bool, queryParams url.Values) {
	queryParams.Add("only_top_images", strconv.FormatBool(onlyTop))
}

func addHotelImageSizes(sizes []string, queryParams url.Values) {
	for _, s := range sizes {
		queryParams.Add("image_sizes", s)
	}
}

func addHotelImagePaging(params *PagingParams, queryParams url.Values) {
	if params != nil {
		queryParams.Add("image_limit", strconv.Itoa(params.Limit))
		queryParams.Add("image_offset", strconv.Itoa(params.Offset))
	}
}

func addHotelImageParams(params *HotelImageParams, queryParams url.Values) {
	if params != nil {
		addHotelImagePaging(params.PagingParams, queryParams)
		addHotelOnlyTopImages(params.OnlyTop, queryParams)
		addHotelImageSizes(params.Sizes, queryParams)
	}
}

func addParentRequestID(reqID string, queryParams url.Values) {
	if reqID != "" {
		queryParams.Add("parent_request_id", reqID)
	}
}

func addHotelReviewSort(sort string, queryParams url.Values) {
	if sort != "" {
		queryParams.Add("text_review_ranking", sort)
	}
}

func addHotelReviewKeyPhrase(phrase *PhraseReq, queryParams url.Values) {
	if phrase != nil {
		if phrase.Filter != "" {
			queryParams.Add("key_phrase_filter", phrase.Filter)
		}
		if phrase.Limit > 0 {
			queryParams.Add("key_phrase_limit", strconv.Itoa(phrase.Limit))
		}
	}
}

func addHotelReviewPaging(params *PagingParams, queryParams url.Values) {
	if params != nil {
		queryParams.Add("text_review_limit", strconv.Itoa(params.Limit))
		queryParams.Add("text_review_offset", strconv.Itoa(params.Offset))
	}
}
