package travelersclient

import (
	"context"
	"fmt"
	"io"
	"net/url"
	"strings"
	"time"

	"github.com/go-resty/resty/v2"
	"github.com/opentracing/opentracing-go"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/ctxlog"
	"a.yandex-team.ru/library/go/core/metrics"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/app/backend/internal/lib/clientscommon"
)

const (
	serviceTicketHeader = "X-Ya-Service-Ticket"
	userTicketHeader    = "X-Ya-User-Ticket"
)

type UserTicketGetter func(context.Context) (string, error)

type ServiceTicketGetter func(context.Context) (string, error)

type HTTPClient struct {
	logger              log.Logger
	config              *Config
	httpClient          *resty.Client
	userTicketGetter    UserTicketGetter
	serviceTicketGetter ServiceTicketGetter
	metrics             *clientscommon.HTTPClientMetrics
}

func NewHTTPClient(
	logger log.Logger,
	config *Config,
	userTicketGetter UserTicketGetter,
	serviceTicketGetter ServiceTicketGetter,
	logRequestAndResponse bool,
	metricsRegistry metrics.Registry,
) *HTTPClient {
	client := resty.New().SetTimeout(config.Timeout).SetLogger(logger).OnRequestLog(clientscommon.DoNotLogTVMHeaders)
	if logRequestAndResponse {
		client.Debug = true // Влияет только на логирование запроса и ответа
	}
	m := clientscommon.NewHTTPClientMetrics(metricsRegistry, "travelers")
	return &HTTPClient{
		logger:              logger,
		config:              config,
		httpClient:          client,
		userTicketGetter:    userTicketGetter,
		serviceTicketGetter: serviceTicketGetter,
		metrics:             m,
	}
}

func (c *HTTPClient) doExecute(ctx context.Context, method, path string, body, result interface{}, queryParams url.Values, requireUserTicket bool) error {
	endpoint := c.config.BaseURL + path

	serviceTicket, err := c.serviceTicketGetter(ctx)
	if err != nil {
		return NoServiceTicketError.Wrap(err)
	}

	var userTicket string
	if requireUserTicket {
		userTicket, err = c.userTicketGetter(ctx)
		if err != nil {
			return NoUserTicketError.Wrap(err)
		}
	}
	var errResponse map[string]interface{}
	r := c.httpClient.R().
		SetContext(ctx).
		SetHeader(serviceTicketHeader, serviceTicket).
		SetBody(body).
		SetError(&errResponse)
	if requireUserTicket {
		r = r.SetHeader(userTicketHeader, userTicket)
	}
	if result != nil {
		r = r.SetResult(result)
	}
	if queryParams != nil {
		r = r.SetQueryParamsFromValues(queryParams)
	}

	if span := opentracing.SpanFromContext(ctx); span != nil {
		err := opentracing.GlobalTracer().Inject(
			span.Context(),
			opentracing.HTTPHeaders,
			opentracing.HTTPHeadersCarrier(r.Header))
		if err != nil {
			err := xerrors.Errorf("err is %s", err.Error())
			ctxlog.Error(ctx, c.logger, "unable attach tracing info to request.", log.Error(err))
		}
	}

	response, err := r.Execute(method, endpoint)
	c.metrics.StoreCallResult(method, path, response)
	if err != nil {
		return ResponseError.Wrap(err)
	}
	if !response.IsSuccess() {
		raw, _ := io.ReadAll(response.RawResponse.Body)
		return xerrors.Errorf("unexpected response from travelers service: %w", StatusError{
			Status:      response.StatusCode(),
			Response:    errResponse,
			ResponseRaw: string(raw),
		})
	}
	return nil
}

func (c *HTTPClient) execute(ctx context.Context, method, path string, body, result interface{}, queryParams url.Values) error {
	span, ctx := opentracing.StartSpanFromContext(ctx, path)
	defer span.Finish()
	return c.doExecute(ctx, method, path, body, result, queryParams, true)
}

func (c *HTTPClient) GetVersion(ctx context.Context) (string, error) {
	var res string
	if err := c.doExecute(ctx, resty.MethodGet, "/version", nil, &res, nil, false); err != nil {
		ctxlog.Debug(ctx, c.logger, "GetVersion", log.Error(err))
		return "", xerrors.Errorf("unable to get version: %w", err)
	}
	ctxlog.Debug(ctx, c.logger, "GetVersion")
	return res, nil
}

func (c *HTTPClient) ListDocumentTypes(ctx context.Context) (*DocumentTypes, error) {
	documentTypes := DocumentTypes{}
	if err := c.doExecute(ctx, resty.MethodGet, "/document_types", nil, &documentTypes, nil, false); err != nil {
		ctxlog.Debug(ctx, c.logger, "ListDocumentTypes", log.Error(err))
		return nil, xerrors.Errorf("unable to get document types: %w", err)
	}
	ctxlog.Debug(ctx, c.logger, "ListDocumentTypes")
	return &documentTypes, nil
}

func (c *HTTPClient) GetTraveler(ctx context.Context, uid string) (*Traveler, error) {
	path := fmt.Sprintf("/travelers/%s", url.PathEscape(uid))
	var traveler Traveler
	if err := c.execute(ctx, resty.MethodGet, path, nil, &traveler, nil); err != nil {
		ctxlog.Debug(ctx, c.logger, "GetTraveler", log.String("uid", uid), log.Error(err))
		return nil, err
	}
	ctxlog.Debug(ctx, c.logger, "GetTraveler", log.String("uid", uid))
	return &traveler, nil
}

func (c *HTTPClient) CreateOrUpdateTraveler(ctx context.Context, uid string, traveler *EditableTraveler) (*Traveler, error) {
	path := fmt.Sprintf("/travelers/%s", url.PathEscape(uid))
	var resp Traveler
	if err := c.execute(ctx, resty.MethodPost, path, traveler, &resp, nil); err != nil {
		err = tryToValidationError(err)
		ctxlog.Debug(ctx, c.logger, "CreateOrUpdateTraveler", log.String("uid", uid), log.Error(err))
		return nil, err
	}
	ctxlog.Debug(ctx, c.logger, "CreateOrUpdateTraveler", log.String("uid", uid))
	return &resp, nil
}

func (c *HTTPClient) ListPassengers(ctx context.Context, uid string, includeCards bool, includeDocuments bool) ([]Passenger, error) {
	path := fmt.Sprintf("/travelers/%s/passengers", url.PathEscape(uid))
	var extraFields []string
	if includeCards {
		extraFields = append(extraFields, string(fieldBonusCards))
	}
	if includeDocuments {
		extraFields = append(extraFields, string(fieldDocuments))
	}
	var queryParams url.Values
	if len(extraFields) > 0 {
		queryParams = url.Values{
			fieldsParamName: {strings.Join(extraFields, ",")},
		}
	}
	var rawPassengers []rawPassenger
	if err := c.execute(ctx, resty.MethodGet, path, nil, &rawPassengers, queryParams); err != nil {
		ctxlog.Debug(ctx, c.logger, "ListPassengers", log.String("uid", uid), log.Error(err))
		return nil, err
	}
	passengers := make([]Passenger, 0, len(rawPassengers))
	for _, rp := range rawPassengers {
		passenger, err := convertRaw2Passenger(rp)
		if err != nil {
			return nil, InternalError.Wrap(err)
		}
		passengers = append(passengers, *passenger)
	}
	ctxlog.Debug(ctx, c.logger, "ListPassengers", log.String("uid", uid))
	return passengers, nil
}

func (c *HTTPClient) GetPassenger(ctx context.Context, uid string, id string, includeCards bool, includeDocuments bool) (*Passenger, error) {
	path := fmt.Sprintf("/travelers/%s/passengers/%s", url.PathEscape(uid), url.PathEscape(id))
	var extraFields []string
	if includeCards {
		extraFields = append(extraFields, string(fieldBonusCards))
	}
	if includeDocuments {
		extraFields = append(extraFields, string(fieldDocuments))
	}
	var queryParams url.Values
	if len(extraFields) > 0 {
		queryParams = url.Values{
			fieldsParamName: {strings.Join(extraFields, ",")},
		}
	}
	var rp rawPassenger
	if err := c.execute(ctx, resty.MethodGet, path, nil, &rp, queryParams); err != nil {
		ctxlog.Debug(ctx, c.logger, "GetPassenger", log.String("uid", uid), log.String("id", id), log.Error(err))
		return nil, err
	}
	passenger, err := convertRaw2Passenger(rp)
	if err != nil {
		return nil, InternalError.Wrap(err)
	}
	ctxlog.Debug(ctx, c.logger, "GetPassenger", log.String("uid", uid), log.String("id", id))
	return passenger, nil
}

func (c *HTTPClient) CreatePassenger(ctx context.Context, uid string, passenger *CreateOrUpdatePassengerRequest) (*Passenger, error) {
	path := fmt.Sprintf("/travelers/%s/passengers", url.PathEscape(uid))
	var createdRawPassenger rawPassenger
	if err := c.execute(ctx, resty.MethodPost, path, convertReq2RawPassenger(passenger), &createdRawPassenger, nil); err != nil {
		err = tryToValidationError(err)
		ctxlog.Debug(ctx, c.logger, "CreatePassenger", log.String("uid", uid), log.Error(err))
		return nil, err
	}
	createdPassenger, err := convertRaw2Passenger(createdRawPassenger)
	if err != nil {
		return nil, InternalError.Wrap(err)
	}
	ctxlog.Debug(ctx, c.logger, "CreatePassenger", log.String("uid", uid))
	return createdPassenger, nil
}

func (c *HTTPClient) UpdatePassenger(ctx context.Context, uid string, id string, passenger *CreateOrUpdatePassengerRequest) (*Passenger, error) {
	path := fmt.Sprintf("/travelers/%s/passengers/%s", url.PathEscape(uid), url.PathEscape(id))
	var updatedRawPassenger rawPassenger
	if err := c.execute(ctx, resty.MethodPut, path, convertReq2RawPassenger(passenger), &updatedRawPassenger, nil); err != nil {
		err = tryToValidationError(err)
		ctxlog.Debug(ctx, c.logger, "UpdatePassenger", log.String("uid", uid), log.String("id", id), log.Error(err))
		return nil, err
	}
	updatedPassenger, err := convertRaw2Passenger(updatedRawPassenger)
	if err != nil {
		return nil, InternalError.Wrap(err)
	}
	ctxlog.Debug(ctx, c.logger, "UpdatePassenger", log.String("uid", uid), log.String("id", id))
	return updatedPassenger, nil
}

func (c *HTTPClient) DeletePassenger(ctx context.Context, uid string, id string) error {
	path := fmt.Sprintf("/travelers/%s/passengers/%s", url.PathEscape(uid), url.PathEscape(id))
	if err := c.execute(ctx, resty.MethodDelete, path, nil, nil, nil); err != nil {
		ctxlog.Debug(ctx, c.logger, "DeletePassenger", log.String("uid", uid), log.String("id", id), log.Error(err))
		return err
	}
	ctxlog.Debug(ctx, c.logger, "DeletePassenger", log.String("uid", uid), log.String("id", id))
	return nil
}

func (c *HTTPClient) ListDocuments(ctx context.Context, uid string, passengerID string) ([]Document, error) {
	path := fmt.Sprintf("/travelers/%s/passengers/%s/documents", url.PathEscape(uid), url.PathEscape(passengerID))
	var rawDocs []rawDocument
	if err := c.execute(ctx, resty.MethodGet, path, nil, &rawDocs, nil); err != nil {
		ctxlog.Debug(ctx, c.logger, "ListDocuments", log.String("uid", uid), log.String("passengerID", passengerID))
		return nil, err
	}
	docs := make([]Document, 0, len(rawDocs))
	for _, rawDoc := range rawDocs {
		doc, err := convertRaw2Document(rawDoc)
		if err != nil {
			return nil, InternalError.Wrap(err)
		}
		docs = append(docs, *doc)
	}
	ctxlog.Debug(ctx, c.logger, "ListDocuments", log.String("uid", uid), log.String("passengerID", passengerID))
	return docs, nil
}

func (c *HTTPClient) GetDocument(ctx context.Context, uid string, passengerID string, documentID string) (*Document, error) {
	path := fmt.Sprintf("/travelers/%s/passengers/%s/documents/%s", url.PathEscape(uid), url.PathEscape(passengerID), url.PathEscape(documentID))
	var rawDoc rawDocument
	if err := c.execute(ctx, resty.MethodGet, path, nil, &rawDoc, nil); err != nil {
		ctxlog.Debug(ctx, c.logger, "GetDocument", log.String("uid", uid), log.String("passengerID", passengerID), log.String("documentID", documentID), log.Error(err))
		return nil, err
	}
	doc, err := convertRaw2Document(rawDoc)
	if err != nil {
		return nil, InternalError.Wrap(err)
	}
	ctxlog.Debug(ctx, c.logger, "GetDocument", log.String("uid", uid), log.String("passengerID", passengerID), log.String("documentID", documentID))
	return doc, nil
}

func (c *HTTPClient) CreateDocument(ctx context.Context, uid string, passengerID string, docReq *CreateOrUpdateDocumentRequest) (*Document, error) {
	path := fmt.Sprintf("/travelers/%s/passengers/%s/documents", url.PathEscape(uid), url.PathEscape(passengerID))
	var rawDoc rawDocument
	if err := c.execute(ctx, resty.MethodPost, path, convertReq2RawDocument(docReq), &rawDoc, nil); err != nil {
		err = tryToValidationError(err)
		ctxlog.Debug(ctx, c.logger, "CreateDocument", log.String("uid", uid), log.String("passengerID", passengerID), log.Error(err))
		return nil, err
	}
	doc, err := convertRaw2Document(rawDoc)
	if err != nil {
		return nil, InternalError.Wrap(err)
	}
	ctxlog.Debug(ctx, c.logger, "CreateDocument", log.String("uid", uid), log.String("passengerID", passengerID))
	return doc, nil
}

func (c *HTTPClient) UpdateDocument(ctx context.Context, uid string, passengerID string, documentID string, docReq *CreateOrUpdateDocumentRequest) (*Document, error) {
	path := fmt.Sprintf("/travelers/%s/passengers/%s/documents/%s", url.PathEscape(uid), url.PathEscape(passengerID), url.PathEscape(documentID))
	var rawDoc rawDocument
	if err := c.execute(ctx, resty.MethodPut, path, convertReq2RawDocument(docReq), &rawDoc, nil); err != nil {
		err = tryToValidationError(err)
		ctxlog.Debug(ctx, c.logger, "UpdateDocument", log.String("uid", uid), log.String("passengerID", passengerID), log.String("documentID", documentID), log.Error(err))
		return nil, err
	}
	doc, err := convertRaw2Document(rawDoc)
	if err != nil {
		return nil, InternalError.Wrap(err)
	}
	ctxlog.Debug(ctx, c.logger, "UpdateDocument", log.String("uid", uid), log.String("passengerID", passengerID), log.String("documentID", documentID))
	return doc, nil
}

func (c *HTTPClient) DeleteDocument(ctx context.Context, uid string, passengerID string, documentID string) error {
	path := fmt.Sprintf("/travelers/%s/passengers/%s/documents/%s", url.PathEscape(uid), url.PathEscape(passengerID), url.PathEscape(documentID))
	if err := c.execute(ctx, resty.MethodDelete, path, nil, nil, nil); err != nil {
		ctxlog.Debug(ctx, c.logger, "DeleteDocument", log.String("uid", uid), log.String("passengerID", passengerID), log.String("documentID", documentID), log.Error(err))
		return err
	}
	ctxlog.Debug(ctx, c.logger, "DeleteDocument", log.String("uid", uid), log.String("passengerID", passengerID), log.String("documentID", documentID))
	return nil
}

func (c *HTTPClient) ListBonusCards(ctx context.Context, uid string, passengerID string) ([]BonusCard, error) {
	path := fmt.Sprintf("/travelers/%s/passengers/%s/bonus-cards", url.PathEscape(uid), url.PathEscape(passengerID))
	var rawCards []rawBonusCard
	if err := c.execute(ctx, resty.MethodGet, path, nil, &rawCards, nil); err != nil {
		ctxlog.Debug(ctx, c.logger, "ListBonusCards", log.String("uid", uid), log.String("passengerID", passengerID), log.Error(err))
		return nil, err
	}
	cards := make([]BonusCard, 0, len(rawCards))
	for _, rawCard := range rawCards {
		card, err := convertRaw2BonusCard(rawCard)
		if err != nil {
			return nil, InternalError.Wrap(err)
		}
		cards = append(cards, *card)
	}
	ctxlog.Debug(ctx, c.logger, "ListBonusCards", log.String("uid", uid), log.String("passengerID", passengerID))
	return cards, nil
}

func (c *HTTPClient) GetBonusCard(ctx context.Context, uid, passengerID, bonusCardID string) (*BonusCard, error) {
	path := fmt.Sprintf("/travelers/%s/passengers/%s/bonus-cards/%s", url.PathEscape(uid), url.PathEscape(passengerID), url.PathEscape(bonusCardID))
	var rawCard rawBonusCard
	if err := c.execute(ctx, resty.MethodGet, path, nil, &rawCard, nil); err != nil {
		ctxlog.Debug(ctx, c.logger, "GetBonusCard", log.String("uid", uid), log.String("passengerID", passengerID), log.String("bonusCardID", bonusCardID), log.Error(err))
		return nil, err
	}
	card, err := convertRaw2BonusCard(rawCard)
	if err != nil {
		return nil, InternalError.Wrap(err)
	}
	ctxlog.Debug(ctx, c.logger, "GetBonusCard", log.String("uid", uid), log.String("passengerID", passengerID), log.String("bonusCardID", bonusCardID))
	return card, nil
}

func (c *HTTPClient) CreateBonusCard(ctx context.Context, uid, passengerID string, editableCard *EditableBonusCard) (*BonusCard, error) {
	path := fmt.Sprintf("/travelers/%s/passengers/%s/bonus-cards", url.PathEscape(uid), url.PathEscape(passengerID))
	var rawCard rawBonusCard
	if err := c.execute(ctx, resty.MethodPost, path, convertEditable2RawBonusCard(editableCard), &rawCard, nil); err != nil {
		err = tryToValidationError(err)
		ctxlog.Debug(ctx, c.logger, "CreateBonusCard", log.String("uid", uid), log.String("passengerID", passengerID), log.Error(err))
		return nil, err
	}
	card, err := convertRaw2BonusCard(rawCard)
	if err != nil {
		return nil, InternalError.Wrap(err)
	}
	ctxlog.Debug(ctx, c.logger, "CreateBonusCard", log.String("uid", uid), log.String("passengerID", passengerID))
	return card, nil
}

func (c *HTTPClient) UpdateBonusCard(ctx context.Context, uid, passengerID, bonusCardID string, editableCard *EditableBonusCard) (*BonusCard, error) {
	path := fmt.Sprintf("/travelers/%s/passengers/%s/bonus-cards/%s", url.PathEscape(uid), url.PathEscape(passengerID), url.PathEscape(bonusCardID))
	var rawCard rawBonusCard
	if err := c.execute(ctx, resty.MethodPut, path, convertEditable2RawBonusCard(editableCard), &rawCard, nil); err != nil {
		err = tryToValidationError(err)
		ctxlog.Debug(ctx, c.logger, "UpdateBonusCard", log.String("uid", uid), log.String("passengerID", passengerID), log.String("bonusCardID", bonusCardID), log.Error(err))
		return nil, err
	}
	card, err := convertRaw2BonusCard(rawCard)
	if err != nil {
		return nil, InternalError.Wrap(err)
	}
	ctxlog.Debug(ctx, c.logger, "UpdateBonusCard", log.String("uid", uid), log.String("passengerID", passengerID), log.String("bonusCardID", bonusCardID))
	return card, nil
}

func (c *HTTPClient) DeleteBonusCard(ctx context.Context, uid, passengerID, bonusCardID string) error {
	path := fmt.Sprintf("/travelers/%s/passengers/%s/bonus-cards/%s", url.PathEscape(uid), url.PathEscape(passengerID), url.PathEscape(bonusCardID))
	if err := c.execute(ctx, resty.MethodDelete, path, nil, nil, nil); err != nil {
		ctxlog.Debug(ctx, c.logger, "DeleteBonusCard", log.String("uid", uid), log.String("passengerID", passengerID), log.String("bonusCardID", bonusCardID), log.Error(err))
		return err
	}
	ctxlog.Debug(ctx, c.logger, "DeleteBonusCard", log.String("uid", uid), log.String("passengerID", passengerID), log.String("bonusCardID", bonusCardID))
	return nil
}

func convertRaw2Passenger(rp rawPassenger) (*Passenger, error) {
	id, err := toString(rp, "id")
	if err != nil {
		return nil, xerrors.Errorf("no id: %w", err)
	}
	createdAtStr, err := toString(rp, "created_at")
	if err != nil {
		return nil, xerrors.Errorf("no created_at: %w", err)
	}
	createdAt, err := strToTime(createdAtStr)
	if err != nil {
		return nil, xerrors.Errorf("invalid created_at: %w", err)
	}
	updatedAtStr, err := toString(rp, "updated_at")
	if err != nil {
		return nil, xerrors.Errorf("no updated_at: %w", err)
	}
	updatedAt, err := strToTime(updatedAtStr)
	if err != nil {
		return nil, xerrors.Errorf("invalid updated_at: %w", err)
	}
	passenger := Passenger{
		ID:        id,
		CreatedAt: &Timestamp{*createdAt},
		UpdatedAt: &Timestamp{*updatedAt},
	}
	fields := make(map[string]interface{})
	for _, name := range passengerFieldNames {
		v, ok := rp[name]
		if !ok {
			return nil, xerrors.Errorf("no field %s", name)
		}
		fields[name] = v
	}
	if len(fields) > 0 {
		passenger.Fields = fields
	}
	rawDocs, ok := rp["documents"]
	if ok {
		switch v := rawDocs.(type) {
		case []interface{}:
			if len(v) > 0 {
				docs := make([]Document, 0, len(v))
				for _, rawDoc := range v {
					switch r := rawDoc.(type) {
					case map[string]interface{}:
						doc, err := convertRaw2Document(r)
						if err != nil {
							return nil, xerrors.Errorf("convert document error: %w", err)
						}
						docs = append(docs, *doc)
					default:
						return nil, xerrors.Errorf("invalid document")
					}
				}
				passenger.Documents = docs
			}
		default:
			return nil, xerrors.Errorf("invalid documents")
		}
	}
	rawCards, ok := rp["bonus_cards"]
	if ok {
		switch v := rawCards.(type) {
		case []interface{}:
			if len(v) > 0 {
				cards := make([]BonusCard, 0, len(v))
				for _, rawCard := range v {
					switch r := rawCard.(type) {
					case map[string]interface{}:
						card, err := convertRaw2BonusCard(r)
						if err != nil {
							return nil, xerrors.Errorf("convert document error: %w", err)
						}
						cards = append(cards, *card)
					default:
						return nil, xerrors.Errorf("invalid document")
					}
				}
				passenger.BonusCards = cards
			}
		default:
			return nil, xerrors.Errorf("invalid documents")
		}
	}
	return &passenger, nil
}

func convertReq2RawPassenger(passenger *CreateOrUpdatePassengerRequest) rawPassenger {
	raw := make(rawPassenger, len(passenger.Fields))
	for k, v := range passenger.Fields {
		raw[k] = v
	}
	return raw
}

func convertRaw2Document(rawDoc rawDocument) (*Document, error) {
	docType, err := toString(rawDoc, "type")
	if err != nil {
		return nil, xerrors.Errorf("no type: %w", err)
	}
	id, err := toString(rawDoc, "id")
	if err != nil {
		return nil, xerrors.Errorf("no id: %w", err)
	}
	passengerID, err := toString(rawDoc, "passenger_id")
	if err != nil {
		return nil, xerrors.Errorf("no passenger_id: %w", err)
	}
	createdAtStr, err := toString(rawDoc, "created_at")
	if err != nil {
		return nil, xerrors.Errorf("no created_at: %w", err)
	}
	createdAt, err := strToTime(createdAtStr)
	if err != nil {
		return nil, xerrors.Errorf("invalid created_at: %w", err)
	}
	updatedAtStr, err := toString(rawDoc, "updated_at")
	if err != nil {
		return nil, xerrors.Errorf("no updated_at: %w", err)
	}
	updatedAt, err := strToTime(updatedAtStr)
	if err != nil {
		return nil, xerrors.Errorf("invalid updated_at: %w", err)
	}
	doc := Document{
		ID:          id,
		PassengerID: passengerID,
		Type:        docType,
		CreatedAt:   &Timestamp{*createdAt},
		UpdatedAt:   &Timestamp{*updatedAt},
	}
	fields := make(map[string]interface{})
	for _, name := range documentFieldNames {
		v, ok := rawDoc[name]
		if !ok {
			return nil, xerrors.Errorf("no field %s", name)
		}
		fields[name] = v
	}
	if len(fields) > 0 {
		doc.Fields = fields
	}
	return &doc, nil
}

func convertReq2RawDocument(doc *CreateOrUpdateDocumentRequest) rawDocument {
	rawDoc := make(rawDocument, len(doc.Fields)+1)
	rawDoc["type"] = doc.Type
	for k, v := range doc.Fields {
		rawDoc[k] = v
	}
	return rawDoc
}

func convertRaw2BonusCard(rawCard rawBonusCard) (*BonusCard, error) {
	cardType, err := toString(rawCard, "type")
	if err != nil {
		return nil, xerrors.Errorf("no type: %w", err)
	}
	id, err := toString(rawCard, "id")
	if err != nil {
		return nil, xerrors.Errorf("no id: %w", err)
	}
	passengerID, err := toString(rawCard, "passenger_id")
	if err != nil {
		return nil, xerrors.Errorf("no passenger_id: %w", err)
	}
	createdAtStr, err := toString(rawCard, "created_at")
	if err != nil {
		return nil, xerrors.Errorf("no created_at: %w", err)
	}
	createdAt, err := strToTime(createdAtStr)
	if err != nil {
		return nil, xerrors.Errorf("invalid created_at: %w", err)
	}
	updatedAtStr, err := toString(rawCard, "updated_at")
	if err != nil {
		return nil, xerrors.Errorf("no updated_at: %w", err)
	}
	updatedAt, err := strToTime(updatedAtStr)
	if err != nil {
		return nil, xerrors.Errorf("invalid updated_at: %w", err)
	}
	card := BonusCard{
		ID:          id,
		PassengerID: passengerID,
		Type:        cardType,
		CreatedAt:   &Timestamp{*createdAt},
		UpdatedAt:   &Timestamp{*updatedAt},
	}
	fields := make(map[string]interface{})
	for _, name := range bonusCardFieldNames {
		v, ok := rawCard[name]
		if !ok {
			return nil, xerrors.Errorf("no field %s", name)
		}
		fields[name] = v
	}
	if len(fields) > 0 {
		card.Fields = fields
	}
	return &card, nil
}

func convertEditable2RawBonusCard(card *EditableBonusCard) rawBonusCard {
	rawCard := make(rawBonusCard, len(card.Fields)+1)
	rawCard["type"] = card.Type
	for k, v := range card.Fields {
		rawCard[k] = v
	}
	return rawCard
}

func toString(entity map[string]interface{}, name string) (string, error) {
	resRaw, ok := entity[name]
	if !ok {
		return "", xerrors.Errorf("no value for entity")
	} else {
		switch v := resRaw.(type) {
		case string:
			return v, nil
		default:
			return "", xerrors.Errorf("type for entity is not string")
		}
	}
}

func strToTime(s string) (*time.Time, error) {
	t, err := time.ParseInLocation(travelersTimestampPattern, s, moscowLocation)
	if err != nil {
		return nil, xerrors.Errorf("can't get time from '%s': %w", s, err)
	}
	return &t, nil
}

func tryToValidationError(err error) error {
	var statusError StatusError
	if xerrors.As(err, &statusError) {
		if statusError.Status == 400 {
			fieldErrors := make(map[string]string)
			for f, errMsgs := range statusError.Response {
				var errMsg string
				if errMsgList, ok := errMsgs.([]interface{}); ok {
					errMsgStrings := make([]string, len(errMsgList))
					for i, m := range errMsgList {
						errMsgStrings[i] = fmt.Sprintf("%v", m)
					}
					errMsg = strings.Join(errMsgStrings, ", ")
				} else {
					continue
				}
				fieldErrors[f] = errMsg
			}
			if len(fieldErrors) > 0 {
				return ValidationError{FieldErrors: fieldErrors}
			}
		}
	}
	return err
}
